// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import java.util.*;
import java.io.*;
import java.security.*;
import jakarta.servlet.*;
import jakarta.servlet.http.*;

import fc.io.*;
import fc.util.*;

/**
Misc utility methods for sessions and cookies.

@author hursh jain
**/
public final class SessionUtil 
{
static final String TestCookieName = "eikooc";
	
/**
Sets a test cookies. The user should be redirected to this or some other
page after calling this method and the presence of this cookie can be
checked via {@link hasTestCookie} from that page. The lifetime of the
test cookie is set to the browsers lifetime (will be lost when the
browser exits).
*/
public static void setTestCookie(
	HttpServletRequest req, HttpServletResponse res)
	{
	Cookie c = new Cookie(TestCookieName, "scotchAndsoda");
	c.setMaxAge(-1);
	res.addCookie(c);
	}

/**
Returns true if the browser accepts cookies. The {@link setTestCookie()}
method must have been invoked by some page prior to calling this method
AND the user must have been either redirected to that or some other page
AND this method must be invoked from that redirected page.
<p>
<b>Note:</b>
Test cookies are entirely self contained. They are meant to figure out if
a browser accepts cookies. Their presence or absence does not mean that
<b>other</b> cookies will not be sent by the browser. These other cookies
may point to a expired session in memory or database and a seperate check
for that should be made.
*/
public static boolean hasTestCookie(
	HttpServletRequest req, HttpServletResponse res)
	{
	Cookie[] cookies = req.getCookies();
	if (cookies == null)
		return false;

	for (int n = 0; n < cookies.length; n++) {
		if (cookies[n].getName().equals(TestCookieName))
			return true;
		}

	return false;
	}

/**
Removes the test cookies if previously set. Normally, the
test cookie set by the {@link #setTestCookie} method is
automatically removed when the browser exists, but this
method allows removing it before such time.
*/
public static void deleteTestCookie(
	HttpServletRequest req, HttpServletResponse res)
	{
	Cookie[] cookies = req.getCookies();
	if (cookies == null)
		return;

	for (int n = 0; n < cookies.length; n++) 
		{
		if (cookies[n].getName().equals(TestCookieName))
			{
			cookies[n].setMaxAge(0);
			res.addCookie(cookies[n]);
			break;
			}
		}
	}

/**
Returns a new session identifier.
<pre>
See: cookies.lcs.mit.edu
See: www.across.si

There are 2 issues with generating sessionid's. 

1) uniqueness - 2 or more sessionid's should not end up being 
   the same.
2) hard-to-guess - For example, sequential values like 
   1, 2, 3 are unique but easy to guess and therefore easy
   to session hijack.

Our sessionid's have 2 parts:
a) a timestamp for guaranteed uniqueness  (easy to guess)
b) random data (hard to guess)
</pre>
*/
public static String newSessionID() 
	{
	return UUIDUtil.newSessionID();
	}
}