// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.page;

import java.io.*;
import java.util.*;

import jakarta.servlet.*;
import jakarta.servlet.http.*;

import fc.util.*;
import fc.io.*;

/** 
A superclass for generated pages. All pages derive from this 
class.

@author hursh jain
*/
public class PageImpl implements Page
{
static private	final boolean internal_dbg = false;

/**	
The default application log, retrieved via {@link
fc.web.servlet.WebApp#getAppLog() WebApp.getAppLog}. All molly pages can use
this variable directly, for example <code>log.info("test")</code>. The log
output level (and hence whether a logging statement at that level will be seen)
is set by the WebApp configuration file. (If WebApp is <i>not</i> used or
configured, then the log level will be set to the {@link Log#DEFAULT_LEVEL})
*/
public				Log	 			log;		
protected   		HttpServlet		servlet;
private		static 	ThreadLocal		watchMap  		= new ThreadLocal();
private		final	ThreadLocal 	threadLocalOut 	= new ThreadLocal();
private 			String  		dbg_prefix = "";
private 			String  		dbg_suffix = "";

public void render(HttpServletRequest req, HttpServletResponse res) 
throws Exception
	{
	throw new ServletException("This page is not implemented");
	}

public void init(PageServlet servlet, String contextRelativePagePath) throws ServletException
	{
	this.log 		= Log.get(contextRelativePagePath);
	this.servlet 	= servlet;
	if (internal_dbg) System.out.println(getClass().getName() + " init() called:" + ClassUtil.getClassLoaderInfo(PageImpl.class));
	}
	
public void destroy() { 
	log.close();
	if (internal_dbg) System.out.println(getClass().getName() + ClassUtil.getClassLoaderInfo(PageImpl.class));
	}

public String getPagePath(HttpServletRequest req)
	{
	return req.getContextPath()  + req.getServletPath();
	}

public String getRealPath(HttpServletRequest req)
	{
	return 
		servlet.getServletConfig().getServletContext().getRealPath(getPagePath(req));
	}

public void clientRedirect(
	HttpServletRequest req, HttpServletResponse res, String newLocation)
throws IOException
	{
	String redirectURL = fc.web.servlet.WebUtil.absolutePath(req, newLocation);
	res.sendRedirect(redirectURL);
	}

public CharArrayWriter getThreadLocalWriter()
	{
	CharArrayWriter out = (CharArrayWriter) threadLocalOut.get();
	
	if (out == null) {
		out = new CharArrayWriter();
		threadLocalOut.set(out);
		}
	
	return out;
	}

/**
If set to <tt>true</tt>, a timer to calculate page render time
is started after this method call. Page render time can then be
displaying the value returned by invoking the {@link #getTime}
method.
*/
public final void startTimer()
	{
	//each thread running thru the page has it's own watch. 	
	Watch watch = getWatch();
	if (watch == null) {
		watch = new Watch();
		watchMap.set(watch);
		}
	watch.start();
	}

/**
Returns the time elapsed after invoking {@link startTime} method (in milliseconds)
*/
public final long getTime()
	{
	Watch watch = getWatch();
	if (watch == null)
		throw new RuntimeException("The startTimer() method must be invoked prior to invoking this method. Timers are thread local different processing threads get their own private timers. You must startTimer() and getTime() inside the render method, don't override the init() method to startTimer().."); 	
	return watch.time();
	}

private final Watch getWatch()
	{
	return (Watch) watchMap.get();
	}

/** 
Controls whether debugging is on or off. For expensive debug statements,
<code>if (dbg) bug(....)</code> type statements can be used in a page.
*/
protected volatile 	boolean 	dbg = false;

/**
Starts/stop debugging with no dbg_prefix/dbg_suffix.

@param	val		true to enable debugging, false to disable.
*/
public final void dbg(final boolean val)
	{
	this.dbg = val;
	}

/**
Sets the prefix for debugging output.

@param	dbg_prefix	some html/text (such as <xmp><font color=red></xmp>) that 
					is printed before each debug statement
*/
public final void dbgPrefix(String dbg_prefix)
	{
	this.dbg_prefix = dbg_prefix;
	}

/**
Sets the suffix for debugging output.

@param	dbg_suffix	some html/text that is printed after each debug statement
*/
public final void dbgSuffix(String dbg_suffix)
	{
	this.dbg_suffix = dbg_suffix;
	}

/**
Prints a debug statement if debugging is turned on for this page.

<p>Typically the implicit page printwriter (the <code>out</code>
variable) will be passed to this method and debug statements will be
printed at the point where they are lexically invoked at the page.
<p>
However, each page request thread can collect debugging information
and print the output at some arbitrary location, such as the bottom
of the page. The method {@link #getThreadLocalWriter()} exists for
this reason and can be used to collect thread-local output during
page execution.
*/
public final void bug(final Writer writer, final Object str1) throws IOException
	{
	if (! dbg)
		return;
		
	writer.append(dbg_prefix);
	writer.append(str1 != null ? str1.toString() : "null");
	writer.append(dbg_suffix);
	}
	
public final void bug(final Writer writer, 
						final Object str1, final Object str2) throws IOException
	{
	if (! dbg)
		return;

	writer.append(dbg_prefix);
	writer.append(str1 != null ? str1.toString() : "null");
	writer.append(str2 != null ? str2.toString() : "null");
	writer.append(dbg_suffix);
	}

public final void bug(final Writer writer, 
	  final Object str1, final Object str2, final Object str3) throws IOException
	{
	if (! dbg)
		return;

	writer.append(dbg_prefix);
	writer.append(str1 != null ? str1.toString() : "null");
	writer.append(str2 != null ? str2.toString() : "null");
	writer.append(str3 != null ? str3.toString() : "null");
	writer.append(dbg_suffix);
	}
	
public final void bug(final Writer writer, 
	final Object str1, final Object str2, final Object str3, final Object... args) 
throws IOException
	{
	if (! dbg)
		return;
		
	writer.append(dbg_prefix);
	writer.append(str1 != null ? str1.toString() : "null");
	writer.append(str2 != null ? str2.toString() : "null");
	writer.append(str3 != null ? str3.toString() : "null");
	final int len = args.length;
	for (int i = 0; i < len; i++) {
		writer.append(args[i] != null ? args[i].toString() : "null");
		}
	writer.append(dbg_suffix);
	}

/* -------- "debug" as well as "bug" ----------- */

/**
Prints a debug statement if debugging is turned on for this page. Same
as calling {@link #bug(Writer, Object)}.
*/
public final void debug(final Writer writer, final Object str1) throws IOException
	{
	bug(writer, str1);
	}
	
public final void debug(final Writer writer, 
						final Object str1, final Object str2) throws IOException
	{
	bug(writer, str1, str2);
	}

public final void debug(final Writer writer, 
	  final Object str1, final Object str2, final Object str3) throws IOException
	{
	bug(writer, str1, str2, str3);
	}
	
public final void debug(final Writer writer, 
	final Object str1, final Object str2, final Object str3, final Object... args) 
throws IOException
	{
	bug(writer, str1, str2, str3, args);
	}

}		
