// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.page;

import java.io.*;
import java.util.*;
import fc.util.*;

/**
Used to compile generated pages by invoking the java compiler.
*/
public class PageCompiler
{
private static final boolean dbg = false;

String 	error;
File	javafile;
String  classpath;
String  encoding;

/**
Creates a new page compiler that will use the default (system) classpath
as seen by 'javac' when it is invoked from the command line. No
seperate "encoding" flag will be specified to the javac.

*/
public PageCompiler(File javafile)
	{
	this.javafile = javafile;
	}

/**
Creates a new page compiler with the specified classpath. This is useful
when the classpath must contain some directories within the servlet web
application, just as <tt>WEB-INF/classes</tt>, <tt>WEB-INF/lib</tt> etc.

@param 	javafile	the source java file to compile
@param	classpath	classpath to use when compiling
@param	encoding	the encoding of the java source file (example ISO-8859-1, UTF-8 etc.).
					Used by the -encoding flag passed to javac. Specify <tt>null</tt> for
					no specific encoding. Modern Java platforms *seem* to default to UTF-8
					(used to be ISO-8859-1 back in the day) if nothing is specified.
					<p>When invoked/used internally (ie not seperately from the command line), we 
					specify a {@link Page#DEFAULT_SRC_ENCODING} as the src encoding anyway.
					*/
public PageCompiler(File javafile, String classpath, String encoding)
	{
	this.javafile	= javafile;
	this.classpath	= classpath;
	this.encoding	= encoding;
	}
	
public String getError()
	{
	return error;
	}
	
public boolean compile() throws IOException
	{
	//javac 1.5.x does not crap out if a 0 byte source file is fed to it
	//it simply returns 0 (implying success) but does NOT generate a classfile. so this hack.
	if (javafile.length() == 0)
		{
		error = "Java source file [" + 
				javafile.getCanonicalPath() + 
				"] size was 0 bytes. This file cannot be compiled";
		
		return false;
		}
	
	//this is ok too: {"bash", "-c", "javac -nowarn /tmp/c.java"});
	List javac_cmd = new ArrayList();
	javac_cmd.add("javac");
	javac_cmd.add("-nowarn");
	
	if (encoding != null) {
		javac_cmd.add("-encoding");
		javac_cmd.add(encoding);
		}
	
	if (classpath != null) {
		javac_cmd.add("-classpath");
		javac_cmd.add(classpath);
		}
	javac_cmd.add(javafile.getCanonicalPath());	

	if (dbg) System.out.println("Compile command: " + javac_cmd);
	
	ProcessBuilder pb = new ProcessBuilder(javac_cmd);
	
	Process p = pb.start();
		
	int exit_val = 0;
	
	/*
	have to read out/err from the process otherwise it may hang if
	javac has lots of output (in case of errors)...waitFor will not
	return since the process is hung.
	See: jdk 1.5 docs and 
	http://www.javaworld.com/javaworld/jw-12-2000/jw-1229-traps.html
	
	try 
		{
		exit_val = p.waitFor();
		}
	catch (InterruptedException e)
		{
		throw new IOException(e.toString());
		}
	*/	

	CharArrayWriter buf = new CharArrayWriter(1024);
	
	InputStream stderr = new BufferedInputStream(p.getErrorStream());
	InputStream stdout  = new BufferedInputStream(p.getInputStream());

	int c = stderr.read();
	while (c != -1) {
		buf.write((char)c);
		c = stderr.read();
		}

	c = stdout.read();
	if (c != -1)
		buf.append("-------------------------------------------");
	
	while (c != -1) {
		buf.write((char)c);
		c = stdout.read();
		}
	
	error = buf.toString();

	try 
		{
		exit_val = p.waitFor();
		}
	catch (InterruptedException e)
		{
		throw new IOException(e.toString());
		}

	if (dbg) System.out.println("Exit value: " + exit_val);
		
	return exit_val == 0;
	}

public static void main (String args[]) throws Exception
	{
	Args myargs = new Args(args);
	myargs.setUsage("java " + ClassUtil.getClassName() + " -file path-to-file-to-compile [-classpath <class-path> -encoding encoding]");
	
	PageCompiler pc = new PageCompiler(
		new File(myargs.getRequired("file")),
		myargs.get("classpath", System.getProperty("java.class.path")),
		myargs.get("encoding", null)
		);
	
	if (! pc.compile())
		System.out.println(pc.getError());
	} 
}
