// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import java.io.*;
import java.util.*;
import java.sql.*;
import jakarta.servlet.*;
import jakarta.servlet.http.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
An HTML Select field 
<p>
There are 3 different kinds of selects:
<dl>
<dt>Select</dt>
	<dd>A normal select that is created/instantiated and added to the
	form. This select displays the same options to all users and these
	options cannot be changed per request/user (since it does not
	implement the <tt>setValue(FormData fd, ...)</tt> method. This field
	will always track that the submitted data is a legal value and was
	not hacked/modified by the client.</dd>
<dt>RefreshableSelect</dt>
	<dd>This select starts out by displaying the same options to all
	users. However, it allows options to be thereafter set/displayed per
	user/request. If per user/request options are shown, then the browser
	can modify/hack/send any option. This field (in contrast to most
	other fields in the form) itself won't track this and application
	logic is responsible (if applicable) for tracking if the submitted
	data is valid.</dd>.
<dt>DependentSelect</dt>
	<dd>This select is similar to a RefreshableSelect but uses an
	external dependency class to set both it's initial values and per
	user/request subsequent values. The dependency typically looks at
	other fields in the form to generate this data.
</dl>

@author hursh jain
**/
public class Select extends Field
{
//option[text='foo',value='1'], getValue()==1
//option[text='foo'], getValue()==foo

//all options shown to user
LinkedHashMap 	options = new LinkedHashMap();			 
//orig selected options when form was constructed
Map				origSelectedMap = new HashMap(); 
int 			size = 0;
boolean 		multiple;

static final class Data 
	{
	//To store selected options submitted by the user
	//acts like a set, used via containsKey() to see if an options
	//is selected (so we can reselect it) during rendering. 
	private Map  selectedMap = new HashMap();	
	
	//to store all options submitted by user
	private List selectedList	 = new ArrayList();
	}

/** 
Creates a new select element with no initial values. Multiple
selections are not initially allowed.

@param 	name		the field name
**/
public Select(String name)
	{
	super(name);
	}
	
/** 
Creates a new select element with the specified initial values
and no multiple selections allowed. There is no explicit default
selection which typically means the browser will show the first
item in the list as the selection item.

@param 	name		the field name
@param	values		the initial option values for this class. The
					objects contained in the list must be of type
					{@link Option} otherwise a 
					<tt>ClassCastException</tt> may be thrown in 
					future operations on this object.

@throws	IllegalArgumentException  if the values parameter was null
								  (note: an empty but non-null List is ok)
**/
public Select(String name, List values)
	{
	super(name);
	Argcheck.notnull(values, "values param was null");
	initOptions(values);
	}

void initOptions(final List values) 
	{	
	for (int n = 0; n < values.size(); n++) {
		final Option item = (Option) values.get(n);
		String itemval = item.getValue();
		this.options.put(itemval, item);
		if (item.isOrigSelected()) {
			this.origSelectedMap.put(itemval, item);	
			}
		}	
	}

/**
The specified query is used to populate this select. The default parameter
(if not null) is the default selected option. If the default option is
<tt>null</tt>, no option is selected for multiple-value selects and the
first option returned by the query is selected for single-value selects.
<p>
Typically, this method may be invoked like:
<blockquote>
<tt>
useQuery(con, "select id, name from my_lookup_table", new Option("--choose--"));
</tt>
</blockquote>
If the lookup table has more than 2 columns, the query can look like (in
standard SQL syntax):
<blockquote>
<tt>
useQuery(con, "select id, <font color=blue>name1 || ' ' || name2</font> from my_lookup_table", new Option("--choose--"));
</tt>
</blockquote>
This shows columns name1 and name2 concatenated with each other.

@param	con		the connection to use for the query. This connection is
				<b>not</b> closed by this method. <u>Close this connection
				from the calling code to prevent connection leaks.</u>

@return	this select for method chaining convenience
*/
public Select useQuery(Connection con, String query, Select.Option defaultOpt) 
throws SQLException
	{
	reset();
	
	if (defaultOpt != null)
		add(defaultOpt);

	final List list = makeOptionsFromQuery(con, query);
	initOptions(list);

	return this;
	}

/**
Convenience method that calls {@link useQuery(Connection, String, String)
useQuery(con, query, null)}.
*/
public Select useQuery(Connection con, String query) throws SQLException
	{
	useQuery(con, query, null);
	return this;
	}

/**
Uses the first column as the option text and if there is a second column
uses it as the corresponding option value.
*/
public static List makeOptionsFromQuery(Connection con, String query) 
throws SQLException
	{
	return makeOptionsFromQuery(con, query, null);
	}

/**
Uses the first column as the option text and if there is a second column
uses it as the corresponding option value. Uses an additional default
option.
*/
public static List makeOptionsFromQuery(
	Connection con, String query, Select.Option defaultOption)
throws SQLException
	{
	final List list = new ArrayList();
	if (defaultOption != null)
		list.add(defaultOption);
	
	final Statement stmt = con.createStatement();
	final ResultSet rs = stmt.executeQuery(query);
	final ResultSetMetaData rsmd = rs.getMetaData();
	final int numberOfColumns = rsmd.getColumnCount();	
	if (numberOfColumns == 0) {
		log.warn("Query [",query, "] returned no columns");
		}
	while (rs.next()) {
		String text = rs.getString(1);
		String value = (numberOfColumns == 1) ? text : rs.getString(2);
		Select.Option opt = new Select.Option(text, value);
		list.add(opt);
		}
	stmt.close();
	return list;
	}

public Field.Type getType() {
	return Field.Type.SELECT;
	}

/**
Sets the values for this select. Any previously set values are
first cleared before new values are set.
<p>
Note, to show user specific options via FormData, use a {@link RefreshableSelect}
instead.

@param	values	a list of {@link Select.Option} objects
*/
public void setValue(List values)
	{
	Argcheck.notnull(values, "specified values param was null");
	options.clear();
	origSelectedMap.clear();
	initOptions(values);	
	}


/** 
Returns a List containing the selected options. Each object in
the collection will be of type {@link Select.Option}. If there
are no selected options, the returned list will be an
empty list.

@param	fd 	the submited form data. This object should not be null
			otherwise a NPE will be thrown.
**/
public List getValue(FormData fd) 
	{
	Select.Data data = (Select.Data) fd.getData(name);
	if (data == null) {
		return Form.empty_list;
		}
	return data.selectedList;
	}

/** 
Convenience method that returns the selected option as a
String. <u>No guarantees are made as to which selection is
returned when more than one selection is selected (this
method is really meant for when the select only allows
single selections as a dropdown).</u>
<p>
The returned value is of type String obtained by called the
selected Option's {@link Select.Option#getValue} method.
<p>
If there is no selection, returns <tt>null</tt>
**/
public String getStringValue(FormData fd) 
	{
	String result = null;
	
	Select.Data data = (Select.Data) fd.getData(name);
	if (data == null) {
		return null;
		}

	List list = data.selectedList;
	if (list.size() == 0) {
		return result;
		}
	else {
		Option opt = (Option) list.get(0);
		return opt.getValue();
		}
	}


/**
Convenience method that returns the single value of this field
as a Integer.
<p>
All the caveats of {@link #getSingleValueAsString()} apply.

@throws NumberFormatException	if the value could not be
								returned as in integer.	
*/
public int getIntValue(FormData fd) {
	return Integer.parseInt(getStringValue(fd));
	}

/**
Convenience method that returns the single value of this field
as a boolean. 
<p>
All the caveats of {@link #getSingleValueAsString()} apply.
In particular, the formdata should contain non-null data
with at least one selection.
*/
public boolean getBooleanValue(FormData fd) {
	return Boolean.valueOf(getStringValue(fd)).booleanValue();
	}

/**
Returns <tt>true</tt> if some option has been selected
by the user, <tt>false</tt> otherwise. (also returns
<tt>false</tt> is the specified form data is <tt>null</tt>).
*/
public boolean isFilled(FormData fd) 
	{
	if (fd == null)
		return false;
		
	Select.Data data = (Select.Data) fd.getData(name);
	if (data == null) {
		return false;
		}
	List list = data.selectedList;
	//if data is present, then list always will be non-null
	return (/*list != null &&*/ list.size() != 0);
	}

/*
 Select values can be present as:
 selectname=value

 or for multiple selections (for say a select called "sel1")
 sel1=value1&sel1=val2
 
 The values sent are those in the value tag in the option
 and if missing those of the corresponding html for the option.

 If a select allows multiple and none are selected, then
 nothing at all is sent (note this is different than for
 single (and not multiple) selects, where some value will always
 be sent (since something is always selected).

 We can track submitted/selected options in 2 ways:
 1. go thru the option list and set a select/unselect on each option
 element and then the option element renders itself appropriately.
 2. keep a separate list of selected elements and at render time display
 the item as selected only if it's also in the select list.

 (w) implies that we have to specify select/no select at
 render time to each option as opposed to setting that flag
 in the option before telling the option to render itself.
 (2) is chosen here.

 the client can send hacked 2 or more options with the same
 value this method will simply add any such duplicate values
 at the end of the list. this is ok.
*/
public void setValueFromSubmit(FormData fd, HttpServletRequest req) 
throws SubmitHackedException
	{
	//value(s) associated with the selection field name
	String[] values = req.getParameterValues(name);
	
	//todo: if ! multiple && enabled && values == null, hacklert
	if (values == null) {
		return;
		}

	//lazy instantiation
	Select.Data data = new Select.Data();
	fd.putData(name, data);
	
	if (multiple && values.length > 1)
		hacklert(req, "recieved multiple values for a single value select");
		
	if (multiple) 
		{
		for (int n = 0; n < values.length; n++) {
			addSelectedOpt(req, data, values[n]);
			}
		}
	else{
		addSelectedOpt(req, data, values[0]);
		}
	}

private void addSelectedOpt(
 HttpServletRequest req, Select.Data data, String submitValue)
throws SubmitHackedException
	{
	// our options were stored with option's value as the key
    Option opt = (Option) options.get(submitValue);

	// this can happen if option values were hacked by the client
    if (opt == null) 
    	{
    	hacklert(req, 
    		"could not match/retrieve a submitted option from the options." + 
    		"; Submited value=[" + submitValue +"]; options=" + options); 
    	}				
	else{
		data.selectedMap.put(opt.getValue(), opt);
		data.selectedList.add(opt);
		}
	}	

public void renderImpl(FormData fd, Writer writer) throws IOException
	{
	Select.Data data = null;

	if (fd != null)  { //we have submit or initial data in the FD
		//can be null, no submit/initial data for this particular field
		data = (Select.Data) fd.getData(name); 
		}
	
	//data can be null here.
	
	writer.write("<select");	
	writer.write(" name='");
	writer.write(name);
	writer.write("'");
	
	if (size > 0) {
		writer.write(" size='");
		writer.write(String.valueOf(size));
		writer.write("'"); 
		}
		
	if (! enabled || ! isEnabled(fd)) {
		writer.write(" disabled");
		}

	if (multiple) {
		writer.write(" multiple"); 
		}	
		
	if (renderStyleTag) {
		writer.write(" style='");
		writer.write(styleTag);
		writer.write("'");
		}
		
	final int arlen = arbitraryString.size();
	for (int n = 0; n < arlen; n++) {
		writer.write(" ");
		writer.write(arbitraryString.get(n).toString());
		}
		
	writer.write(">\n");
	
	Iterator it = options.values().iterator(); 
	while (it.hasNext()) 
		{
	 	Option item = (Option) it.next();
	 	String itemval = item.getValue();
	 	
		boolean selected = false;
		
		//NOTE: DO NOT COMBINE  and say:
		//if (fd != null && data != null)
		//	that's _the_ classic bug, because if:
		//	fd != null but data == null, we will drop down
		//	to (3) but we need to go to (2).
		if (fd != null) 	/* maintain submit state */
			{					
/*1*/		if (data != null) {  
				selected = data.selectedMap.containsKey(itemval);
				}
			else {  
/*2*/			//form submitted but option not selected				
				//selected should be false here [which it is by default]
				}
			}
		else{  	 /* form not submitted, show original state */
/*3*/		selected = origSelectedMap.containsKey(itemval);
		   	}
		   
		writer.write(item.render(selected));
		writer.write("\n");	//sufficient for easy view source in browser		
		}	

	writer.write("</select>");
	}
	
/** 
Adds a new option to the selection list. Replaces any previous
option that was added previously with the same value.

@param	opt	the option to be added
**/
public void add(Option opt) 
	{
	Argcheck.notnull(opt, "opt param was null");
	String optval = opt.getValue();
	options.put(optval, opt);
	if (opt.isOrigSelected()) {
		origSelectedMap.put(optval, opt);	
		}
	}	

/** 
Adds a new option to the selection list. Replaces any previous
option that was added previously with the same value. This method
will have the same effect as if the {@link #add(Option) 
add(new Select.Option(item))} method was invoked.

@param	item the option to be added
**/
public void add(String item) 
	{
	Argcheck.notnull(item, "item param was null");
	Select.Option opt = new Select.Option(item);
	add(opt);	
	}	
	
/**
Clears all data in this select.
*/
public void reset()
	{
	options.clear(); 
	origSelectedMap.clear();
	}
	
/** 
This value (if set) is rendered as the html <tt>SIZE</tt> tag. 
If the list contains more options than specified by size, the browser 
will display the selection list with scrollbars.

@return	this object for method chaining convenience
**/
public Select setSize(int size) {
	this.size = size;
	return this;
	}	

/** 
<tt>true</tt> is multiple selections are allowed, <tt>false</tt> otherwise. 
This value (if set) is rendered as the html <tt>MULTIPLE</tt> tag.

@return	this object for method chaining convenience
**/
public Select allowMultiple(boolean allow) {
	this.multiple = allow;
	return this;
	}	

/** Represents an option in the selection list **/
public static final class Option 
	{
	private String 	value;
	private String 	text;
	private boolean orig_selected;
	
	/** 
	Constructs a new option with the specified text and 
	value of the option tag.
	
	@param	text		the html text of the option tag 
	@param	value	 	the value of the option tag
	@param	selected 	<tt>true</tt> if this option is selected
						by default. If more than one selected option
						is added to a select field and that select
						field does <b>not</b> have it's multiple attribute
						set, then the option displayed as selected is 
						browser dependent (Moz1, IE6 show
						the last selected, N4 the first). More than one
						selected option should not be shown for non multiple
						select fields anyway.
	**/
	public Option(String text, String value, boolean selected) 
		{
		this.text = text;
		this.value = value;
		this.orig_selected = selected;
		}

	/** 
	Constructs a new unselected option with the specified 
	text and value of the option tag.
	
	@param	text	the html text of the option tag 
	@param	value	the value of the option tag
 	**/
	public Option(String text, String value) {
		this(text, value, false);
		}
	
 	/** 
 	Constructs a new option with the specified text (and no
 	separate value tag).
 	
	@param	text		the html text of the option tag 
	@param 	selected 	<tt>true</tt> to select this option
						<tt>false</tt> otherwise
 	**/
 	public Option(String text, boolean selected) {
 		this(text, null, selected);
 		}

 	/** 
 	Constructs a new unselected option with the specified
	html text and no separate value.

	@param	text	the html text of the option tag 
 	**/
 	public Option(String text) {
 		this(text, null, false);
 		}
		
 	boolean isOrigSelected() {
		return orig_selected;
		}
				
	/** 
	Returns the value of this option tag. If no value is set, 
	returns the html text value for this option tag 
	**/
	public String getValue() 
		{
		if (value != null)
			return value;
		else
			return text;
		}	
				
	public String render(final boolean selected) {
		StringBuffer buf = new StringBuffer(32);
		buf.append("<option");
		if (value != null) {
			buf.append(" value='");
			buf.append(value);
			buf.append("'");
			}
		if (selected) {
			buf.append(" SELECTED");
			}
		buf.append(">");
		buf.append(text);
		buf.append("</option>");
		return buf.toString();
		} 
	
	public String toString() {
		return render(false);
		}
	} //~class Option
		
}          //~class Select