// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import jakarta.servlet.*;
import jakarta.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
Represents a form level validator that may examine more than
one form field for a grouped/form level validation. Form level
validators differ from {@link FieldValidator} because one may
need to validate a group of fields together.
<p>
Examples include address validation where street, city and zip might
need to be verified together (for a valid address) or password
validation where two separate password boxes might need to be
validated (to be identical). Various subclasses implement
concrete code (via {@link #validate}) for various validation
strategies. 
<p>
<b>Note 1</b>: Validator objects have state and a particular instance of a
validator should only be assigned to one form field.
<p>
<b>Thread safety:</b> None of the validation classes (like all other
form-related classes) are thread safe. Synchronization must be 
done at a higher level, typically the session level. 

@author hursh jain
**/
public abstract class FormValidator 
{
protected String 	name;
protected String 	errorMessage;

/** Creates a new validator and adds it to the specified
form.

@parm	f				the form to validate
@pararm	name			an arbitrary name for this group validator (used in
						debugging and logging messages)
@param	errorMessage	the error message associated with invalid data
						See {@link #getErrorMessage}

**/
public FormValidator(Form f, String name, String errorMessage)
	{
	this.name = name;
	this.errorMessage = errorMessage;
	f.addValidator(this);
	}

/** 
Validates multiple fields together. Note, implementations need
not call the validator for each field if that field also has a field
level validator (since each field is individually validated by the
form). Implementations should validate multiple fields to see if
they make sense when analysed as a whole. 
<p>
<b>Important notes</b>:
Typically, validation should be skipped in the following circumstances:
<ul>
<li>The field is disabled.
<li>The field is an instance of {@link DependentField} and the
{@link DependentField#shouldValidate} method returns <tt>false</tt>.

@return	<tt>true</tt> is validation succeeded, <tt>false</tt> otherwise
**/
abstract public boolean validate(FormData fd, HttpServletRequest req);

public String getErrorMessage() {
	return errorMessage;
	}

/**
Sets the error message for this validator -- useful when
the error message need to be changed dynamically.
*/
public void setErrorMessage(String message) {
	this.errorMessage = message;
	}
	
public String getName() {
	return name;
	}	

public String toString() {
	return "Form validator: " + name;
	}

}          //~class FormValidator