// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import jakarta.servlet.*;
import jakarta.servlet.http.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;
import java.sql.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
A dependent HTML DependentSelect field. The values of this field change based
on the chosen value of some other field in the form. These new values are
calculated/set on the server side. 
<p>
Another good (and equally valid) approach is to instead use javascript arrays
(with different sets of values) and swap out values on the client and/or
use AJAX on the client side.

@author hursh jain
**/
public final class DependentSelect extends DependentField
{
public static class Data
	{
	//---------- Select Data -------------
	private LinkedHashMap 	options  = new LinkedHashMap();
	//the initially selected options when form was constructed
	private Map				origSelectedMap = new HashMap();
	private int 			size = 0;
	private boolean 		multiple;
	//---------- Client Submit Data -------
	//[option1.getValue() -> option1] ...
	private Map  selectedMap = null;  //created if there is data	
	//[option1, option2....]
	private List selectedList = null; //created if there is data
	//-------------------------------------
	/**
	Convienence method for use by the {@link Dependency#getInitialValues}
	method. 
	*/
	public void addOption(Select.Option item) {
		String itemval = item.getValue();
		options.put(itemval, item);
		if (item.isOrigSelected()) {
			origSelectedMap.put(itemval, item);	
			}	
		}
	
	public void clearSubmittedData() {
		if (selectedMap != null)
			selectedMap.clear();
		if (selectedList != null)
			selectedList.clear();
		}
	
	public String toString() { 
		return new ToString(this, ToString.Style.VisibleLevel.PRIVATE)
		.reflect().render(); 
		}
	}

/** 
Creates a new select element with no initial values. Multiple
selections are not initially allowed.

@param 	name		the field name
**/
public DependentSelect(String name)
	{
	super(name);
	}
	
public Field.Type getType() {
	return Field.Type.SELECT;
	}

/** 
Returns a List containing the selected options. Each object in
the collection will be of type {@link Select.Option}. If there
are no selected options, the returned list will be an empty list.

@param	fd 	the submited form data
**/
public List getValue(FormData fd) 
	{
	Data data = (Data) fd.getData(name);
	if (data == null || data.selectedList == null) {
		return Form.empty_list;
		}
	return data.selectedList;
	}

/** 
Convenience method that returns the selected option as a String. <u>No
guarantees are made as to which selection is returned when more than one
selection is selected (this method is really meant for when the select only
allows single selections as a dropdown).</u>
<p>
The returned value is of type String obtained by called the selected
Select.Option's {@link Select.Option#getValue} method.
<p>
If there is no selection, returns <tt>null</tt>
**/
public String getStringValue(FormData fd) 
	{
	Data data = (Data) fd.getData(name);
	if (data == null) {
		return null;
		}

	List list = data.selectedList;
	if (list == null || list.size() == 0) {
		return null;
		}
	else {
		Select.Option opt = (Select.Option) list.get(0);
		return opt.getValue();
		}
	}


/**
Convenience method that returns the single value of this field
as a Integer.
<p>
All the caveats of {@link #getSingleValueAsString()} apply.

@throws NumberFormatException	if the value could not be
								returned as in integer.	
*/
public int getIntValue(FormData fd) {
	return Integer.parseInt(getStringValue(fd));
	}

/**
Convenience method that returns the single value of this field
as a boolean. 
<p>
All the caveats of {@link #getSingleValueAsString()} apply.
In particular, the formdata should contain non-null data
with at least one selection.
*/
public boolean getBooleanValue(FormData fd) {
	return Boolean.valueOf(getStringValue(fd)).booleanValue();
	}

/**
Returns <tt>true</tt> if some option has been selected by the user,
<tt>false</tt> otherwise. (also returns <tt>false</tt> is the specified
form data is <tt>null</tt>).
*/
public boolean isFilled(FormData fd) 
	{
	if (fd == null)
		return false;
		
	Data data = (Data) fd.getData(name);
	
	if (data == null) 
		return false;
	
	List list = data.selectedList;
	return (list != null && list.size() != 0);
	}

/**
@see Select#setValueFromSubmit
*/
public void setValueFromSubmit(FormData fd, HttpServletRequest req) 
	{
	dependency.setDependencyDataFromSubmit(fd, req);
	
	//value(s) associated with the selection field name
	String[] values = req.getParameterValues(name);
	
	Data data = new Data();
	fd.putData(name, data);

	if (values == null) {
		return;
		}

	data.selectedMap = new HashMap();
	data.selectedList = new ArrayList();
				
	for (int n = 0; n < values.length; n++) {
		Select.Option opt = new Select.Option(values[n]);
		data.selectedMap.put(opt.getValue(), opt);
		data.selectedList.add(opt);
		}
	}

public void renderImpl(FormData fd, Writer writer) 
throws SQLException, IOException
	{
	Data data = null;
	
	if (fd == null) {
		data = (Data) dependency.getInitialValues(this);
		}
	else{
	//Case 1. Form submitted
	//	dependency updates the data object for this field in the formdata.
	//	_after_ the form is submitted. The dependency will set render data
	//  in a non-null data object (creating it if necessary).
	//	(even though the submitted data parts in the data object can be null)
	//
	//Case 2. Form not submitted but FormData created for initial values.
	// The dependency will not be invoked by the form handling
	//	mechanism so we still need to get initial values manually. 
	//
		data = (Data) fd.getData(name);
		if (data == null) {
			data = (Data) dependency.getInitialValues(this);
			}
		}
	
	Argcheck.notnull(data, "Internal error: unexpected state, data was null");
		
	writer.write("<select");	
	writer.write(" name='");
	writer.write(name);
	writer.write("'");
	
	if (data.size > 0) {
		writer.write(" size='");
		writer.write(String.valueOf(data.size));
		writer.write("'"); 
		}
		
	if (data.multiple) {
		writer.write(" multiple"); 
		}	
	
	if (! enabled || ! isEnabled(fd)) {
		writer.write(" disabled");
		}
		
	if (renderStyleTag) {
		writer.write(" style='");
		writer.write(styleTag);
		writer.write("'");
		}
		
	final int arlen = arbitraryString.size();
	for (int n = 0; n < arlen; n++) {
		writer.write(" ");
		writer.write(arbitraryString.get(n).toString());
		}
		
	writer.write(">\n");
	
	Iterator it = data.options.values().iterator(); 
	while (it.hasNext()) 
		{
	 	Select.Option item = (Select.Option) it.next();
	 	String itemval = item.getValue();
	 	
		boolean selected = false;
	
		if (fd != null) 	/* maintain submit state */
			{					
			if (data.selectedMap != null){
				selected = data.selectedMap.containsKey(itemval);
				}
			else {  
				//form submitted but no submit data for this field
				//hence selecteMap == null
				}
			}
		else{  	 /* form not submitted, show original state */
		  	selected = data.origSelectedMap.containsKey(itemval);
		   	}
	
		writer.write(item.render(selected));
		writer.write("\n");	//sufficient for easy view source in browser		
		}	

	writer.write("</select>\n");

	dependency.renderDependencyData(fd, writer);
	}
						
/** 
<tt>true</tt> if multiple selections are allowed, <tt>false</tt> otherwise. 
This value (if set) is rendered as the html <tt>MULTIPLE</tt> tag.

@return	this object for method chaining convenience
**/
public DependentSelect allowMultiple(FormData fd, boolean allow) 
	{
	Data d = getDataObject(fd);
	d.multiple = allow;
	return this;
	}	

/** 
This value (if set) is rendered as the html <tt>SIZE</tt> tag. 
If the list contains more options than specified by size, the browser 
will display the selection list with scrollbars.

@return	this object for method chaining convenience
**/
public DependentSelect setSize(FormData fd, int size) 
	{
	Data d = getDataObject(fd);
	d.size = size;
	return this;
	}	
	
/**
Sets the selected values for this select in the specified form data. This
is useful for showing different <i>initial</i> values to each user
(before the form has been submitted by that user). The values are set in
a newly created {@link DependentSelect.Data} object which is then stored
inside the form data. Since these are set in the FormData object, these
values are request specific and can differ per user.
<p>
If the form has not been submitted, there is no form data object. A form
data object should be manually created if needed for storing the value.

@param	data	an existing form data object.
@param	values	a list of {@link Select.Option} objects
*/
public void setValue(FormData fd, List values)
	{
	Argcheck.notnull(fd, "specified fd param was null");

	DependentSelect.Data data = new DependentSelect.Data();
	fd.putData(name, data);
	setValue(data, values);
	}
	
/**
Convenience method to set values in the specified {@link
DependentSelect.Data} object. 
The following two are equivalent:
<blockquote>
<pre>
	List values .....
	DependentSelect.Data data = new DependentSelect.Data();
	fd.putData(name, data);
	setValue(<i>data</i>, values);
</blockquote>
</pre>
and
<blockquote>
<pre>
	List values .....
	setValue(<i>fd</i>, values);
</blockquote>
</pre>
@param	data	an existing DependentSelect.Data object.
@param	values	a list of {@link Select.Option} objects
*/
public void setValue(DependentSelect.Data data, List values)
	{
	Iterator it = values.iterator();
	while (it.hasNext()) {
		Select.Option item = (Select.Option) it.next();
		String itemval = item.getValue();
		data.options.put(itemval, item);
		if (item.isOrigSelected()) {
			data.origSelectedMap.put(itemval, item);	
			}
		}		
	}
	
/**
Returns the data object for this field from the formdata. returns
the initial data (if applicable) if the form is being rendered for
the first time or if no dependent data has been created.
*/
Data getDataObject(FormData fd)
	{
	Data d = (Data) fd.getData(name);
	return d;
	}
		
}          //~class DependentSelect