// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import jakarta.servlet.*;
import jakarta.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;


/** 
Represents a field that:
<ol>
<li>returns one (1) string as their value. Contrast this with
selects/with/multiple and radiogroups that may return
a String[].
<li>
In addition, the <i>name</i> of this field is always
returned on a form submit by the browser (barring browser
hacks etc). For example, if a text field is not filled out
by the user, the browser still returns that text field (with
no value). Contrast this with radio buttons that are not
returned at all (not even their name) if the user does not
select them.
</ol>
<p>Important note: the {@link trimValues(boolean)} method
will by default trim whitespace from the beginning/end of
user input. If the resulting string is empty (""), then 
it will be treated as if the user had not entered anything.
(which is typically the desired behavior).

@author hursh jain
**/
public abstract class AbstractText extends Field
{
static class Data {
	String 	 value;
	}

String 	orig_value;
boolean trimValues = true;

AbstractText (String name) {
	super(name);
	}

AbstractText (String name, String value) 
	{
	super(name);
	if (value == null) 
		{
		log.warn("specified value was null, defaulting to \"\"");
		value = "";
		}
	orig_value = value;
	}

/**
Sets the value of the field from the submitted form data.

<b>Note, by default, it is assumed that all abstract text fields have
different names in the html form</b>. Mulitple fields with the same name
are not supported. <u>If there is more than one field with the same name
(such as multiple text fields with the same name), then only 1 value will
be saved (and the rest will be ignored)</u> [This is one of - or perhaps
the only - restriction of this form API; without it, things become too
complex to manage].
*/
public void setValueFromSubmit(FormData fd, HttpServletRequest req) 
throws SubmitHackedException
	{
	String value = req.getParameter(name);
	
	//can be null if not sent by client for some reason
	//will be "" is not isFilled by the client since the servlet
	//api, for the request params 'foo=&bar=, getParam("foo")
	//returns an empty string (not null).
	//	
	if (value == null) {  
		//Browsers return no value for disabled fields !!
		if (! enabled || ! isEnabled(fd))
			return;
		//client was buggy or hacked
		hacklert(req, "Did not find ["+name+"] field in the request (but expected to), defaulting to \"\"");
		return;
		}
	
	if (trimValues) {
		value = value.trim();
		}
		
	//servlet api returns an empty string for param 'foo=' [not null]
	//we treat that as a null value, we also treat white space
	//as null if trimming is on.
	if (value.equals(""))  
		return;			   
		
	//we have a non-null value
	AbstractText.Data data = new AbstractText.Data();
	fd.putData(name, data);
	data.value = value;
	}

/** 
Sets the <b>initial</b> value of this text element to the specified string.
If the specified value is <tt>null</tt>, then the initial value is set to
"" (the empty string).
**/
public void setValue(String value) 
	{
	if (value == null) {
		log.warn("specified value was null, defaulting to \"\"");
		value = "";
		}
	this.orig_value = value;
	}	

/**
Sets the selected values for this select in the specified form data. This
is useful for showing different <i>initial</i> values to each user (before
the form has been submitted by that user).
<p>
If the form has not been submitted, there is no form data object. A form
data object should be manually created if needed for storing the value.

@param	fd		the non-null form data used for rendering the form
@param	value	the value to be set
*/
public void setValue(FormData fd, String value) 
	{
	Argcheck.notnull(fd, "specified fd param was null");

	if (value == null) {
		log.warn("specified value was null, defaulting to \"\"");
		value = "";
		}
	AbstractText.Data data = new AbstractText.Data();

	fd.putData(name, data);
	data.value = value;
	}	


/** 
Returns a string representing the value of this field or <tt>null</tt> if
there is no current value. The <i>names</i> of Form fields like
<tt>text</tt> and <tt>textarea</tt> are always sent back by the browser
if the field is enabled. However if the <i>values</i> are empty, then
this method will return <tt>null</tt>. For example, if the browser
returned <xmp>foo=&bar=baz</xmp>, then <tt>foo</tt> will have a
<tt>null</tt> value (even though the <i>servlet api</i> returns an empty
string).
*/
public String getValue(FormData fd) 
	{
	AbstractText.Data data = (AbstractText.Data) fd.getData(name);
	
	if (data == null)
		return null;
			
	return data.value;
	}


/**
Convenience method that returns the value of this field as a String.
<tt>null</tt> values (ie which can happen when the field is disabled and
no value sent by the browser) <i>are returned as an empty string.</i>

@throws NumberFormatException	if the value could not be
								returned as in integer.	
*/
public String getStringValue(FormData fd)
	{
	String  s = getValue(fd);
	if (s == null)
		s = "";
	return s;
	}

/**
Convenience method that returns the value of this 
field as a Integer. 

@throws NumberFormatException	if the value could not be
								returned as an integer.	
*/
public int getIntValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Integer.parseInt(value);
	}

/**
Convenience method that returns the value of this field as a
Short.

@throws NumberFormatException	if the value could not be
								returned as a short.	
*/
public short getShortValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Short.parseShort(value);
	}

/**
Convenience method that returns the value of this field as a
Float.

@throws NumberFormatException	if the value could not be
								returned as a float.	
*/
public float getFloatValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Float.parseFloat(value);
	}


/**
Convenience method that returns the value of this field as a
Double.

@throws NumberFormatException	if the value could not be
								returned as a double.	
*/
public double getDoubleValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Double.parseDouble(value);
	}

/**
Convenience method that returns the value of this field as a boolean. The
value is converted into a boolean as per the {@link
Boolean.valueOf(String)} method.
*/
public boolean getBooleanValue(FormData fd) {
	return Boolean.valueOf(getValue(fd)).booleanValue();
	}

/**
The value to render this field with. Maintains state and returns 
whatever the user typed in last if applicable. If the form has
not been shown to the user or if this field is disabled at all, returns 
the original value.
*/
String getRenderValue(FormData fd) 
	{
	if (! enabled) {
		return orig_value;
		}
		
	if (fd != null) 
		{
		AbstractText.Data data = (AbstractText.Data) fd.getData(name);	
		if (data == null)
			return "";
		else	
			return data.value;
		}
	else { //fd == null, no form data, showing form for first time
		return orig_value;
		}
	}

public boolean isFilled(FormData fd) 
	{
	AbstractText.Data data = (AbstractText.Data) fd.getData(name);
	
	//data is created only if the submitted value was not null. If
	//data is not null, then there was a sumbitted value.
	if (data == null)
		return false;
	
	String value = data.value;
	
	if (value == null) {
		log.error("Internal error: unexpected state");
		}
	
	//since it's always at least some string -- possibly
	//all spaces but it'll be non-null
	return true;
	}

public void reset(FormData fd) {
	AbstractText.Data data = (AbstractText.Data) fd.getData(name);
	if (data != null)
		data.value = orig_value;
	}

/**
Trims leading and ending spaces from all entered values. 
This is <tt>true</tt> by default. Specify <tt>false</tt> to
turn this off.
*/
public void trimValues(boolean val)
	{
	trimValues = val;
	}

public String toString() 
	{
	return super.toString() + "; Orig. value: [" + orig_value + "]"; 
	}	

}        