// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util.pagetemplate;

import java.io.*;
import java.util.*;

import fc.util.*;
import fc.io.*;
import fc.jdbc.*;

/**
A page template is a simplified molly page, that runs in a non web environment. Good for creating
dynamic web pages (for email campaigns, etc) from the command line.
<p>
There is another alternate/simpler template mechanism here: {@link fc.util.Template}.
<p>
The page template loads template files (can have any suffix but .template is suggested as
opposed to .mp to keep the non-web distinction clear). The template file is compiled
(uses the system temp directory for compilation) as needed and the resulting class file
is loaded/run. If a template is changed, it is automatically recompiled, reloaded and
rerun. If the template has a compilation error, that page remains unloaded until the
error is fixed.
<p>
The output of the template is written to an output destination (could be a buffer, string, file, etc)
and the invoker can then send that output to an email destination.
<p>
Template pages only use the system classpath for compilation. Pages can refer to any class
found in the system classpath. 
<p>
Here is some sample usage:
<pre><code>
TemplatePage pt = TemplateMgr.<b>getTemplate</b>("/path/to/file.template");
pt.<b>render</b>(new File("/path/to/output.file"));

//To optionally send information to the page, you can say (before calling render):
Map m = new HashMap();
m.put(...whatever..);
pt.setContext(m);
</code></pre>

@author hursh jain
*/
public final class TemplateMgr
{
private static final boolean dbg = false;

static Map	pagemap = new HashMap();
static Log	log;
static File	tmpdir; 
static String classpath;

/**
Returns a runnable page template, alias for the {@link getTemplate(String)} method.

@param	templateFile	the full path to the template file
*/
public static TemplatePage get(String templateFilePath) throws Exception
	{
	return getTemplate(new File(templateFilePath));	
	}

/**
Returns a runnable page template, alias for the {@link getTemplate(File)} method.

@param	templateFile	the full path to the template file
*/
public static TemplatePage get(File templateFile) throws Exception
	{
	return getTemplate(templateFile);	
	}

/**
Returns a runnable page template.

@param	templateFile	the full path to the template file
*/
public static TemplatePage getTemplate(String templateFilePath) throws Exception
	{
	return getTemplate(new File(templateFilePath));	
	}

/**
Returns a runnable page template.

@param	templateFile	the template file
*/
public static TemplatePage getTemplate(File templateFile) throws Exception
	{
	if (! templateFile.exists())
		throw new IllegalArgumentException("The specified template file " + templateFile + "] does not exist. How am I supposed to load it, eh ?");

	StringBuilder buf = new StringBuilder(1028);

	log = Log.getDefault();
	classpath = System.getProperty("java.class.path");
	tmpdir = new File(System.getProperty("java.io.tmpdir"));

	TemplatePage page = null;

	final CharArrayWriter fbuf = new CharArrayWriter(128);

	String classname = getClassNameFromPageName(templateFile);
	
	File javafile 	= new File(tmpdir, classname + ".java");
	File classfile 	= new File(tmpdir, classname + ".class");

	if (dbg) {
		System.out.println(
			String.format("tmpdir=%s, TemplatePage=%s, javafile=%s, classfile=%s\n",
					tmpdir, templateFile, javafile,classfile));
		}

	synchronized (TemplateMgr.class) 
		{		
		long page_modified 	= templateFile.lastModified();
		long java_modified 	= javafile.lastModified();  //returns 0 if !exist
		long class_modified = classfile.lastModified();	//returns 0 if !exist

		if (dbg)
			{
			System.out.format(
			" %-20s %10d\n %-20s %10d\n %-20s %10d\n", 
					"Modified: page:", page_modified, 
					"java:", java_modified, 
					"class:", class_modified);
			}
		
		if ( java_modified == 0L || page_modified > java_modified) 
			{
			if (dbg) System.out.format("page_mod > java_mod, parsing the page.........\n");
			TemplateParser parser = new TemplateParser(templateFile, javafile, classname, log);

			log.bug("PARSING page:", javafile.getPath());

			try {
				parser.parse();
				}
			catch (IOException e) {
				//the parser may write a partially/badly written file if the parse failed.
				if (javafile.exists()) {
					javafile.delete();
					}
				throw e; //rethrow the parse exception
				}
				
			java_modified 	= javafile.lastModified();  //since newly parsed
			}
	
		boolean forceReload = false;
		//Java source could be generated or hacked by hand
		// if nothing needs compiling, then we still need to load the
		// page the first time it's accessed
		if ( class_modified == 0L || java_modified > class_modified) 
			{
			if (dbg) System.out.format("java_mod > class_mod, compiling the java source.........\n");
			
			log.bug("COMPILING page:", javafile.getPath());
			
			//src_encoding can be null, that's fine.
			TemplateCompiler pc = new TemplateCompiler(javafile, classpath);
			
			if (! pc.compile())
				throw new TemplateParseException(pc.getError());
			
			forceReload = true; //since we recompiled, we reload even if
								//page exists in the page cache
			}

		final boolean page_in_map = pagemap.containsKey(templateFile.getAbsolutePath());
		
		if (forceReload || ! page_in_map)
			{
			TemplateClassLoader loader = new TemplateClassLoader(log);
			Class c = loader.loadClass(TemplatePage.PACKAGE_NAME + "." + classfile.getCanonicalPath());
			
			if (dbg) System.out.println("class = " + c);
			page = (TemplatePage) c.newInstance();
			if (dbg) System.out.println("page = " + page);
			page.setSourcePath(templateFile);	
					
			if (page_in_map) {
				TemplatePage oldpage = (TemplatePage) pagemap.get(templateFile.getAbsolutePath());
				//notused
				//oldpage.destroy();
				}

			pagemap.put(templateFile.getAbsolutePath(), page);
			}
		else{
			page = (TemplatePage) pagemap.get(templateFile.getAbsolutePath());
			}
		}		
	
	if (dbg) log.bug("Returning PAGE=", page);
 	return page;
	}

/*
a.template 	-> 	a_template
b.template 	-> 	b_template
c.template 	-> 	c_template
p/q.template  ->  p_q.template
5!#.template -> 	5__template   ---> name clash 
5!!.template ->   5__template   ---> name clash

So a simplistic mapping(any bad classname chars--> '_') will not work.
We should therefore hex-encode every special char..not sure if doing this right now.
*/
static String getClassNameFromPageName(File templateFile) throws IOException
	{
	if (dbg) System.out.println("getClassNameFromPageName(), page=["+templateFile+"]");
		
	StringBuilder buf = new StringBuilder();
	char c;
	
	String path = templateFile.getAbsolutePath();
	
	c = path.charAt(0);
	if (c == '/' || c == File.separatorChar) {
		buf.append("_");
		}
	else if (! Character.isJavaIdentifierStart(c)) {
		buf.append(Integer.toHexString(c));
		}
	else {
		buf.append(c);
		}
	
	for (int n = 1; n < path.length(); n++) 
		{
		c = path.charAt(n);
		
		if (c == '/' || c == File.separatorChar || c == '.') {
			buf.append("_");
			}
		else {		
			if (! Character.isJavaIdentifierPart(c))
				buf.append(Integer.toHexString(c));
			else 
				buf.append(c);
			}
		}
			
	String str = buf.toString();

	if (str.length() > 1 && str.startsWith("_")) {
		str = str.substring(1);  //starting with _ is ugly
		}
		
	return str;
	}

		
/**
Interactive page manager use for testing.
*/
public static void main (String args[]) throws Exception
	{
	Args myargs = new Args(args);
	myargs.setUsage("java fc.util.template.TemplateMgr \n"
		+ "-file <path-to-page-template-file>\n"
		+ "-out <path-to-output-file>\n"
        + " ------------------- optional ------------------ \n"
	    + "-conf <db-config file>\n"
        + "-dbg|debug (turns log level to debug)\n"
        );

	log = Log.getDefault();
	log.setLevel(Log.INFO);

	if (myargs.flagExists("dbg") || myargs.flagExists("debug")) {
			log.setLevel(Log.DEBUG);
			}

	Map m = new HashMap();
	
	String conf = myargs.get("conf");
	if (conf != null) { 
		PropertyMgr props = new FilePropertyMgr(new File(conf));
		ConnectionMgr cmgr = new SimpleConnectionMgr(props);
		m.put("cmgr", cmgr);
		}
	
	TemplatePage pt = TemplateMgr.getTemplate(myargs.getRequired("file"));
	pt.setContext(m);
	pt.render(new File(myargs.getRequired("out")));
	} //main

}
