// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;
import java.text.*;

/** 
Useful to store thread-local number format instances.  This class is
intended for servlets/molly pages. Instead of static get/set methods,
this class must be instantiated and the instance methods used to get/set
the calendar object. This allows multiple instances of this class in the
webapp, with each instance being able to get/set a separate calendar. [If
the methods in this class were static, then only 1 calendar could be
get/set per thread].
<p>
<font color=red><b>Each thread must remember to individually create a
separate calendar instance and store it via the set method</b></font>. The
usage idiom is:
<blockquote>
<pre>
<font color=blue>
//WebApp has a map of ThreadLocalNumberFormat's and also a instance variable
//pointing to a default ThreadLocalNumberFormat
ThreadLocalNumberFormat mynf = WebApp.getThreadLocalNumberFormat("foo");
</font><font color=red>
if (mynf.isNull())  {
	mynf.set(NumberFormat.getInstance());
	}
</font><font color=blue>
NumberFormat nf = mynf.get();
</font>
</pre>
</blockquote>
Note, the lines in red are always needed anywhere/anytime this class is used.
(note, it just so happens that the {@link NumberFormat#getInstance} returns (at
least for now) a new object every time it is called, which is why it is
being used in the example above, else we would have to manually create a new
NumberFormat instance instead).
*/
public final class ThreadLocalNumberFormat
{
public ThreadLocalNumberFormat()
	{ }
/*
Each get/set into the threadlocal must be seperately by each thread (the
initialValue() method is good for auto-assigning a new value but we
may need to assign a custom calendar value per thread, so we can't use
initialValue()
*/
private final ThreadLocal tlnf = new ThreadLocal();

public NumberFormat get()
	{
	return (NumberFormat) tlnf.get();
	}

public void set(NumberFormat nf)
	{
	tlnf.set(nf);
	}

public boolean isNull()
	{
	return tlnf.get() == null;
	}

public static void main (String args[]) throws Exception
	{ 
	final ThreadLocalNumberFormat nf1 = new ThreadLocalNumberFormat();
	
	Thread t1 = new TestThread(false, nf1);
	Thread t2 = new TestThread(false, nf1);
	t1.start();
	t2.start();
	
	for (int n = 0; n < 100; n++) {
		new TestThread(true, nf1).start();
		}
	}

static class TestThread extends Thread
	{
	boolean timing_only = false;
	ThreadLocalNumberFormat nf;
	TestThread(boolean timing_only, ThreadLocalNumberFormat nf) {
		this.timing_only = timing_only;
		this.nf = nf;
		}
		
	public void run()
		{
		//warm-up
		if (nf.isNull()) {
			nf.set(NumberFormat.getInstance());
			}

		Watch w = new NanoWatch();
		w.start();
		if (nf.isNull()) {
			//all threads will have the same instance but thats ok
			//we are testing the time taken to get a thread local
			//variable, we don't have to create a new number format
			//per thread for this example (but we DO for real-world
			//since these are not thread safe formatters). We use
			//a thread-specific formatter below (nf)
			nf.set(NumberFormat.getInstance());
			}
		NumberFormat c = nf.get();
		w.stop();
		
		NumberFormat nf = NumberFormat.getNumberInstance();
		nf.setMinimumFractionDigits(2);
		nf.setMaximumFractionDigits(2);
		nf.setMinimumIntegerDigits(2);
	
		if (timing_only)
			System.out.println("ThreadLocal get: [" + nf.format(w.getTime() / 1000000.00D) 
				+ " ms]");
		else
			System.out.println("[" + nf.format(w.getTime() / 1000000.00D) 
				+ " ms]" + Thread.currentThread() 
				+ "/NumberFormat-ID:[" + System.identityHashCode(c) 
				+ "] "+ c);
		}
	}
}