// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;

/** 
A range of dates.
<p>

@author hursh jain
**/
public class DateRange extends Range
{
Date start;
Date end;

/** 
Constructs a date range between the specified start and end 
dates (both inclusive).

@throws IllegalArgumentException 
		if the end date is lesser (via an <tt>compareTo</tt> 
		comparison) than the start date (equal to the start date 
		is ok)
**/
public DateRange(Date start, Date end) 
	{
	Argcheck.istrue(start.compareTo(end) <= 0, "end lesser than start in the specified range, start=["+start+"], end=["+end+"]");
	this.start = start;
	this.end = end;
	}

/** Get the start date for this range **/
public Date getStart() 
	{
	return start;
	}


/** Get the end date for this range **/
public Date getEnd() 
	{
	return end;
	}
	
/** 
@return true if the passed in date is inside this range. Note,
the <u>entire</u> date (including the milliseconds/seconds portion)
is considered by this function.

@throws IllegalArgumentException
		if the specified date was null
**/
public boolean inRange(Date d) 
	{
	Argcheck.notnull(d);
	boolean result = d.compareTo(start) >= 0 &&
					 d.compareTo(end) <= 0;
	
	if (negated)
	 	return !result;
	else
		return result;
	}
 
 
/** 
@return true if any part of the passed in date range overlaps this range. Note,
the <u>entire</u> date (including the milliseconds/seconds portion) is considered by this function.

@throws IllegalArgumentException
		if the specified date was null
**/
public boolean overlaps(DateRange dr) 
	{
	Argcheck.notnull(dr);
	
	boolean result = inRange(dr.getStart()) || inRange(dr.getEnd());			
	if (negated)
	 	return !result;
	else
		return result;
	} 

 
/** 
Set the start date for this range 

@throws NullPointerException
		if the specified date was null
@throws IllegalArgumentException 
		if the specified date is greater (via an <tt>compareTo</tt> 
		comparison) than the currently set end date (equal to the 
		end date is ok)
**/
public void setStart(Date d) 
	{
	Argcheck.notnull(d, "specified argument was null");
	Argcheck.istrue(d.compareTo(end) <= 0, "start greater than end date, specified start=["+d+"], end=["+end+"]");
	start = d;
	}

/** 
Sets the end date for this range 

@param	d	the end date

@throws NullPointerException
		if the specified date was null
@throws IllegalArgumentException 
        if the specified end date is lesser (via an <tt>compareTo</tt> 
		comparison) than the currently set start date (equal to 
		the start date is ok)
**/
public void setEnd(Date d) 
	{
	Argcheck.notnull(d, "specified argument was null");
	Argcheck.istrue(d.compareTo(start) >= 0, "end lesser than start, start=["+start+"], specified end=["+d+"]");
	end = d;
	}


/** Output a string representation of the date range **/
public java.lang.String toString() 
	{
	String str = "DateRange:[";
	if (isNegated())
		str += "^";
	str += start + "," + end + "]";
	return str;
	}

public static void main(String[] args)
	{
	Calendar cal = Calendar.getInstance();
	Date d1 = cal.getTime();
	
	//roll back 1 month
//	cal.set(Calendar.MONTH, false);
	cal.add(Calendar.MONTH, -1);
	Date d2 = cal.getTime();
	   
	DateRange r = new DateRange(d2, d1);
	System.out.println("constructed range:" + r);
	//normal
	test(r);
	
	System.out.println();

	DateRange r2 = new DateRange(d2, new Date(d1.getTime() - CalendarUtil.ONE_DAY));
	System.out.println("r1:" + r + "\n    -- overlaps? --\n" + "r2:" + r2);
	System.out.println();
	System.out.println("r1 overlaps r2 (should be true): " + r.overlaps(r2));
	System.out.println("r2 overlaps r1 (should be true): " + r2.overlaps(r));

	System.out.println("----------------------------------");

	r2 = new DateRange(
	 	new Date(d1.getTime() + CalendarUtil.ONE_DAY),
		new Date(d1.getTime() + CalendarUtil.THREE_DAY));

	System.out.println("r1:" + r + "\n    -- overlaps? --\n" + "r2:" + r2);
	System.out.println();
	System.out.println("r1 overlaps r2 (should be false): " + r.overlaps(r2));
	System.out.println("r2 overlaps r1 (should be false): " + r2.overlaps(r));

	System.out.println("----------------------------------");

	r2 = new DateRange(
	 	new Date(d2.getTime() - CalendarUtil.THREE_DAY),
		new Date(d2.getTime() - CalendarUtil.ONE_DAY));

	System.out.println("r1:" + r + "\n    -- overlaps? --\n" + "r2:" + r2);
	System.out.println();
	System.out.println("r1 overlaps r2 (should be false): " + r.overlaps(r2));
	System.out.println("r2 overlaps r1 (should be false): " + r2.overlaps(r));

	System.out.println("----------------------------------");

	}

private static void test(DateRange r) 
	{
	Date d = new Date();

	System.out.println("in range: " + d + "=" + r.inRange(d));

	Calendar cal = Calendar.getInstance();
	cal.setTime(d);
	cal.add(Calendar.MONTH, -2);

	d = cal.getTime();
	System.out.println("in range: " + d + "=" + r.inRange(d));
	
	cal.add(Calendar.MONTH, 2);
	cal.roll(Calendar.DAY_OF_MONTH, false);
	d = cal.getTime();	
	System.out.println("in range: " + d + "=" + r.inRange(d));
	}

}          //~class CharRange