// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.net.*;
import java.security.*;

public final class ClassUtil
{
static final String nl = fc.io.IOUtil.LINE_SEP;
static final boolean dbg = false;

/**
Returns classloader information about the specified object
*/
public static String getClassLoaderInfo(final Object obj)
	{
	Argcheck.notnull(obj, "specified obj param was null");
	return getClassLoaderInfo(obj.getClass());
	}

/**
Returns classloader information about the specified class.
*/
public static String getClassLoaderInfo(final Class c)
	{
	Argcheck.notnull(c, "specified class param was null");
	
    ClassLoader cl = c.getClassLoader();

    StringBuffer buf = new StringBuffer(512);

	buf.append("============== Classloader Information ==================");
	buf.append(nl);
  
    buf.append("System classloader = ");
    buf.append(ClassLoader.getSystemClassLoader());
	buf.append(nl);
    
    Thread t = Thread.currentThread();
    buf.append("Current thread = ").append(t);
	buf.append(nl);
    buf.append("Current thread context classloader = ");
    buf.append(t.getContextClassLoader());
	buf.append(nl);

    buf.append("Class loader hierarchy: ");
    buf.append(nl);
    buf.append("[").append(c);
	buf.append("][package:");
	Package p = c.getPackage();
	buf.append(p != null ? p.getName() : "N/A");
	buf.append("]");
	
	buf.append("/ClassID:");
	buf.append(System.identityHashCode(cl));
	buf.append(" loaded by:");
	buf.append(nl);

	String pad = "  ";
	while (cl != null) {
		buf.append(pad);
		buf.append(cl);
		buf.append(nl);
		cl = cl.getParent();
		pad += "   ";
		}
    buf.append(pad + "null (bootstrap class loader)");
	buf.append(nl);
	
	buf.append("=========================================================");
	buf.append(nl);            
    return buf.toString();
    }

/* to get the class name from a static method */
//securitymanager is deprecated, keeping this still because stack walker is
//brain-deadingly complex and stupid
private static class CurrentClassGetter extends SecurityManager {
	public String getClassName() {
		return getClassContext()[2].getName();
		}
	}

/**
This method can be called from either a instance <b>or</b> a static method.

Returns the class name in which the calling method was declared. 
<p> 
<b>Important note:</b> The returned class name is wherever the method is invoked from. For
instance methods, it is the runtime virtual class. For static it is the class name where the method
was declared (static methods are not virtual and bound at compile time).

For example:
<blockquote><pre><tt>
Class A 
	{	
	static void foo(...) {
		System.out.print(ClassUtil.getClassName());
		}
	}

public class B extends A
	{
	}
	
% java B 	
will print:
	A (NOT B)
</pre></tt></blockquote>
*/
public static String getClassName()
	{
	return new CurrentClassGetter().getClassName();
	}


/** 
Returns the fully qualified name of the class that contained the <tt>main()</tt> method used to
invoke the application. 

<b>Note: </b> this method must be called from the same thread (the "main" thread) that started the
application. Newly started threads that call this method will instead obtain the class name
containing that thread's <tt>run()</tt> method.
**/
public static String getMainClassName() 
	{
   	final StackTraceElement[] st = new Throwable().getStackTrace();
	//System.out.println("stack trace=" + Arrays.asList(st));
	final StackTraceElement mainElement = st[st.length-1];
	final String className = mainElement.getClassName();
	return className;
	}

/**
Returns true is the specified class is a system/JDK class. This method should work, but given
the vagaries of classes, classloaders, etc, there isn't any guarantee.

@param	caller	the object <i>or</i> {@link java.lang.Class} Class  corresponding to that object
*/
public static boolean isSystemClass(final Class c) 
	{
	/* The param HAS to be Class, cannot be Object
	
	isSystemClass(Object caller)
		{
		Class c = caller.getClass();
		...
		}
		
	does NOT work - since
	==============ClassLoaderTest.java==
	public static void main (String args[]) {
		new ClassLoaderTest();
		}
	public void printClassLoader(Object obj) {
		System.out.println(obj);
		Class c = obj.getClass();
		System.out.println(c);
		System.out.println(c.getClassLoader());
		}
	public ClassLoaderTest() {
		printClassLoader(ClassLoaderTest.class);  
//====>		class ClassLoaderTest => class java.lang.Class => null

		printClassLoader(this);   
//====>		ClassLoaderTest@4e25154f => class ClassLoaderTest => sun.misc.Launcher$AppClassLoader@15db974
	}	
	==============
	*/
		
	//all of the below return null. so this is true for java.* as well as javax.* at least.
	//System.out.println(String.class.getClassLoader()); 
	//System.out.println(Object.class.getClassLoader());  
	//System.out.println(java.sql.Connection.class.getClassLoader());
	//System.out.println(javax.sql.PooledConnection.class.getClassLoader());

	//boostrap classloaders are null (written in JDK C). This is an
	//artifact of the JDK implementation, the spec doesn't mandate it.	

	if (c.getClassLoader() == null) {
		return true;
		}
	
	return false;
	}


/*
For regular files:
	protection domain path: 
		/private/tmp/
	resource URL path to loaded class:
		file:/private/tmp/test.class

For jar files:
	protection domain path: 
		/private/tmp/foo.jar
	resource URL path to loaded class:
		jar:file:/private/tmp/foo.jar!/test.class
			
*/


/**
Returns either the directory or jar file from which the specified class was loaded
<p>
This does not work for system/bootstrap classes and will return <tt>null</tt> for system classes.

@param	caller	the {@link java.lang.Class}  class to be looked up
*/
public static String getLoadedFromPath(final Class caller)
	{
	//param has to be class, cannot be Object, static compile time binding
	
	final ProtectionDomain pd = caller.getProtectionDomain();
	if (pd == null) { //should not be null for system/boostrap classes but just to be sure
		if (dbg) System.out.println("ProtectionDomain for " + caller + " was null");
		return null;
		}
		
	final CodeSource cs = pd.getCodeSource();
	if (cs == null) { //is definitely null for system/boostrap classes
		if (dbg) System.out.println("CodeSource for " + caller + " was null");
		return null;
		}
	
	final String path = cs.getLocation().getPath();
	return path;
	}

/**
Returns the right most <b>parent</b> directory from which the actual class was loaded. This is the leaf directory where
the actual class is located.
<p>
For classes loaded from other sources (jar, etc), the return value will be null.
<p>
This also does not work for system/bootstrap classes (they are typically loaded from jars) either (those will
return null also)
<p>
@param	caller	the {@link java.lang.Class}  class to be looked up
*/
public static String getParentDir(final Class caller)
	{
	final String classname = StringUtil.fileName(caller.getName().replace('.', '/')) + ".class";
	final URL location = caller.getResource(classname);
	
	if (location == null) return null;	
	if (dbg) System.out.println("protocol: " + location.getProtocol());
	
	String protocol = location.getProtocol();
	if (protocol == null) return null;
	
	if (protocol.equalsIgnoreCase("file")) {
		return StringUtil.dirName(location.getFile());	
		}
			
	return null;
	}


/**
Returns the  <b>parent</b> directory from which the actual class was loaded. This is the leaf directory where
the actual class is located.
<p>
For classes loaded from other sources (jar, etc), a URL will be returned. For jar URL's, the 
URL.getProtocol() method returns "jar", so that can be useful for further processing as needed.
<p>
@param	caller	the {@link java.lang.Class} class to be looked up
*/
public static URL getParentDirAsURL(final Class caller)
	{
	final String classname = StringUtil.fileName(caller.getName().replace('.', '/')) + ".class";
	final URL location = caller.getResource(classname);
	final String parent = StringUtil.dirName(location.toString());
	if (dbg) System.out.println(parent);
	URL url = null;
	try {
		url = new URL(parent);
		}
	catch (Exception e) {
		System.err.println(fc.io.IOUtil.throwableToString(e));
		}
	return url;
	}
	
public static void main (String args[])
	{
	System.out.println(ClassUtil.getClassLoaderInfo(ClassUtil.class));
	
	Class obj = new Object().getClass();

	System.out.println(ClassUtil.getClassLoaderInfo(obj));
	System.out.println("getClassName(): " + getClassName());
	System.out.println("getMainClassName(): " + getMainClassName());
	
	System.out.println();

	System.out.println("-------- these should return null --(system classes) ------"); 

	test(obj);
	
	System.out.println("-------- these should return a value (either jar or directory) ------"); 
	test (ClassUtil.class);
	
	System.out.println("-------- these should return a value (either jar or directory) ------"); 
	test(new org.json.JSONObject().getClass());
	}

private static void test (Class obj)
	{
	System.out.println(obj);
	System.out.println("isSystemClass: " + isSystemClass(obj));
	System.out.println("getLoadedFromPath: " + getLoadedFromPath(obj));
	System.out.println("getParentDir: " + getParentDir(obj));
	System.out.println("getParentDirAsURL: " + getParentDirAsURL(obj));
	System.out.println();
	}
}



