// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc.dbo;

import java.sql.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
SQL and java type related functions 
*/
public class Types
{
static final String AVAIL = "available";
static final String INTMAX = "intmax";

String		PStmt_SetXXX_Length_Param_Type;
boolean 	mysqlBooleanHack;
boolean		booleanObjectForNullableBooleans;
boolean		postgresGetObjectToString;
SystemLog	log;

/**
Constructs a new object. 

@param	log		logging destination
@param	props	PropertyManager that may contain:
				<blockquote>
				<ul>
				<li>dbspecific.mysql.boolean_hack</li>
				<li>generate.preparedstatement.setxxx.length_param</li>
				</ul>
				If the above key values are missing, then the
				defaults are used (see usage info for {@link Generate})
				</blockquote>
*/
public Types(SystemLog log, PropertyMgr props)
	{
	this.log = log;
	String key = "dbspecific.mysql.boolean_hack";
	mysqlBooleanHack = Boolean.valueOf(props.get(key, "true")).booleanValue();

	key = "generate.booleanObjectForNullableBooleans";
	booleanObjectForNullableBooleans = Boolean.valueOf(props.get(key, "true")).booleanValue();

	key = "dbspecific.postgres.getObjectToString";
	postgresGetObjectToString = Boolean.valueOf(props.get(key, "true")).booleanValue();

	key = "generate.preparedstatement.setxxx.length_param";
	
	String temp = props.get(key, AVAIL);
	if (temp == null) {
		log.warn("Did not understand value for key:", key, " defaulting to using available()");
		PStmt_SetXXX_Length_Param_Type = AVAIL;
		}
	else {
		temp = temp.toLowerCase().intern();
		if (temp == INTMAX)
			PStmt_SetXXX_Length_Param_Type = INTMAX;
		else if (temp == AVAIL)
			PStmt_SetXXX_Length_Param_Type = AVAIL;
		else { //defaults to avail
			log.warn("Did not understand value for key:", key, "defaulting to using available()");
			PStmt_SetXXX_Length_Param_Type = AVAIL;
			}
		}
	}
	
/**
Returns a suitable getResultXXX method name to retrieve
the data for some column. Used when generating manager code.

@param	java_sql_type 	the {@link java.sql.Type} corresponding
						to the column for which the method will
						be generated
@param	pos				A string containing the column index 
						number (<b>starts from 1)</b>
*/
public String getResultSetMethod(int java_sql_type, String pos, ColumnData cd)
throws SQLException
	{
	return createMethod("get", java_sql_type, pos, null, null, cd);
	}


/**
Returns a suitable getResultXXX method name to retrieve
the data for some column (which will use a runtime prefix argument
prepended to the column name).

@param	java_sql_type 	the {@link java.sql.Type} corresponding
						to the column for which the method will
						be generated
@param	name			column name
*/
public String getRuntimeResultSetMethod(int java_sql_type, String name, ColumnData cd)
throws SQLException
	{
	return createMethod("getruntime", java_sql_type, name, null, null, cd);
	}

/**
Returns a suitable setXXX method name to set the prepared statement
placeholder data for some column. Used when generating manager code. For
example, for an integer column at position 3 and a variable name "x" (which
would contain a integer value at runtime), the returned method will be of
the form: <tt>"setInt(3, x)"</tt>

@param	java_sql_type 	the {@link java.sql.Type} corresponding
						to the column for which the method will
						be generated
@param	pos				a String containing the column index number
						(<b>starts from 1</b>) or a runtime value
						to insert in generate code like "pos++"
@param	varname			the name of the variable containing the
						value to be set at runtime.
*/
public String getPreparedStmtSetMethod(
 int java_sql_type, String psvar, String pos, String varname, ColumnData cd) throws SQLException
	{
	return createMethod("set", java_sql_type, pos, varname, psvar, cd);
	}

public String getPreparedStmtSetNullMethod(
 int java_sql_type, String pos, String varname, ColumnData cd) throws SQLException
	{
	return createMethod("setNull", java_sql_type, pos, varname, null, cd);
	}
	
	
private String createMethod(
	String prefix, int java_sql_type, String pos, String varname, String psvar, ColumnData cd)
throws SQLException
	{
	//PreparedStatement methods for setAsciiStream etc.,
	//needs an additional length parameter, in which case this will be set to true
	boolean needs_length = false;
	
	String result = null;
	
	String typestr = null;
  	switch(java_sql_type)
   		{
   		 //integral types
    	case java.sql.Types.TINYINT:  
    		typestr = (mysqlBooleanHack) 
    					? "Boolean(": "Byte("; 
    		break;
		
		case java.sql.Types.SMALLINT: typestr = "Short("; break;
		case java.sql.Types.INTEGER:  typestr = "Int(";   break;
		case java.sql.Types.BIGINT:   typestr = "Long(";  break;
           	
		//floating
    	case java.sql.Types.FLOAT:  typestr  = "Float(";  break;
		case java.sql.Types.REAL:   typestr  = "Float(";  break;
   		case java.sql.Types.DOUBLE: typestr = "Double("; break;
		
		//arbitrary precision
		case java.sql.Types.DECIMAL: typestr = "BigDecimal("; break; 
    	case java.sql.Types.NUMERIC: typestr = "BigDecimal("; break;
     	
		//char
		case java.sql.Types.CHAR: 	 typestr = "String("; break;
    	case java.sql.Types.VARCHAR: typestr = "String("; break;
		case java.sql.Types.LONGVARCHAR: 
				typestr = "AsciiStream("; 
				needs_length = true;
				break;
    	
		//date-time
		case java.sql.Types.DATE: typestr = "Date("; break;
    	case java.sql.Types.TIME: typestr = "Time("; break;
    	case java.sql.Types.TIMESTAMP: typestr = "Timestamp("; break;
    	
		//stream and binary
 		case java.sql.Types.BLOB: typestr = "Blob("; break;
    	case java.sql.Types.CLOB: typestr = "Clob("; break;
   		case java.sql.Types.BINARY: typestr = "Bytes("; break;
    	case java.sql.Types.VARBINARY: typestr = "Bytes("; break;
    	case java.sql.Types.LONGVARBINARY: 
    		typestr = "BinaryStream("; 
    		needs_length = true;
    		break;
    	
		//misc
		case java.sql.Types.ARRAY: typestr = "Array("; break;

		case java.sql.Types.BIT: 
		case java.sql.Types.BOOLEAN: 
			typestr = "Boolean("; 
			break;
			
		case java.sql.Types.DATALINK: unsupported(java.sql.Types.DATALINK); break;
    	case java.sql.Types.DISTINCT: unsupported(java.sql.Types.DISTINCT); break;
    	case java.sql.Types.JAVA_OBJECT: unsupported(java.sql.Types.JAVA_OBJECT); break;
    	case java.sql.Types.NULL: unsupported(java.sql.Types.NULL); break;
    	
    	case java.sql.Types.OTHER: 
		if (cd.getTable().generate.isPostgres() && postgresGetObjectToString) {
    			typestr = "String("; 
    			}
    		else{
    			typestr = "Object("; 
    			}
    		break;
    		    	
    	case java.sql.Types.REF: typestr = "Ref("; break;
    	case java.sql.Types.STRUCT: typestr = "Struct("; break;
    
		default: unsupported(java_sql_type); 
		}

	if (prefix.equals("setNull"))
		{
		result = "setNull(" 
				  + pos + "," 
				  + java_sql_type + ")"
  				  + "/*" + getSQLTypeName(java_sql_type) + "*/";
		}
		
	else if (prefix.equals("set"))
		{
		result = "";
		
		if (needs_length && PStmt_SetXXX_Length_Param_Type.equals(AVAIL)) {
			result += "try { "; 
			}
			
		if (cd.useBooleanObject()) { 
			//set needs "Object", type string = Boolean or boolean
			//in JDBC, cannot set null like so: setBoolean(1, null)
			result += psvar + "setObject(" + pos + ", " + varname;
			}
		/* special casing OTHER, since we have to provide Types.OTHER as a 3rd param in setObject*/
		else if (java_sql_type == java.sql.Types.OTHER)  {
			//we do this even when converting OTHER->String for postgres, else driver complains since 
			//it's NOT a string in the DB.
			result += psvar + "setObject(" + pos + ", " + varname + ", java.sql.Types.OTHER";
			}
		else{
			result += psvar + "set" + typestr + pos + ", " + varname;
			}
	
		if (needs_length) 
			{
			result += ", ";
			if (PStmt_SetXXX_Length_Param_Type.equals(AVAIL))
				result += varname + ".available()";
			else if (PStmt_SetXXX_Length_Param_Type.equals(INTMAX))
				result += "Integer.MAX_VALUE";
			}
		result += "); ";

		if (needs_length && PStmt_SetXXX_Length_Param_Type.equals(AVAIL)) {
			result += "} catch (IOException e) { throw new SQLException(e.toString()); } "; 
			}
		}
		
	else if (prefix.equals("get"))
		{
		if (cd.useBooleanObject()) { //"(Boolean) rs." so we put "rs." here
			result = "((Boolean) rs.getObject(" + pos + "))";
			}
		else{
			result = "get" + typestr + pos + ")";		
			}
		}
		
	else if (prefix.equals("getruntime"))
		{
		if (cd.useBooleanObject()) {  
			result = "((Boolean) rs.getObject(prefix" + "+\"_" + pos + "\"))";				
			}
		else{
			result = "get" + typestr + "prefix" + "+\"_" + pos + "\")";
			}
		}
		
	else{
		throw new SQLException("I dont understand how to handle: " + prefix);
		}
		
  	return result;
	}

/**
Converts a value corresponding to {@link java.sql.Type} to the Java
type used to represent that type. The {@link java.sql.Type Sql-Type} is
returned by the jdbc driver for some column in a table and we map it to a
corresponding java type that will be used to represent/work with that sql
type in our java programs. This mapping follows the JDBC guidelines.

Similar JDBC API method like {@link
ResultSetMetaData#getColumnTypeName(column)} and {@link
DatabaseMetaData#getColumns()} return tpye names that can be driver/db
specific (and don't have to correspond to Java types anyway).

@param	java_sql_type 	 the {@link java.sql.Type} to convert to a
						 java language type
@param	cd		 		 used to find out if the column is nullable
						 (this can have the effect of using primitive or
						 object types, in some cases, for example, Boolean
						 vs boolean)
**/
public String getJavaTypeFromSQLType(int java_sql_type, ColumnData cd) throws SQLException
	{
	boolean columnIsNullable = cd.isNullable();
	String result = null;
	switch (java_sql_type) 
		{
		//integral types
    	case java.sql.Types.TINYINT:  
    		result = (mysqlBooleanHack) ? "Boolean" : "byte";  
    		break;
		
		case java.sql.Types.SMALLINT: result = "short"; break;
		case java.sql.Types.INTEGER:  result = "int";   break;
		case java.sql.Types.BIGINT:   result = "long";  break;
           	
		//floating
    	case java.sql.Types.FLOAT: result  = "float";  break;
		case java.sql.Types.REAL: result   = "float";  break;
   		case java.sql.Types.DOUBLE: result = "double"; break;
		
		//arbitrary precision
		case java.sql.Types.DECIMAL: result = "BigDecimal"; break; 
    	case java.sql.Types.NUMERIC: result = "BigDecimal"; break;
     	
		//char
		case java.sql.Types.CHAR: result 		= "String"; break;
    	case java.sql.Types.VARCHAR: result 	= "String"; break;
		case java.sql.Types.LONGVARCHAR: result = "InputStream"; break;
    	
		//date-time
		case java.sql.Types.DATE: result = "java.sql.Date"; break;
    	case java.sql.Types.TIME: result = "Time"; break;
    	case java.sql.Types.TIMESTAMP: result = "Timestamp"; break;
    	
		//stream and binary
 		case java.sql.Types.BLOB: result = "java.sql.Blob"; break;
    	case java.sql.Types.CLOB: result = "java.sql.Clob"; break;
   		case java.sql.Types.BINARY: result = "byte[]"; break;
    	case java.sql.Types.VARBINARY: result = "byte[]"; break;
    	case java.sql.Types.LONGVARBINARY: result = "InputStream"; break;
    	
		//misc
		case java.sql.Types.ARRAY: result = "java.sql.Array"; break;

		//note: postgres booleans are/seen as BIT by the driver.
		case java.sql.Types.BIT: 
		case java.sql.Types.BOOLEAN: 
			if (columnIsNullable && booleanObjectForNullableBooleans) {
				result = "Boolean";
				}
			else{
				result = "boolean";
				}
    		break;

		case java.sql.Types.DATALINK: unsupported(java.sql.Types.DATALINK); break;
    	case java.sql.Types.DISTINCT: unsupported(java.sql.Types.DISTINCT); break;
    	case java.sql.Types.JAVA_OBJECT: unsupported(java.sql.Types.JAVA_OBJECT); break;
    	case java.sql.Types.NULL: unsupported(java.sql.Types.NULL); break;
    	
		//json, jsonb, enum, etc are all returned as OTHER by pg
		//use setObject(int, object, Types.OTHER) when saving. And set the java type to be Object as well 
		//(doesn't have to be declared as String - even if the postgres convert to string option is true)
		//keeping it as object gives some potential flexibility down the road for as yet unknown use cases
    	case java.sql.Types.OTHER: 
    		if (cd.getTable().generate.isPostgres() && postgresGetObjectToString) {
    			result = "String"; //however, when savings, we still have to save as setObject(..., Type.OTHER)
    								//else the driver craps out (since NOT defined as String in DB). This
    								//conversion is maybe not worth the code complexity
    			}
    		else{
    			result = "Object"; 
    			}
    		break;
    		
    	case java.sql.Types.REF: result = "java.sql.Ref"; break;
    	case java.sql.Types.STRUCT: result = "java.sql.Struct"; break;
    
		default: unsupported(java_sql_type); 
		}

	return result;
 	}

/**
Uses the same conversion criteria as the {@link getJavaTypeFromSQLType()}
and then returns <tt>true</tt> if the Java type used to represent the
specified {@link java.sql.Type} is primitive (int, boolean etc) as opposed
to an Object type.

@param	java_sql_type 	 the {@link java.sql.Type} for the corresponding
						 java type.
@param	cd		 		 used to find out if the column is nullable
						 (this can have the effect of using primitive or
						 object types, in some cases, for example, Boolean
						 vs boolean)
*/						
public boolean usesPrimitiveJavaType(int java_sql_type, ColumnData cd)
throws SQLException
	{
	boolean primitive = false;
	switch (java_sql_type) 
		{
		//integral types
    	case java.sql.Types.TINYINT:  /*short*/
   		case java.sql.Types.SMALLINT: /*short*/ 
		case java.sql.Types.INTEGER:  /*int*/  
		case java.sql.Types.BIGINT:  /*long*/
				primitive = true;
				break;
       	
		//floating
    	case java.sql.Types.FLOAT: /*float*/  
   		case java.sql.Types.REAL:  /*float*/  
   		case java.sql.Types.DOUBLE: /*double*/ 
   				primitive = true;
				break;

		//arbitrary precision
		case java.sql.Types.DECIMAL: /*BigDecimal*/ break; 
    	case java.sql.Types.NUMERIC: /*BigDecimal*/ break; 
   
		//char
		case java.sql.Types.CHAR: /*String*/ break;
		case java.sql.Types.VARCHAR: /*String*/ break;
		case java.sql.Types.LONGVARCHAR: /*InputStream*/ break;
    	
		//date-time
		case java.sql.Types.DATE: /*java.sql.Date*/ break;
    	case java.sql.Types.TIME: /*Time*/ break;
    	case java.sql.Types.TIMESTAMP:/*Timestamp*/ break;
    	
		//stream and binary
		case java.sql.Types.BLOB:/*java.sql.Blob*/ break;
    	case java.sql.Types.CLOB:/*java.sql.Clob*/ break;
   		case java.sql.Types.BINARY: /*byte[]*/ break;
    	case java.sql.Types.VARBINARY:/*byte[]*/ break;
    	case java.sql.Types.LONGVARBINARY:/*InputStream*/ break;
    	
		//misc
		case java.sql.Types.ARRAY: /*java.sql.Array*/ break;
		
		case java.sql.Types.BIT:     /*boolean or Boolean*/ 
		case java.sql.Types.BOOLEAN: /*boolean or Boolean*/ 
			if (cd.useBooleanObject()) {
				//primitive is already false
				}
			else{
				primitive = true;
				}
			break;
	
		case java.sql.Types.DATALINK: unsupported(java.sql.Types.DATALINK); break;
    	case java.sql.Types.DISTINCT: unsupported(java.sql.Types.DISTINCT); break;
    	case java.sql.Types.JAVA_OBJECT: unsupported(java.sql.Types.JAVA_OBJECT); break;
    	case java.sql.Types.NULL: unsupported(java.sql.Types.NULL); break;
    	
    	case java.sql.Types.OTHER: 
    			primitive = false; 
    			break;
    	
    	case java.sql.Types.REF:  /*java.sql.Ref*/ break;
    	case java.sql.Types.STRUCT: /*java.sql.Struct*/ break;
    
		default: unsupported(java_sql_type); 
		}

	return primitive;
	}


/**
Uses the same conversion criteria as the {@link getJavaTypeFromSQLType()}
and then returns <tt>true</tt> if the Java type used to represent the
specified {@link java.sql.Type} is integral. This is used for creating
the inc/dec methods (only for short, int and long)

@param	java_sql_type 	 the {@link java.sql.Type} for the corresponding
						 java type
*/						
public boolean usesSimpleIntegralJavaType(int java_sql_type)
throws SQLException
	{
	boolean simple = false;
	
  	switch(java_sql_type)
   		{
		case java.sql.Types.INTEGER:  /*int*/  
		case java.sql.Types.BIGINT:  /*long*/
				simple = true;
				break;
   		}  
	
	return simple;
	}


/**
Converts the {@link java.sql.Type} for some column (returned by the
driver) to a readable value. More convenient than using the "constant
field values" section of the not neatly arranged javadocs for {@link
java.sql.Type}.
<p>
Note, this method is different from {@link } because unlike {@link },
this simply returns the variable name corresponding to the parameter
value (for example, <tt>java.sql.Type.INTEGER == 4</tt> and passing
<tt>4</tt> to this method will return "<tt>INTEGER</tt>").

@param java_sql_type	a type from {@link java.sql.Type}
*/
public String getSQLTypeName(int java_sql_type)
	{
	switch (java_sql_type)
		{
		case 2003: 	return "ARRAY"; 
		case -5: 	return "BIGINT"; 
		case -2: 	return "BINARY"; 
		case -7: 	return "BIT"; 
		case 2004: 	return "BLOB"; 
		case 16: 	return "BOOLEAN"; 
		case 1: 	return "CHAR"; 
		case 2005: 	return "CLOB"; 
		case 70: 	return "DATALINK"; 
		case 91:	return "DATE"; 
		case 3: 	return "DECIMAL"; 
		case 2001: 	return "DISTINCT"; 
		case 8: 	return "DOUBLE"; 
		case 6: 	return "FLOAT"; 
		case 4: 	return "INTEGER"; 
		case 2000: 	return "JAVA_OBJECT"; 
		case -4: 	return "LONGVARBINARY"; 
		case -1: 	return "LONGVARCHAR"; 
		case 0: 	return "NULL"; 
		case 2: 	return "NUMERIC"; 
		case 1111: 	return "OTHER"; 
		case 7: 	return "REAL"; 
		case 2006: 	return "REF"; 
		case 5: 	return "SMALLINT"; 
		case 2002: 	return "STRUCT"; 
		case 92: 	return "TIME"; 
		case 93: 	return "TIMESTAMP"; 
		case -6: 	return "TINYINT"; 
		case -3: 	return "VARBINARY"; 
		case 12: 	return "VARCHAR"; 
		default: 	return "NOT KNOWN/ERROR";
		}
	}

void unsupported(int type) throws SQLException {
	throw new SQLException("This framework does not understand/support columns of this type. [java.sql.Type: " + type + "]");	
	}

} //~Types

