// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc.dbo;

import java.io.*;
import java.sql.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;
import fc.web.forms.*;

/**
Generates java objects that represent tables in a database. Each object represents 1
table in the database.
<p>
This framework is great for inserting and updating data in various tables and even
fetching one or more rows form individual tables.
<p>
For each table <font color="blue"><tt>foo</tt></font> in our database, the following
classes are generated:
<ol>
<li><font color="blue">class <tt>Foo</tt></font> which contains all columns of table
<tt>foo</tt> as fields and represents a row in that table. Accessor (get/set) methods are
provided to modify the values of fields in this class (Note, all generated fields are
themselves declared private and we always go through accessor methods so we can keep
track of various modifications etc.)
<li><font color="blue">class <tt>FooMgr</tt></font> which contains "manager" type
functions to read, save, create, etc., instances of class Foo from/to the database. The
methods of <tt>FooMgr</tt> could equivalently have been implemented as static methods in
<tt>Foo</tt> but they have been separated out in a separate manager class to reduce
clutter.
<p>
This framework is <i>not</i> intended to transparently allow arbitrary joins and data
from multiple tables. A <i>better</i> way is to use prepared statements directly to run
ad-hoc SQL queries including those containing arbitrary joins.
<p>
However, to somewhat facilitate arbitrary select queries/joins across multiple tables,
each generated "Mgr" class has a <font color=blue><code>columns()</code></font> method
that returns a list of columns for the corresponding table. For example, in say a Molly
Server Page where information from two tables (table1, table2) is displayed from both
tables on the same page.
<blockquote><pre style="background: #ccccc;">
String my_query =  "select "
  + <font color=blue>table1Mgr.columns()</font> + ", " + <font color=blue>table2.columns()</font>
  + " from table1, table2 WHERE table1.x = table2.x";

PreparedStatement ps = connection.prepareStatement(my_query);
ResultSet rs = connection.executeQuery();
while (rs.next())
    {
    table1 t1 = table1Mgr.getFromRS(rs);  //create a new table1 from the rs
    table2 t2 = table2Mgr.getFromRS(rs);  //ditto for table2
    //..use t1 and t2...
    //....
    }
</pre></blockquote>
</ol>
<hr>
<h2>Configuration</h2>
This program uses a user specified configuration file that allows for 
many code generation options. This file takes the following 
<a href='doc-files/Generate_usage.txt'> configuration options</a>
<p>
Here is a minimal <a href='doc-files/sample.conf'>sample configuration</a> file.
<hr>
<h3>Notes</h3> <b>Note 1</b>: This framework always retrieves and saves data directly to
and fro from the database and never caches data internally. This is a design feature and
keeps this framework orthogonal to caching issues/implementations. The results returned
by the framework can always be cached as needed via say, the  {@link fc.util.cache.Cache}
utility class(es).
<p>
<b>Note 2</b>: <b>MySQL</b> 3.x, 4.x or 5.x does not have true boolean types and silently
converts bool types to TINYINT. This wreaks havoc with auto-generated code which creates
methods with the wrong signature (TINYINT as opposed to bool).
<p>
There are 2 approaches to solving this mysql-specific problem: 
<blockquote>
	a) Require all boolean columns to begin with some keyword (say bool_) and if a column
	begins with this word, then treat it as a boolean, regardless of the type returned by
	the database meta data.<br><br>
	b) Treat all TINYINT's as boolean types. This is the approach I have chosen since
	TINYINT's are NOT portable across databases (for example PostgresQL does not have
	TINYINT's). Therefore we should not use TINYINT's in physical database models; if
	booleans are turned into TINYINT's by MySQL then so be it..since that will not clash
	with any of our modelled types.
</blockquote>
If the flag <tt>mysqlBooleanHack</tt> is set to <tt>false</tt> in the configuration file,
then TINYINT's are <b>not</b> transformed to booleans. There should be no practical need
to do this however.
<p>
<b>Note 3</b>: <b>MySQL</b> allows its tables and columns to start with a numeral. For
example: <i>52_weeks</i>, <i>3_col</i>, etc. This is wrong, not-standard and
not-supported. From the spec:
</p>
<blockquote>
SQL identifiers and key words must begin with a letter (a-z, but also letters with
diacritical marks and non-Latin letters) or an underscore (_). Subsequent characters in
an identifier or key word can be letters, underscores, digits (0-9), or dollar signs ($).
</blockquote>
<p>
So compiling these wrongly named tables (which only MySQL and no other database allows)
results in a bunch of java compiler errors, which may confuse neophytes into believing
that the generator is outputting buggy code. No, the generated code is proper and exactly
the way its intended to be. Java variables/classes <b>cannot</b> start with a number.
Hence, compiler errors. So if you <i>must</i> use MySQL, at least don't name your tables
with a number.
</p>

@author hursh jain
**/
public final class Generate
{
static final String nl 					 = IOUtil.LINE_SEP;
static final String mgrSuffix 			 = "Mgr";
static final String DBO_BaseClassName 	 = 	"fc.jdbc.dbo.DBO";
static final String DBO_MgrBaseClassName =  "fc.jdbc.dbo.DBOMgr";

java.util.Date 		rundate;
DatabaseMetaData	md;
PropertyMgr 		props;
SystemLog			log;
Connection 			con;
String				url;
DBName				dbname;
DBspecific			dbspecific;
File 				outputDir;
String				catalogName;
String				packageName; 
String				classVis;
String				fieldVis;
boolean				accessors;
String[]			tableTypesToProcess;
List				tableNamesToInclude;
Map					tableNamesToIncludeAction;
final int			INCLUDE_ANY_PREFIX 	= 1, 
					INCLUDE_ANY_SUFFIX 	= 2, 
					INCLUDE_CONTAINS 	= 3,
					INCLUDE_EXACT 		= 4;
List				tableNamesToExclude;
Map					tableNamesToExcludeAction;
PrintWriter			out;
NameWrangle			wrangler;
int					processedTablesCount;

//TODO: These need to come from a i18n resource file
//validate errors/messages
String validateNull_ErrorMsg = "Error: Required field, please enter a value";
String validateIntegerOnly_ErrorMsg = "Error: Please enter only numbers in this field";
String validateText_ErrorMsg_MaxSize = "Not enough or too many characters";

//list of tables in our database
List 				tableList;
//used as a temp string buffer by various methods, can
//be reset by setLength(0);
StringBuffer		strbuf = new StringBuffer(2048);

//Changed/set for the CURRENT TABLE being processed
String				beanClassName;         //FOO
String				mgrClassName;          //FOO_Mgr
Table				table;                 //foo
List				pklist;                //[a, b]
List				fklist;                //[b]
String				colsCommaDelimString;  //"a, b, c, d, e"
String				pkCommaDelimString;    //"a, b"
String				fkCommaDelimString;    //"b, c"
String				pkFormalParamString;   //"int a, byte b....."

//not enabled -- don't think this is good
//enabled again
boolean				modifiableAutoIncrementColumns;

/* 
INTERNAL NOTES:
- if a new mgr method is added, also update 
mgrWriteMethodFields and mgrWriteMethodStats
*/

public Generate(String[] args) throws Exception
	{
	log = Log.getDefault();
	log.printTimestamp(true);
	log.printRelativeTimestamp(true);

	//turn on to debug connection manager etc.
	//log.setLevel(SystemLog.DEBUG);	

	Args myargs = new Args(args);
	myargs.setDefaultUsage(this);
	String conf 	= myargs.getRequired("conf");

	props = new FilePropertyMgr(new File(conf));

	//default logging level for the rest of our app
	String loglevel = props.get("log.level");	

	if (loglevel != null) {
		log.setLevel(loglevel);
		}

	ConnectionMgr cmgr = new SimpleConnectionMgr(props);	

	String driver_name = cmgr.getDriver().getClass().getName();
	dbname = DBName.fromDriver(driver_name);
	
	if (dbname == null) {
		log.bug("dbname=", dbname);
	 	log.error("Could not understand the name of the target database. See documentation for more info.");
	 	System.exit(1);
	 	}
	
	dbspecific = (DBspecific) Class.forName("fc.jdbc.dbo.DB" + dbname.toString().toLowerCase()).newInstance();
	
	url = cmgr.getURL();
	con = cmgr.getConnection();	
	md = con.getMetaData();	

	rundate = new java.util.Date();

	String catalogName = props.get("jdbc.catalog");
	if (catalogName == null) {
		catalogName = "";
		}

	//Generate options	
	String output_dir = props.getRequired("generate.output.dir");	
	outputDir = new File(output_dir);

	if (! outputDir.isDirectory() || ! outputDir.canWrite())
		{
		log.error("Specified output location '" + output_dir + "' is not a directory and/or is not writable");
		System.exit(1);
		}

	if (! outputDir.exists()) 
		{
		System.out.print("Output directory: " + output_dir + " does not exist. Creating..");
		boolean mkdirs = outputDir.mkdirs();
		if (! mkdirs) {
			System.out.println("Error creating output directory: " + output_dir);
			System.exit(1);
			}
			
		System.out.println("..done");
		}

	log.info("Output Directory: ", outputDir);

	modifiableAutoIncrementColumns = 
		Boolean.valueOf(
			props.get("generate.modifiableAutoIncrementColumns", "false")
			).booleanValue();
	
	packageName = props.get("generate.class_package");

	accessors = Boolean.valueOf(
					props.get("generate.accessors", "true")).
				booleanValue();

	classVis = props.get("generate.class_vis", "public");
	fieldVis = (accessors) ?
					props.get("generate.field_vis", "private") :
					/* public by default if no accessors */
					props.get("generate.field_vis", "public");


	String tabletypes = props.get("target.types_to_process");
	if (tabletypes != null) 
		{
		tableTypesToProcess = tabletypes.split(",\\s*");
		for (int n=0; n < tableTypesToProcess.length; n++) {
			tableTypesToProcess[n] = tableTypesToProcess[n].toUpperCase();
			}
		}

	String table_include = props.get("target.tables_to_process");
	if (table_include != null) 
		{
		tableNamesToInclude = new ArrayList();
		tableNamesToIncludeAction = new HashMap();
		createActions(table_include, tableNamesToInclude, tableNamesToIncludeAction);
		log.bug("Table names to include:", tableNamesToInclude);
		log.bug("Table names to include action:", tableNamesToIncludeAction);
		}
		
	String table_exclude = props.get("target.tables_to_ignore");
	if (table_exclude != null) 
		{
		tableNamesToExclude = new ArrayList();
		tableNamesToExcludeAction = new HashMap();
		createActions(table_exclude, tableNamesToExclude, tableNamesToExcludeAction);
		log.bug("Table names to exclude:", tableNamesToExclude);
		log.bug("Table names to exclude action:", tableNamesToExcludeAction);
		}


	wrangler = new NameWrangle(props, log);
	
	Watch w = new Watch();
	w.start();
	readTables();
	generateCode();
	cmgr.close();
	
	long elapsed = w.timeInSeconds();
	
	System.out.println("Generator processed "
		+ processedTablesCount 
		+ " table"
		+ ((processedTablesCount > 1) ? "s" : "")
		+ " in "
		+ elapsed
		+ (elapsed > 1 ? " seconds." :  " second."));
	}

void createActions(String parameter, List names, Map actionMap) throws Exception
	{
	String[] temp = parameter.split(",\\s*");
	for (int n = 0; n < temp.length; n++) 
		{			
		//we match table names from database to those specified
		//in the config file, both are lowercased before matching
		//trim() in case table_to_process="foo   "     
		temp[n] = temp[n].trim().toLowerCase();

		boolean startsWithStar = false, endsWithStar = false;
		if (temp[n].startsWith("*")) 
			{
			if (temp[n].length() == 1) {
				throw new Exception("Bad option in config file.\nIn line: " + parameter + "\nA star must be a prefix/suffix to a tablename, not standalone.");
				}
			temp[n] = temp[n].substring(1, temp[n].length());
			startsWithStar = true;
			}
		if (temp[n].endsWith("*")) 
			{
			if (temp[n].length() == 1) {
				throw new Exception("Bad option in config file.\nIn line: " + parameter + "\nA star must be a prefix/suffix to a tablename, not standalone.");
				}
			temp[n] = temp[n].substring(0, temp[n].length()-1);
			endsWithStar = true;
			}
		
		if (startsWithStar) 
			{
			if (endsWithStar) { //both start/end star
				actionMap.put(temp[n], INCLUDE_CONTAINS);
				}
			else{ //start_star
				actionMap.put(temp[n], INCLUDE_ANY_PREFIX); 
				}
			}
		else {
			if (endsWithStar) { //end_star
				actionMap.put(temp[n], INCLUDE_ANY_SUFFIX);
				}
			else{ //exact_match
				actionMap.put(temp[n], INCLUDE_EXACT); 
				}
			}
			
		names.add(temp[n]);
		}
	}

boolean matchTable(String tablename, List names, Map actions) throws IOException
	{	
	boolean match = false;
	
	for (int p = 0; p < names.size(); p++) 
		{
		String  target = (String) names.get(p);
		int 	action = (Integer) actions.get(target);

		//target=foo*, lower_tablename = foo_xxx, foo etc
		if (action == INCLUDE_ANY_SUFFIX) 
			{  
			if (tablename.startsWith(target)) {
				match = true;
				break;
				}
			}
		//target=*foo, lower_tablename = xxx_foo, foo etc
		else if (action == INCLUDE_ANY_PREFIX) 
			{ //*<tablee>
			if (tablename.endsWith(target)) {
				match = true;
				break;
				}
			}
		else if (action == INCLUDE_CONTAINS) 
			{
			if (tablename.indexOf(target) != -1) {
				match = true;
				break;
				}					
			}
		else if (action == INCLUDE_EXACT) 
			{
			if (tablename.equals(target)) {
				match = true;
				break;
				}					
			}
		else{
			throw new IOException("Internal error. Unrecognized action = " + action);
			}
		} //for
	
	return match;
	}
	
void readTables() throws IOException, SQLException
	{
	tableList = new ArrayList();
	
	Table.init(this, md, log, props, dbspecific);

	//not useful generally at least with mysql, postgresql
	String schemaPattern = null;
	//we get all tables, maybe if this is needed at all
	//we can add this as a config file option
	String tableNamePattern = "%";
	
	ResultSet rs = md.getTables(catalogName, schemaPattern, 
						tableNamePattern, tableTypesToProcess);
	
	while (rs.next()) 
		{
		String tablename = rs.getString("TABLE_NAME");
		//tablename cannot be null if the driver is unbroken
		//but can it be an empty string ?
		if (tablename.intern() == "") {
			throw new SQLException("The returned tablename was an empty string, looks like the JDBC driver is broken");
			}
			
		String lower_tablename = tablename.toLowerCase();

		boolean include = false;
		
		if (tableNamesToInclude != null) 
			{
			//including only certain tables, if it is NOT in
			//include list we continue to the next table
			include = matchTable(lower_tablename, tableNamesToInclude, tableNamesToIncludeAction);
			
			if (! include) {
				log.bug("Ignoring table: ", tablename);
				continue;		
				}
			}
		
		boolean exclude = false;
		
		if (tableNamesToExclude != null) 
			{
			exclude = matchTable(lower_tablename, tableNamesToExclude, tableNamesToExcludeAction);	
			
			if (exclude) {
				log.bug("Ignoring table (via exclude): ", tablename);
				continue;		
				}
			}

		if (MiscUtil.isJavaKeyword(tablename)) {
			throw new SQLException("The returned tablename [" + tablename + "] is a Java Reserved Keyword. Either change the name in the DB or exclude this table in the config file");	
			}
			
		log.info(">>>> Processing table: ", tablename);		
		processedTablesCount++;
		
		String tabletype 	= rs.getString("TABLE_TYPE");
		String remarks  	= rs.getString("REMARKS");

		Table table = new Table(con, catalogName, schemaPattern, tablename, tabletype, remarks); 
		tableList.add(table);
		} //~while rs.next
	} //~process


void generateCode() throws IOException, SQLException
	{
	for (int n = 0; n < tableList.size(); n++)
		{
		table = (Table) tableList.get(n);
		pklist = (List) table.getPKList();
		fklist = (List) table.getFKList();
		
		//list of pk's for this table as formal parameters
		//in a generated method, for example:
		//      int col_a, String col_b, Date col_c
		int size = pklist.size();
		StringBuffer buf = new StringBuffer(128);
		for (int m = 0; m < size; m++)
			{
			ColumnData cd = (ColumnData) pklist.get(m);
			buf.append(cd.getJavaTypeFromSQLType());
			buf.append(" ");
			buf.append(cd.getName());
			if ( m < (size-1))
				buf.append(", ");
			}
		pkFormalParamString = buf.toString();
		
		colsCommaDelimString = Table.getListAsString(table.getColumnList());
		pkCommaDelimString = Table.getListAsString(pklist);
		fkCommaDelimString = Table.getListAsString(fklist);
		writeBeanCode();
		writeMgrCode();
		}
	}

void writeBeanCode() throws IOException, SQLException
	{
	beanClassName = wrangler.getClassName(table.getName());
	String filename =  beanClassName + ".java";
	File f = new File(outputDir, filename);

	/* this ensures that the new file will not have the same case as the
	old file, jdk1.5 on osx 10.4 and possibly others keep the existing case 
	of the file if it already exists.
	*/	
	if (f.exists())
		f.delete();

	out = new PrintWriter(new BufferedWriter(new FileWriter(f)));

	writePackage();
	writePrologue();
	writeImports();
	beanWriteClass();
	out.close();
	}	

void writeMgrCode() throws IOException, SQLException
	{
	mgrClassName = 
		wrangler.getClassName(table.getName()) + mgrSuffix;
		
	String filename = mgrClassName + ".java";
	File f = new File(outputDir, filename);
	
	/* this ensures that the new file will not have the same case as the
	old file, jdk1.5 on osx 10.4 and possibly others keep the existing case 
	of the file if it already exists.
	*/	
	if (f.exists())
		f.delete();
		
	out = new PrintWriter(new BufferedWriter(new FileWriter(f)));
	
	writePackage();
	writePrologue();
	writeImports();
	mgrWriteClass();
	out.close();
	}	
	

void writePrologue() 
	{
	String name = getClass().getName();
	ol("/*");
	ol(" * Auto generated on: " + rundate);
	ol(" * JDBC url: [" + url + "]");
	ol(" * WARNING: Manual edits will be lost if/when this file is regenerated.");
	ol(" */");
	}

void writePackage() 
	{
	if (packageName != null) 
		{
		ol("package " + packageName + ";");
		ol();
		}
	}
		
void writeImports() 
	{
	ol("import java.io.*;");
	ol("import java.math.*;");
	ol("import java.sql.*;");
	ol("import java.util.*;");
	ol();
	ol("import fc.io.*;");
	ol("import fc.jdbc.*;");
	ol("import fc.jdbc.dbo.*;");
	ol("import fc.util.*;");
	ol("import fc.web.forms.*;");
	ol();
	}
	

/** 
returns the comment used for get/set methods and for fields in the bean class 
*/
final HashMap commentMap = new HashMap();
String getBeanComment(ColumnData item) throws SQLException
	{
	String comment = (String) commentMap.get(item);
	if (comment != null) 
		return comment;
	
	comment = 
		"/** " + item.getSQLTypeDriverSpecificName() +
				" (" + item.getSQLTypeName() + ")";	
	if (item.isPK()) {
		comment += "; PK=yes";
		}
	if (item.isFK()) {
		ColumnData.FKData fkdata = item.getFK();
		comment += "; FK=yes, refers to: " 
			+ fkdata.getPKTableName() + "." 
			+ fkdata.getPKColName();
		}	
	comment += "; Nullable=" + item.isNullable();	
	comment += "; AutoInc=" + item.isAutoIncrement();
	comment += "; MaxSize=" + item.getSize();
	if (item.hasRemarks()) {
		comment += "; Remarks: " + item.getRemarks();
		}
	comment += " */";
	
	commentMap.put(item, comment);
	return comment;
	}


void beanWriteClass() throws SQLException
	{
	ol("/**");
	o ("Represents a row in the  ");
	o ( table.getName());
	ol(" table. ");
	
	String remarks = table.getRemarks(); 
	if ( remarks != null)
		{
		ol("<p><b>Table Remarks: </b>");		
		ol(remarks);
		}

	ol("*/");
	o(classVis + " class ");
	o(beanClassName);	//classname	
	o(" extends ");
	ol(DBO_BaseClassName);
	ol("{");
		
	beanWriteConstructor();
	beanWriteDBFields();
	beanWriteDBFieldsTracking();
	beanWriteMiscMethods();
	beanWriteGetSet();	
	beanWriteIncDec();
	ol("}");
	}

void beanWriteConstructor() {
	ol("/* Default constructor */");
	o(classVis);
	o(" ");
	o(beanClassName);
	ol("()");
	ol("	{");
	ol("	this.__isNew 	= true;");
	ol("	}");
	ol();
	}

void beanWriteDBFields() throws SQLException
	{
	List cols = table.getColumnList();

	ol("/*--------------------- Columns ------------------------*/");
	//write database fields
	TablePrinter.PrintConfig config = new TablePrinter.PrintConfig();
	config.setPrintBorders(false);
	config.setCellSpacing(1);
	config.setCellPadding(0);
	config.setAutoFit(true);
	TablePrinter p = new TablePrinter(4, out, config);
	p.startTable();
	for (int n = 0; n < cols.size(); n++)
		{
		ColumnData cd = (ColumnData) cols.get(n);
		log.bug("Processing column: ", cd);
		p.startRow();
		p.printCell(fieldVis);
		p.printCell(cd.getJavaTypeFromSQLType());
		p.printCell(cd.getName() + ";");
		p.printCell(getBeanComment(cd));	
		p.endRow();
		}
	p.endTable();		
	ol("/*------------------------------------------------------*/");
	}
	
void beanWriteDBFieldsTracking() throws SQLException
	{
	if (pklist.size() > 0) 
		{
		ol("/*");
		ol("Original PK saved here for updates. If a row is retrieved from the database and");
		ol("the PK value is changed, and then if the object is saved, we need the orignal PK");
		ol("value to find the row in the db for our update to work."); 
		ol("*/");
		for (int n = 0; n < pklist.size(); n++) 
			{
			ColumnData cd = (ColumnData) pklist.get(n);
			o (cd.getJavaTypeFromSQLType());
			o (" __orig_");
			o (cd.getName());
			ol(";");
			}
		}

	ol();
	ol("		boolean __force_update	= false;");
	ol("private	Map		__extra_data;			");
	ol("private boolean __isNew			= false;");
	ol("private boolean __isModified	= false;");

	List cols = table.getColumnList();
	for (int n = 0; n < cols.size(); n++)
		{
		//modified column ?
		ol();
		ColumnData cd = (ColumnData) cols.get(n);
		String colname = cd.getName();

		o("private   boolean __isModified_");
		o(colname);
		ol(" = false;");
		if (cd.isNullable() && cd.usesPrimitiveJavaType()) {
			ol("/* this primitive type is nullable, this tracks whether we should save it as null to the database*/");
			o("private   boolean __isModified_");
			o(colname);
			ol("_setnull = false;");
			}
		o("protected boolean __isNullInDB_");
		o(colname);
		ol(" = false;");
		o("/**returns <tt>true</tt> if ");
		o(colname);
		o(" has changed since it was created/loaded, <tt>false</tt> otherwise");
		ol("*/");
		o("public boolean ");
		o(wrangler.getIsModifiedName(cd));
		o("() { return this.__isModified_");
		o(colname);
		ol("; }");

		if (cd.isNullable() && cd.usesPrimitiveJavaType()) {
			o("/**returns <true> is primitive type ");
			o(colname);
			ol(" has been set to null via the setNull mechanism*/");
			o("public boolean ");
			o(wrangler.getIsModifiedNullName(cd));
			o("() { return this.__isModified_");
			o(colname);
			ol("_setnull; }");
			}

		//column null in database ?
		o("/**returns <tt>true</tt> if ");
		o(colname);
		o(" was null in the database");
		ol("*/");
		o("public boolean ");
		o(wrangler.getIsNullInDBName(cd));
		o("() { return this.__isNullInDB_");
		o(colname);
		ol("; }");
		}
	}

void beanWriteMiscMethods() throws SQLException
	{
	ol();
	//isnew
	ol();
	ol("/** returns <tt>true</tt> if this object is newly created and has <b>not</b> been loaded from the database, <tt>false</tt> otherwise */");
	ol("public boolean isNew() ");
	ol("	{");
	ol("	return this.__isNew;");
	ol("	}");
		
	ol();
	ol("/** Specify <tt>true</tt> to set this object's status as newly created (and not read from a database) */");
	ol("protected void setNew(boolean val) ");
	ol("	{");
	ol("	this.__isNew = val;");
	ol("	}");
		
	//modified
	ol();
	ol("/** returns <tt>true</tt> if this object's data (for any field) has changed since it was created/loaded, <tt>false</tt> otherwise */");
	ol("public boolean isModified() ");
	ol("	{");
	ol("	return this.__isModified;");
	ol("	}");
	
	ol();
	ol("/** Resets the modified status of this object to not-modified");
	ol("this is useful when loading an object via a prepared statement");
	ol("(by using various setXXX methods when we do so, we inadvertently");
	ol("set the modified status of each field to true)");
	ol("*/");	
	ol("void resetModified() ");
	ol("	{");
	List collist = table.getColumnList();
	ol("	this.__isModified = false;");
	for (int n = 0; n < collist.size(); n++)
		{
		ColumnData cd = (ColumnData) collist.get(n);
		String colname = cd.getName();
		o("	this.__isModified_");
		o(colname);
		ol(" = false;");
		
		if (cd.isNullable() && cd.usesPrimitiveJavaType()) {
			o("	this.__isModified_");
			o(colname);
			ol("_setnull = false;");		
			}
		}	
	ol("	}");
			
	ol();
	ol("/**");
	ol("Allows putting arbitrary object-specific data into this object.");
	ol("This is useful to store additional query-specific columns when performing");
	ol("arbitrary queries. For example: ");
	ol("<blockquote><pre>");
	o ("String query = \"select <font color=blue>1+1 as my_sum, \n\t\tnow() as my_time</font>, ");
	o (table.getName());
	o ("Mgr.columns() \n\t\tfrom ");
	o (table.getName());
	ol("\";");
	ol("PreparedStatement ps = con.prepareStatment(query);");
	ol("ResultSet rs = ps.executeQuery();");
	ol("List list = new ArrayList();");
	ol("while (rs.next()) {");
	o ("	<font color=green>"); o (table.getName()); o ("</font> obj = "); 
	o(table.getName()); 
	ol("Mgr.getFromRS(rs);"); 
	ol("	obj.<font color=blue>putExtraData(\"my_sum\"</font>, rs.getInt(\"my_sum\"));");
	ol("	obj.<font color=blue>putExtraData(\"my_time\"</font>, rs.getDate(\"my_time\"));");
	ol("	}");
	o ("//use the list later on...each <font color=green>");
	o (table.getName());
	ol(" </font>object in the list will ");
	ol("//have the extra data..");
	ol("</pre></blockquote>");
	ol("*/");
	ol("public void putExtraData(Object key, Object value) ");
	ol("	{");
	ol("	synchronized (this) {");
	ol("		if (__extra_data == null) {");
	ol("			__extra_data = new HashMap();");
	ol("			}");
	ol("		}");
	ol("	__extra_data.put(key, value);");
	ol("	}");

	ol("/*Alias for the {@link putExtraData} method */");
	ol("public void put(Object key, Object value) ");
	ol("	{");
	ol("	putExtraData(key, value);");
	ol("	}");

	ol();
	ol("/**");
	ol("Allows retrieving arbitrary object-specific data from this object.");
	ol("This data should have been put via the {@link #putExtraData putExtraData} ");
	ol("method prior to invoking this method");
	ol("*/");
	ol("public Object getExtraData(Object key) ");
	ol("	{");
	ol("	synchronized (this) {");
	ol("		if (__extra_data == null) {");
	ol("			return null;");
	ol("			}");
	ol("		}");
	ol("	return __extra_data.get(key);");
	ol("	}");

	ol("/*Alias for the {@link getExtraData} method */");
	ol("public Object get(Object key) ");
	ol("	{");
	ol("	return getExtraData(key);");
	ol("	}");

			
	//toString		
	ol();
	ol("public String toString() ");
	ol("	{");
	ol("	final String nl = fc.io.IOUtil.LINE_SEP;");
	ol("	StringBuffer buf = new StringBuffer(256);");
	o ("	buf.append(\"Class Name: [");
	o (beanClassName);
	ol("]\");");
	ol("	buf.append(\" [isDiscarded=\").append(this.isDiscarded()).append(\"]\");");
	ol("	buf.append(\" [isNew=\").append(this.isNew()).append(\"]\");");
	ol("	buf.append(\" [isModified=\").append(this.isModified()).append(\"]\");");
	ol("	buf.append(nl);");
	ol("	buf.append(\"Note: IsNullInDB only meaningful for existing rows (i.e., isNew=false)\");");
	ol("	buf.append(nl);");
	ol();
	ol("	ByteArrayOutputStream out = new ByteArrayOutputStream(768);");
	ol("	TablePrinter.PrintConfig config = new TablePrinter.PrintConfig();");
	ol("	config.setPrintBorders(false);");
	ol("	config.setCellSpacing(1);");
	ol("	config.setCellPadding(0);");
	ol("	config.setAutoFit(true);");
	ol("	TablePrinter p = new TablePrinter(7, new PrintStream(out), config);");
	ol("	p.startTable();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"PK\");");
	ol("	p.printCell(\"FK\");");
	ol("	p.printCell(\"Field\");");
	ol("	p.printCell(\"Value\");");
	ol("	p.printCell(\"isModified\");");
	ol("	p.printCell(\"isNullinDB\");");
	ol("	p.printCell(\"isSerial/AutoInc\");");	
	ol("	p.endRow();");
	
	for (int n = 0; n < collist.size(); n++)
		{
		ColumnData cd = (ColumnData) collist.get(n);
		String isPK		= cd.isPK() ? "x" : "-";
		String isFK 	= "-";
		if (cd.isFK()) {
			ColumnData.FKData fk = cd.getFK();
			isFK = "x [" +  fk.getPKTableName()  + "." + 
							fk.getPKColName() + "]"; 
							
			}
		String colname 	= cd.getName();
		String value	= "String.valueOf(this." + colname + ")";
		String modified = "(this.__isModified_" + colname + ")";
		String isnull	= "(this.__isNullInDB_" + colname + ")";
		String isAI  	= cd.isAutoIncrement() ? "x" : "-";

		ol();
		ol("	p.startRow();");
		o ("	p.printCell(\""); o(isPK);  ol("\");");
		o ("	p.printCell(\""); o(isFK);  ol("\");");
		o ("	p.printCell(\""); o(colname);  ol("\");");
		o ("	p.printCell(");   o(value);    ol(");");
		o ("	p.printCell(");   o(modified); ol(" ?\"x\":\"-\");");
		o ("	p.printCell(");   o(isnull);   ol(" ?\"x\":\"-\");");
		o ("	p.printCell(\""); o(isAI);  ol("\");");
		ol("	p.endRow();");
		}
	ol();
	ol("	p.endTable();");
	ol("	buf.append(out.toString());");
	ol("	return buf.toString();");
	ol("	}");

	ol();	
	ol("/**");
	ol("Returns a map of all fields-&gt;values (as Strings) contained in this");
   	ol("object. This is useful when sending auto converting the object to JSON, etc.");
	ol("*/");  
	ol("public Map allFieldsMap() ");
	ol("	{");
	ol("	final HashMap m = new HashMap();");
	for (int n = 0; n < collist.size(); n++)
		{
		ColumnData cd = (ColumnData) collist.get(n);
		String colname = cd.getName();

		o ("	m.put(\"");
		o (colname);
		o ("\", ");

		if (! cd.usesPrimitiveJavaType()) {
			o ("("); o(colname); o(" == null ? null : ");
			o ("String.valueOf(this."); o(colname); o(")");
			o (")");
			}
		else{  //primitive type
			if (cd.isNullable()) 
				{
				ol();
				o ("		( __isModified_");o(colname);ol(" ? ");
				o ("			( __isModified_");o(colname);o("_setnull ? null : ");
				o ("String.valueOf(this."); o(colname);o(") )");
				ol(" : ");
				o ("			( __isNullInDB_");o(colname);o(" ? null : ");
				o ("String.valueOf(this.");o(colname);o(")))");
				}			
			else{
				o ("String.valueOf(this.");o(colname);o(")");
				}
			}
					
		ol(");");
		}
	ol("	return m;");
	ol("	}");
	ol();
	}

void beanWriteGetSet() throws SQLException
	{
	List cols = table.getColumnList();

	ol("/* ============== Gets and Sets ============== */");
	for (int n = 0; n < cols.size(); n++)
		{
		ColumnData cd = (ColumnData) cols.get(n);
		ol();
		ol(getBeanComment(cd));
		String colname = cd.getName();
		
		//====================== get ==================
		o("public ");
		o(cd.getJavaTypeFromSQLType());
		o(" ");
		o(wrangler.getGetName(cd));
		ol("()  { ");
		o("	return ");
		o(colname);
		ol(";");
		ol("	}");
	
		ol();
		//===================== set ======================
		
		ol(getBeanComment(cd));
		if (cd.isAutoIncrement() &&  (! modifiableAutoIncrementColumns)) {
			//we need to generate set with package access
			//to set the id when loading the object internally
			//from a result set
			o("/* Generating set for ");
			o(wrangler.getSetName(cd));
			ol(" with non public access since it's an auto-increment column */");
			}
		else {
			o("public");
			}	
		o(" ");
		o(beanClassName);
		o(" ");
		o(wrangler.getSetName(cd));
		o("(");
		o(cd.getJavaTypeFromSQLType());
		o(" ");
		o(colname);
		ol(")	{");
		ol("	this." + colname + " = " + colname + ";"); 
		ol("	this.__isModified_" + colname + " = true;" );
		ol("	this.__isModified = true;" );
		
		if (! cd.isNullable() && ! cd.usesPrimitiveJavaType()) {
			o("	if ("); o(colname); ol(" == null) {");
			o("		Log.getDefault().warn(\"");
			o(colname);
			o(" was set to null via ");
			o(beanClassName);
			o(".");
			o(wrangler.getSetName(cd));
			ol("() and it is non-nullable in the database. Queries using this object will not work properly and you will get a runtime error if saving this object\");");
			ol("		}");
			}
		
		ol("	return this;");
		ol("	}");
		
		if (cd.isNullable() && cd.usesPrimitiveJavaType())
			{
			if (cd.isAutoIncrement() &&  (! modifiableAutoIncrementColumns)) 
				{
				//we need to generate set with package access
				//to set the id when loading the object internally
				//from a result set
				o("/* Generating setNull for ");
				o(wrangler.getSetNullName(cd));
				ol(" with non public access since it's an auto-increment column */");
				}
			else{
				o("public");
				}	
			o(" ");
			o(beanClassName);
			o(" ");
			o(wrangler.getSetNullName(cd));
			ol("()");
			ol("	{");
			ol("	this.__isModified_" + colname + " = true;" );
			ol("	this.__isModified_" + colname + "_setnull = true;" );
			ol("	this.__isModified = true;" );
			ol("	return this;");
			ol("	}");
			}
		}
	}


void beanWriteIncDec() throws SQLException
	{
	List cols = table.getColumnList();
	
	ol();
	ol("/* =========== Increments / Decrements =========== */");
	for (int n = 0; n < cols.size(); n++)
		{
		ColumnData cd = (ColumnData) cols.get(n);

		if (! cd.usesSimpleIntegralJavaType()) {
			continue;
			}

		if (cd.isAutoIncrement() &&  (! modifiableAutoIncrementColumns)) {
			continue;
			}

		//no point inc/dec for key columns. If need be, can always be set by hand.
		if (cd.isPK() || cd.isFK()) {
			continue;
			}

		ol();
		//ol(getBeanComment(cd));
		String colname = cd.getName();
		
		//====================== inc ==================
		//inc by 1
		o ("/** ");
		o ("Increments ");
		o (colname);
   		o (" by 1");
		ol(" */"); 
		o("public");
		o(" ");
		o(beanClassName);
		o(" ");
		o(wrangler.getIncName(cd));
		ol("() {");
		ol("	this." + colname + " = this." + colname + " + 1;"); 
		ol("	this.__isModified_" + colname + " = true;" );
		ol("	this.__isModified = true;" );		
		ol("	return this;");
		ol("	}");
		ol();

		//inc by an amount
		o ("/** ");
		o ("Increments ");
		o (colname);
   		o (" by the specified amount");
		ol(" */");  
		o("public");
		o(" ");
		o(beanClassName);
		o(" ");
		o(wrangler.getIncName(cd));
		o("(");
		o(cd.getJavaTypeFromSQLType());
		o(" amount");
		ol(") {");
		ol("	this." + colname + " = this." + colname + " + amount;"); 
		ol("	this.__isModified_" + colname + " = true;" );
		ol("	this.__isModified = true;" );		
		ol("	return this;");
		ol("	}");
		ol();

		//inc by an amount constrained by upper bound
		o ("/** ");
		o ("Increments ");
		o (colname);
   		o (" by the specified amount, upto but not above the upper bound (bound is inclusive)");
		ol(" */");  
		o("public");
		o(" ");
		o(beanClassName);
		o(" ");
		o(wrangler.getIncName(cd));
		o("(");
		o(cd.getJavaTypeFromSQLType());
		o(" amount, int upper_bound");
		ol(") {");
		ol("	this." + colname + " = Math.min(this." + colname + " + amount, upper_bound);"); 
		ol("	this.__isModified_" + colname + " = true;" );
		ol("	this.__isModified = true;" );		
		ol("	return this;");
		ol("	}");
		ol();

		//===================== dec ======================
	
		//dec by 1
		o ("/** ");
		o ("Decrements ");
		o (colname);
   		o (" by 1");
		ol(" */");  
		o("public");
		o(" ");
		o(beanClassName);
		o(" ");
		o(wrangler.getDecName(cd));
		ol("() {");
		ol("	this." + colname + " = this." + colname + " - 1;"); 
		ol("	this.__isModified_" + colname + " = true;" );
		ol("	this.__isModified = true;" );		
		ol("	return this;");
		ol("	}");
		ol();

		//dec by an amount
		o ("/** ");
		o ("Decrements ");
		o (colname);
   		o (" by the specified amount");
		ol(" */");  
		o("public");
		o(" ");
		o(beanClassName);
		o(" ");
		o(wrangler.getDecName(cd));
		o("(");
		o(cd.getJavaTypeFromSQLType());
		o(" amount");
		ol(") {");
		ol("	this." + colname + " = this." + colname + " - amount;"); 
		ol("	this.__isModified_" + colname + " = true;" );
		ol("	this.__isModified = true;" );		
		ol("	return this;");
		ol("	}");
		ol();
		
		//dec by an amount constrained by lower bound
		o ("/** ");
		o ("Decrements ");
		o (colname);
   		o (" by the specified amount, upto but not below the lower bound (bound is inclusive)");
		ol(" */");  
		o("public");
		o(" ");
		o(beanClassName);
		o(" ");
		o(wrangler.getDecName(cd));
		o("(");
		o(cd.getJavaTypeFromSQLType());
		o(" amount, int lower_bound");
		ol(") {");
		ol("	this." + colname + " = Math.max(this." + colname + " - amount, lower_bound);"); 
		ol("	this.__isModified_" + colname + " = true;" );
		ol("	this.__isModified = true;" );		
		ol("	return this;");
		ol("	}");
		}
	}
	
void mgrWriteClass() throws SQLException
	{
	//classname
	ol("/**");
	o("Manages various operations on the ");
	o(table.getName());
	ol(" table. ");
	ol();
	ol("<p>Most methods of this class take a {@link java.sql.Connection Connection} as an argument");
	ol("and use that connection to run various queries. The connection parameter is never closed");
	ol("by methods in this class and that connection can and should be used again. Methods of this");
	ol("class will also throw a <tt>IllegalArgumentException</tt> if the specified connection");
	ol("object is <tt>null</tt>.");
	ol();
	ol("<p>Thread Safety: Operations on this class are by and large thread safe in that multiple");
	ol("threads can call the methods at the same time. However, seperate threads should use");
	ol("seperate connection objects when invoking methods of this class.");
	ol("*/");
	o(classVis + " final class " + mgrClassName);
	o(" extends ");
	ol(DBO_MgrBaseClassName);
	ol("{");
	
	mgrWriteFields();
	mgrWriteConstructor();
	mgrWriteMethods();	
	ol("}");
	}

void mgrWriteConstructor()
	{
	ol();
	ol("/** Constructor is private since class is never instantiated */");
	o("private ");
	o(mgrClassName);
	ol("() {");
	ol("\t}");
	ol();
	}

void mgrWriteFields() throws SQLException
	{
	ol("/* --- Fields used for collecting usage statistics --- ");
	ol("Increments to these don't need to be synchronized since these are");
	ol("ints and not longs and memory visibility is not an issue in the");
	ol("toString() method (in which these are read).");
	ol("*/");
	ol("private static int __getall_called = 0;");
	ol("private static int __getlimited_called = 0;");
	ol("private static int __getbykey_called = 0;");
	ol("private static int __getwhere_called = 0;");
	ol("private static int __getusing_called = 0;");
	ol("private static int __getusing_ps_called = 0;");
	ol("private static int __getfromrs_called = 0;");
	ol("private static int __save_called = 0;");
	ol("private static int __delete_called = 0;");
	ol("private static int __deletebykey_called = 0;");
	ol("private static int __deletewhere_called = 0;");
	ol("private static int __deleteusing_called = 0;");
	ol("private static int __count_called = 0;");
	ol("private static int __countwhere_called = 0;");
	ol("private static int __countusing_called = 0;");
	ol("private static int __exists_called = 0;");
	ol("/* -------------- end statistics fields -------------- */");
	}

void mgrWriteMethods() throws SQLException
	{
	//get
	mgrWriteMethodGetAll();
	mgrWriteMethodGetAllNoClause();
	mgrWriteMethodGetLimited();
	mgrWriteMethodGetByKey();
	mgrWriteMethodGetWhere();
	mgrWriteMethodGetUsing();
	mgrWriteMethodGetUsingNoClause();
	mgrWriteMethodGetUsingPS();
	mgrWriteMethodGetUsingNamedPS();
	mgrWriteMethodGetColumnNames();
	mgrWriteMethodGetColumnNames2();
	mgrWriteMethodGetFromRS();
	mgrWriteMethodGetFromRS2();
	mgrWriteMethodGetFromRS1Table();	
	mgrWriteMethodDecodeFromRS();
	//save
	mgrWriteMethodSave();
	mgrWriteMethodUpdate();
	//delete
	mgrWriteMethodDelete();
	mgrWriteMethodDeleteByKey();
	mgrWriteMethodDeleteUsing();
	mgrWriteMethodDeleteUsingNoClause();
	mgrWriteMethodDeleteWhere();
	//count, exists, misc.
	mgrWriteMethodCount();
	mgrWriteMethodCountWhere();
	mgrWriteMethodCountUsing();
	mgrWriteMethodCountUsingNoClause();
	mgrWriteMethodExists();
	mgrWriteMethodExistsUsing();
	mgrWriteMethodPrepareStatement();
	mgrWriteCheckDiscarded();
	mgrWriteMethodStats();
	mgrWriteMethodToString();
	//validation stuff
	mgrWriteValidators(); 
	}

/*
We return a list for no particular reason, it may have
been better to return a bean_type[] instead, which would
save typing casts. Testing shows that:

bean_type[] arr = (bean_type[]) list.toArray(new bean_type[]);

takes 1-2 ms for a 1000 elements so speed is not an issue 
(although space might be since the list.toArray is newly 
created).
*/
final void mgrWriteMethodGetAll()
	{
	ol();
	o("static private final String getAllStmt = \"SELECT "); 
	o(colsCommaDelimString);
	o(" from ");
	o(table.getName());
	ol("\";");
	ol("/** ");
	ol("Returns all rows in the table. Use with care for large tables since");
	ol("this method can result in VM out of memory errors. <p>This method");
	ol("also takes an optional (can be null) <tt>clause</tt> parameter which");
	ol("is sent as is to the database. For example, a clause can be:");
	ol("<blockquote><pre>");
	ol("order by some_column_name");
	ol("</pre> </blockquote>");		
	o("@return 	a list containing {@link ");
	o(beanClassName);
	o(" } objects <i>or an empty list</i> if there are no rows in the database");
	ol("*/");
	ol("public static List getAll(final Connection con, final String clause) throws SQLException");
	ol("	{");	
	ol("	__getall_called++;");
	ol("	final List list = new ArrayList();");
	//prepared statement has no parameters to set, used for
	//caching advantage only, (thus, no need to clear params)
	ol("	final String getAllStmtClaused = (clause == null) ? ");
	ol("						   getAllStmt : getAllStmt + \" \" + clause;");
	ol("	PreparedStatement ps = prepareStatement(con, getAllStmtClaused);");
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	final ResultSet rs = ps.executeQuery();");
	ol("	while (true) {");
	ol("		" + beanClassName +  " bean = decodeFromRS(rs);");
	ol("		if (bean == null) { break; } ");
	ol("		list.add(bean);");
	ol("		}");
	ol("	rs.close();");
	ol("	return list;");	
	ol("	}");
	} //~write getall

final void mgrWriteMethodGetAllNoClause()
	{
	ol();
	ol("/** ");
	o ("Convenience method that invokes {@link getAll(Connection, ");
	o (beanClassName);
	ol(", String) getAll} with an empty additional clause.");
	ol("*/");
	o ("public static List getAll(final Connection con) ");
	ol("throws ValidateException, SQLException");
	ol("	{");
	ol("	return getAll(con, null);");
	ol("	}");
	} //~write getUsing

final void mgrWriteMethodGetLimited()
	{
	ol();
	o("static private final String getLimitedStmt = \"SELECT "); 
	o(colsCommaDelimString);
	o(" from ");
	o(table.getName());
	ol("\";");
	ol("/** ");
	ol("Returns all rows in the table starting from some row number and limited");
	ol("by a certain number of rows after that starting row. ");
	ol("<p>");
	ol("This method takes a required (non-null) <code>order_clause</code>, since when using");
	ol("a limit clause, rows must be ordered for the limit to make sense. The");
	ol("clause should be of the form <font color=blue>order by ...</font>");
	ol("<p>");
	ol("The <code>limit</code> specifies the number of rows that will be returned. (those many");
	ol("or possibly lesser rows will be returned, if the query itself yields less");
	ol("rows).");
	ol("<p>");
	ol("The <code>offset</code> skips that many rows before returning rows. A zero offset is");
	ol("the same as a traditional query with no offset clause, where rows from");
	ol("the beginning are returned. If say, offset = 10, then rows starting from");
	ol("row 11 will be returned.");
	ol("<p>");
	ol("The sql-query generated by this method is database specific but will (typically) look like:");
	ol("<blockquote><pre>");
	ol("select &lt;column_list&gt; from &lt;table&gt; order by &lt;clause&gt; limit 5 offset 10");
	ol("</pre> </blockquote>");		
	o("@return 	a list containing {@link ");
	o(beanClassName);
	o(" } objects <i>or an empty list</i> if there are no rows in the database");
	ol("*/");
	ol("public static List getLimited(final Connection con, final String order_clause, int limit, int offset) throws SQLException");
	ol("	{");	
	ol("	__getlimited_called++;");
	ol("	final List list = new ArrayList();");
	//prepared statement has no parameters to set, used for
	//caching advantage only, (thus, no need to clear params)
	ol("	final String tmp = getLimitedStmt + \" \" + order_clause + \" LIMIT \" + limit + \" OFFSET \" + offset;");
	ol("	PreparedStatement ps = prepareStatement(con, tmp);");
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	final ResultSet rs = ps.executeQuery();");
	ol("	while (true) {");
	ol("		" + beanClassName +  " bean = decodeFromRS(rs);");
	ol("		if (bean == null) { break; } ");
	ol("		list.add(bean);");
	ol("		}");
	ol("	rs.close();");
	ol("	return list;");	
	ol("	}");
	} //~write getlimited

final void mgrWriteMethodGetByKey() throws SQLException
	{
	ol();
	if (pklist.size() == 0) {
		ol("/* getByKey() not implemented since this table does not have any primary keys defined */");
		return;
		}

	o("static private final String getByPKStmt = \"SELECT "); 
	o(colsCommaDelimString);
	o(" from ");
	o(table.getName());
	o(" WHERE ");
	o(Table.getPreparedStmtPlaceholders(pklist));
	ol("\";");
	ol("/** ");
	ol("Returns <b>the</b> row corresponding to the specified primary key(s) of this table ");
	ol("or <b><tt>null</tt></b> if no row was found.");
	ol("<p>This method uses a prepared statement and is safe from SQL injection attacks");
	ol("*/");
	o("public static ");
	o(beanClassName);
	o(" getByKey(final Connection con, ");
	o(pkFormalParamString);
	ol(") throws SQLException");
	ol("	{");
	ol("	__getbykey_called++;");
	ol("	PreparedStatement ps = prepareStatement(con, getByPKStmt);");
	ol("	StringBuilder errbuf = null;");
	ol();
	for (int n = 0; n < pklist.size(); n++)
		{
		ColumnData cd = (ColumnData) pklist.get(n);	
		if (! cd.usesPrimitiveJavaType())
			{
			o ("	if ("); o(cd.getName()); ol(" == null) {");
			ol("		if (errbuf == null) { errbuf = new StringBuilder(); }");		
			o ("		errbuf.append(\"");
			o (cd.getName());
			ol(" was set to null (but is non-nullable)\").append(IOUtil.LINE_SEP);");
			ol("		}");
			ol();
			}
		else{
			o ("	//");
			o (cd.getName());
			o (" [");
			o (cd.getJavaTypeFromSQLType());
			ol("] is primitive, skipping null test");	
			}
			
		String varname = cd.getName();
		o("\t");
		String pos = String.valueOf((n+1));
		ol(cd.getPreparedStmtSetMethod("ps.", pos, varname) );
		}
	
	ol("	");
	ol("	if (errbuf != null) {");
	ol("		throw new ValidateException(errbuf.toString());"); 
	ol("		}");
	ol("	final ResultSet rs = ps.executeQuery();");
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	" + beanClassName + " bean = decodeFromRS(rs);");
	ol("	rs.close();");
//	ol("	ps.clearParameters();");
	ol("	return bean;");
	ol("	}");
	} //~write getbykey
	
	
final void mgrWriteMethodGetWhere() throws SQLException
	{
	ol();
	ol("/** ");
	ol("Returns the rows returned by querying the table with the specified");
	ol("<tt>WHERE</tt> clause or <i>an empty list</i> if no rows were found.");
	ol("(note: the string <tt>\"WHERE\"</tt> does <b>not</b> have to be");
	ol("specified in the clause. It is added automatically by this method).");
	ol("<p>Queries can use database functions such as: <code>lower()</code>,");
	ol("<code>upper()</code>, <code>LIKE</code> etc. For example:");
	o ("<pre><blockquote>"); o(table.getName());
	ol("Mgr.getWhere(\"lower(col_a) = 'foo'\")");
	ol("//compares the lower case value of col_a with the string 'foo'");
	ol("</blockquote></pre>");	
	ol("<p><b>The \"where\" clause is sent as-is to the database</b>. SQL");
	ol("injection attacks are possible if it is created as-is from a <b><u>untrusted</u></b> source.");
	ol();
	ol("@throws	IllegalArgumentException if the specified <tt>where</tt> parameter is null"); 
	ol("*/");
	ol("public static List getWhere(final Connection con, final String where) throws SQLException");
	ol("	{");
	ol("	__getwhere_called++;");
	ol("	Argcheck.notnull(where, \"the where parameter was null (and should not be null)\");");
	ol("	final String where_stmt = \"SELECT " + colsCommaDelimString + " from " + table.getName() + " WHERE \" + where ;");
	ol("	Statement stmt = QueryUtil.getRewindableStmt(con);");
	ol("	log.bug(\"Query to run: \", stmt, \" \", where_stmt);");
	ol("	final List list = new ArrayList();");
	ol("	final ResultSet rs = stmt.executeQuery(where_stmt);");
	ol("	while (true) {");
	ol("		" + beanClassName +  " bean = decodeFromRS(rs);");
	ol("		if (bean == null) { break; } ");
	ol("		list.add(bean);");
	ol("		}");
	ol("	stmt.close();");
	ol("	return list;");	
	ol("	}");
	} //~write getwhere

final void mgrWriteMethodGetUsing() throws SQLException
	{
	ol();
	ol("/** ");
	ol("Returns the rows returned by querying the table with the value of the");
	o ("specified <tt>");
	o (beanClassName);
	ol("</tt> object or <i>an empty list</i> if no rows were found. As many");
	ol("fields in <tt>alltypes</tt> can be set as needed and the values of");
	ol("all set fields (including fields explicitly set to <tt>null</tt>)");
	ol("are then used to perform the query.");
	ol("<p>");
	ol("This method is often convenient/safer than the {@link #getWhere");
	ol("getWhere} method (because the <tt>getWhere</tt> method takes an");
	ol("arbitrary query string which has to be properly escaped by the");
	ol("user)."); 
	ol("<p>Essentially, this method is a more convenient way to use a");
	ol("PreparedStatement. Internally, a prepared statement is created and");
	ol("it's parameters are set to fields that are set in this object).");
	ol("Using PreparedStatements directly is also perfectly fine. For");
	ol("example, the following are equivalent. ");
	ol("<p> Using a PreparedStatement:");
	ol("<blockquote><pre>");
	ol("String foo = \"select * from table_foo where x = ? and y = ?\";");
	ol("PreparedStatement ps = con.prepareStatement(foo);");
	ol("ps.setString(1, \"somevalue\");");
	ol("ps.setString(2, \"othervalue\");");
	ol("ResultSet rs  = ps.executeUpdate();");
	ol("while (rs.next()) {");
	ol("    table_foo bean = table_fooMgr.getFromRS(rs);");
	ol("    }");
	ol("</pre> </blockquote>");
	ol("");
	ol("Using this method:");
	ol("<blockquote><pre>");
	ol("table_foo <font color=blue>proto</font> = new table_foo();");
	ol("proto.set_x(\"somevalue\");	//compile time safety");
	ol("proto.set_y(\"othervalue\");	//compile time safety");
	ol("List beans = table_fooMgr.<font color=blue>getUsing(proto)</font>;");
	ol("</pre> </blockquote>");	
	ol("<p>This method also takes an <tt>clause</tt> parameter which");
	ol("is sent as is to the database. For example, a clause can be:");
	ol("<blockquote><pre>");
	ol("List beans = table_fooMgr.<font color=blue>getUsing(proto, <b>\"order by some_column_name\"</b>)</font>;");
	ol("</pre> </blockquote>");	
	ol("This clause is optional. Specify <tt>null</tt> to not use it at all.");
	ol("If the clause is specified, do <font size='+1'><b>NOT</b></font> include the word <tt>WHERE</tt>.");
	ol();
	ol("The fields that are set in the proto object (as shown above) are sent as");
	ol("part of a WHERE clause constructed internally. If you are specifying a clause");
	ol("as well, you should not specify the word <tt>WHERE</tt>. However, you may have");
	ol("to specify <tt>AND</tt> to add to the internal WHERE clause, if you have set any");
	ol("fields in the proto object. For example");
	ol("<blockquote><pre>");
	ol("List beans = table_fooMgr.<font color=blue>getUsing(proto, <b>\"and bar = 5\"</b>)</font>;");
	ol("</pre> </blockquote>");	
	ol("<p>Note: For a <i>very</i> large number of rows, it may be more");
	ol("efficient to use a prepared statement directly (as opposed to using");
	ol("this method). In most cases, this is not something to worry about,");
	ol("but your mileage may vary...");
	ol("*/");
	o("public static List getUsing(final Connection con, final ");
	o(beanClassName);
	ol(" bean, final String clause) throws ValidateException, SQLException");
	ol("	{");
	ol("	__getusing_called++;");
	ol("	Argcheck.notnull(bean, \"the bean parameter was null (and should not be null)\");");
	ol("	if (! bean.isModified()) { ");
	ol("		throw new ValidateException(\"bean=\" + bean + \" not modified, ignoring query\");");
	ol("		}");
	ol();
	ol("	int __count = 0;");
	ol("	final StringBuilder buf = new StringBuilder(512);");
	ol("	buf.append(\"SELECT " + colsCommaDelimString + " from " +
				table.getName() + " WHERE \");");

	List cols = table.getColumnList();
	//we allow any col to be set, including pk or partial pk	
	List relevant_cols = cols; 

	for (int n = 0; n < cols.size(); n++) 
		{
		ColumnData cd = (ColumnData) cols.get(n);	
	
		o ("	if (bean.");
		o (wrangler.getIsModifiedName(cd));
		ol("()) { ");
		if (! cd.usesPrimitiveJavaType()) {
			o ("		if (bean."); o(wrangler.getGetName(cd)); ol("() == null) {");
			o ("			buf.append(\""); o(cd.getName()); ol(" is NULL and \");");
			ol("			}");
			ol("		else{");
			o ("			buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
			ol("			__count++;");
			ol("			}");
			}
		else{ //primitive type
			if (cd.isNullable()) 
				{
				o ("		if (bean."); o(wrangler.getIsModifiedNullName(cd)); ol("()) {");
				o ("			buf.append(\""); o(cd.getName()); ol(" is NULL and \");");
				ol("			}");
				ol("		else{");
				o ("			buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
				ol("			__count++;");
				ol("			}");
				}
			else{
				o ("		buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
				ol("		__count++;");
				}
			}
		ol("		}");
		}		
	
	ol();
	//get rid of last "and "
	ol("	buf.setLength(buf.length() - 4);"); 
	ol();
	ol("	if (clause != null) {");
	ol("		buf.append(\" \");");
	ol("		buf.append(clause);");
	ol("		}");
	ol();
	ol("	final String getUsingPKStmt = buf.toString();");
	ol("	PreparedStatement ps = prepareStatement(con, getUsingPKStmt);");

	utilFillPStmtFromList_IfModified_Object(relevant_cols, "\t");

	ol("	log.bug(\"Query to run: \", ps);");
	ol("	final List list = new ArrayList();");
	ol("	final ResultSet rs = ps.executeQuery();");
	ol("	while (true) {");
	ol("		" + beanClassName +  " row = decodeFromRS(rs);");
	ol("		if (row == null) { break; } ");
	ol("		list.add(row);");
	ol("		}");
	ol("	rs.close();");
//	ol("	ps.clearParameters();");
	ol("	return list;");	
	ol("	}");
	} //~write getusing

final void mgrWriteMethodGetUsingNoClause() throws SQLException
	{
	ol();
	ol("/** ");
	o ("Convenience method that invokes {@link getUsing(Connection, ");
	o (beanClassName);
	o (", String) getUsing} with an empty <i><tt>clause</tt></i> parameter.");
	ol("*/");
	o("public static List getUsing(final Connection con, final ");
	o(beanClassName);
	ol(" bean) throws ValidateException, SQLException");
	ol("	{");
	ol("	return getUsing(con, bean, null);");
	ol("	}");
	} //~write getUsingNoClause

//the actual method written is called getUsing(..) not getUsingPS(..)
//i.e., getUsing is overloaded
final void mgrWriteMethodGetUsingPS() throws SQLException
	{
	ol();
	ol("/**");
	ol("This is a <i>convenience</i> method that runs the specified ");
	ol("prepared statement to perform an arbitrary query. For example: ");
	ol("<blockquote>");
	ol("<pre>");
	ol("PreparedStatement <font color=blue>ps</font> = con.prepareStatement(");
	ol("		\"select * from some_table where some_column = ?\");");
	ol("ps.setString(1, \"foo\");");
	ol("List list = fooMgr.<font color=blue>getUsing</font>(con, <font color=blue>ps</font>);");
	ol("for (int n = 0; n < list.size(); n++) {");
	ol("	sometable t = (sometable) list.get(n);");
	ol("	//do something");
	ol("	}");
	ol("</pre>");
	ol("</blockquote>");
	ol("The effect of the above is <u>equivalent</u> to the following (larger) block ");
	ol("of code:");
	ol("<blockquote>");
	ol("<pre>");
	ol("PreparedStatement <font color=blue>ps</font> = con.prepareStatement(");
	ol("	\"select * from sometable where some_column = ?\"");
	ol("	);");
	ol("ps.setString(1, \"foo\");");
	ol("ResultSet rs = <font color=blue>ps.executeQuery()</font>;");
	ol("List list = new ArrayList();");
	ol("while (rs.next()) {");
	ol("	list.add(sometableMgr.<font color=blue>getFromRS(rs)</font>);");
	ol("	}");
	ol("");
	ol("for (int n = 0; n < list.size(); n++) {");
	ol("	sometable t = (sometable) list.get(n);");
	ol("	//do something");
	ol("	}");
	ol("</pre>");
	ol("</blockquote>");
	ol("");
	ol("Note: Just as with other get<i>XXX</i> methods, for large amounts of");
	ol("rows (say many thousands), it may be more efficient use and iterate");
	ol("through a JDBC result set directly.");
	ol("*/");
	o("public static List getUsing(final Connection con, ");
	ol(" final PreparedStatement ps) throws ValidateException, SQLException");
	ol("	{");
	ol("	__getusing_ps_called++;");
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	final List list = new ArrayList();");
	ol("	final ResultSet rs = ps.executeQuery();");
	ol("	while (true) {");
	ol("		" + beanClassName +  " row = decodeFromRS(rs);");
	ol("		if (row == null) { break; } ");
	ol("		list.add(row);");
	ol("		}");
	ol("	rs.close();");
//	ol("	ps.clearParameters();");
	ol("	return list;");	
	ol("	}");
	} //~write getusingps


//the actual method written is called getUsing(..) not getUsingPS(..)
//i.e., getUsing is overloaded
final void mgrWriteMethodGetUsingNamedPS() throws SQLException
	{
	ol();
	ol("/**");
	ol("This is a <i>convenience</i> method that runs the specified ");
	ol("{@link fc.jdbc.dbo.NamedParamStatement NamedParamStatement} to perform an arbitrary query.");
	ol("For example: <blockquote>");
	ol("<pre>");
	ol("NamedParamStatement <font color=blue>ps</font> = queryReadeer.getQuery(\"somequery\");");
	ol("ps.setString(\"some_placeholder\", \"foo\");");
	ol("List list = fooMgr.<font color=blue>getUsing</font>(con, <font color=blue>ps</font>);");
	ol("for (int n = 0; n < list.size(); n++) {");
	ol("	sometable t = (sometable) list.get(n);");
	ol("	//do something");
	ol("	}");
	ol("</pre>");
	ol("</blockquote>");
	ol("");
	ol("Note: Just as with other get<i>XXX</i> methods, for large amounts of");
	ol("rows (say many thousands), it may be more efficient use and iterate");
	ol("through a JDBC result set directly.");
	ol("*/");
	o("public static List getUsing(final Connection con, ");
	ol(" final NamedParamStatement ps) throws ValidateException, SQLException");
	ol("	{");
	ol("	__getusing_ps_called++;");
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	final List list = new ArrayList();");
	ol("	final ResultSet rs = ps.executeQuery();");
	ol("	while (true) {");
	ol("		" + beanClassName +  " row = decodeFromRS(rs);");
	ol("		if (row == null) { break; } ");
	ol("		list.add(row);");
	ol("		}");
	ol("	rs.close();");
//	ol("	ps.clearParameters();");
	ol("	return list;");	
	ol("	}");
	} //~write getusingps


final void mgrWriteMethodGetColumnNames() throws SQLException
	{
	ol();
	ol("/** ");
	o("Returns a <i>comma delimited list</i> of <i>all</i> columns in <tt>");
	o(table.getName());
	ol("</tt>. These column names are fully qualified, i.e., they contain ");
	ol("table name as a prefix to the column name. For example:");
	ol("<blockquote><pre>");
	ol("<tt>tablename.column1 AS tablename_column1, tablename.column2 AS tablename_column2 ...</tt>");
	ol("</pre></blockquote>");
	ol("<p>This list is suitable for placing in the column(s) clause of a select query, such as: ");
	ol("<blockquote>");
	ol("<tt>Single table: select <i><font color=blue>[column_list_A]</font></i> from table_A</tt><br>");
	ol("<tt>Join: select <i><font color=blue>[column_list_A], [column_list_B]</font></i> from table_A, table_B</tt>");
	ol("</blockquote>");
	ol("The ResultSet returned by the query can be used directly or can be passed");
	o ("to the {@link #getFromRS getFromRS} method to convert it into a list of <code>");
	ol(table.getName());
	ol("</code> objects. If the query is a join across multiple tables,");
	ol("then the {@link #getFromRS getFromRS} method for each table manager");
	ol("can be called on the same ResultSet to retrieve the row object for");
	ol("that table.");
	ol("Note: the returned list of names has a trailing space, which is good when");
	ol("the rest of the query is appended to this list.");
	ol("*/");
	o("public static String"); 
	ol(" columns() throws SQLException");
	ol("	{");
	o ("	return \"");
	//has a trailing space, which is good, we want to return this 
	//with a trailing space
	o (		table.getFullyQualifiedColumnString()); 
	ol("\";");	
	ol("	}");	
	} //~write columns


final void mgrWriteMethodGetColumnNames2() throws SQLException
	{
	ol();
	ol("/** ");
	o("Returns a <i>comma delimited list</i> of <i>all</i> columns in <tt>");
	o(table.getName());
	ol("</tt>. These column names are prefix with the specified prefix, which corresponds to the");
	ol("table abbreviation used in the \"AS\" clause. For example:");
	ol("<blockquote><pre>");
	ol("<tt>xyz.column1 AS xyz_column1, xyz.column2 AS xyz_column2 ...</tt>");
	ol("</pre></blockquote>");
	ol("<p>This list is suitable for placing in the column(s) clause of a select query, such as: ");
	ol("<blockquote>");
	ol("<p><b>Note:</b> the \".\" will automatically be appended between the prefix and column name");
	ol("so the prefix should not end with a \".\" or \"_\", etc<p>");
	ol("<tt>Single table: select <i><font color=blue>[column_list_A]</font></i> from table_A <b>AS</b> xyz</tt><br>");
	ol("<tt>Join: select <i><font color=blue>[column_list_A], [column_list_B]</font></i> from table_A <b>AS</b> xyz, table_B <b>AS</b> zzz</tt>");
	ol("</blockquote>");
	ol("The ResultSet returned by the query can be used directly or can be passed");
	o ("to the {@link #getFromRS getFromRS(String)} method to convert it into a list of <code>");
	ol(table.getName());
	ol("</code> objects. If the query is a join across multiple tables,");
	ol("then the {@link #getFromRS getFromRS(String)} method for each table manager");
	ol("can be called on the same ResultSet to retrieve the row object for");
	ol("that table.");
	ol("Note: the returned list of names has a trailing space, which is good when");
	ol("the rest of the query is appended to this list.");
	ol("*/");
	o("public static String"); 
	ol(" columns(String prefix) throws SQLException");
	ol("	{");
	//has a trailing space, which is good, we want to return this 
	//with a trailing space
	ol(		table.getPrefixQualifiedColumnString()); 
	ol("	}");	
	} //~write columns



final void mgrWriteMethodGetFromRS() throws SQLException
	{
	ol();
	ol("/** ");
	o("Creates and returns a new <tt>"); o(table.getName());	
	ol("</tt> object that represents a row from the specified ResultSet. The ResultSet is");
	ol("typically obtained via a handwritten query/PreparedStatement. The resulting ");
	ol("ResultSet should contain all of the");
	ol("column names of table, and this will only happen if the handwritten query had");
	ol("a select statement that specified all fields or used a <tt>select <b>*</b>..</tt>");
	ol("clause.");
	ol("<p>");
	ol("In the select clause, we could also be selecting multiple tables. To disambiguate");
	ol("between the same field names that may exist in multiple tables, this method ");
	ol("also requires that the query should use <font color=blue>fully qualified</font>");
	ol("(prefixed with the table name) column names, such as:");
	ol("<blockquote><pre>");
	ol("<font color=blue>tablename</font>_column1");
	ol("<font color=blue>tablename</font>_column2");
	ol("...etc.");
	ol("</pre></blockquote>");
	ol("<p>");
	ol("For example:");
	ol("<blockquote>");
	ol("<code>select <font color=blue>foo</font>.a <b>AS</b> <font color=blue>foo</font>_a, <font color=red>bar</font>.a <b>AS</b> <font color=red>bar</font>_a from <font color=blue>foo</font>, <font color=red>bar</font> where foo.a = bar.a;</code>");
	ol("</blockquote>");
	ol("The {@link #columns} method conveniently returns a list of column names in fully qualified format ");
	ol("and is useful for this purpose.");
	ol("<p>Note: This method will read the <i>current</i> row from the specified result set");
	ol("and will <b>not</b> move the result set pointer to the next row after the current");
	ol("row has been read. The result set should be appropriately positioned [via <tt>rs.next()</tt>]"); 
	ol("<i>before</i> calling this method.");
	ol();
	ol("@return a new {@link ");
	o(beanClassName);
	o("} object populated with the contents of the next");
	ol("		row from the result set or <tt> null </tt> if");
	ol("		the ResultSet was empty.");
	ol("*/");
	o("public static "); 
	o(beanClassName);
	ol(" getFromRS(final ResultSet rs) throws SQLException");
	ol("	{");
	ol("	__getfromrs_called++;");
	//decode from RS
	ol("	Argcheck.notnull(rs, \"the specified resultset parameter was null\");");
	ol("	boolean hasrow = ! rs.isAfterLast();");
	ol("	if (! hasrow) { ");
	ol("		return null; ");
	ol("		} ");
	o ("	");o(beanClassName);o(" bean = new ");o(beanClassName);ol("();"); 
	ol();
	List col_list = table.getColumnList();
	for (int n = 0; n < col_list.size(); n++)
		{
		//THIS METHOD USES COLUMN NAMES NOT COLUMN POSITIONS. THIS IS
		//-THE- MAIN DIFF BETWEEN THIS AND THE INTERNAL DECODE_FROM_RS METHOD
		//  If a column does not exist in the result set and we try to get
		//  that column by name, a sql exception (per the jdbc spec,
		//  although the spec is not totally clear on this) is allowed
		//  to be thrown (for example, the postgres drives does this)... 
		//  so we require all columns to be present while decoding a row
		//  typically a  select * must be issued. if the same column exists with
		//  the same name in more than 2 tables, we have a problem. We therefore
		//  require column names to be qualified with tablenames, such as:
		//    select foo.a as foo_a, bar.a as bar_a  from foo, bar;
		ColumnData cd = (ColumnData) col_list.get(n);	
		String setmethod_name = wrangler.getSetName(cd);
		o("\tbean.");
		o(setmethod_name);
		o("( ");
		
		if (! cd.useBooleanObject()) {
			o("rs.");
			}
		o(cd.getResultSetMethod(table.getName() + "_"));
			
		ol(" );");	
		
		if (cd.isPK()) {
			o("\tbean.__orig_");
			o(cd.getName());
			o(" = ");

			if (! cd.useBooleanObject()){
				o("rs.");
				}
			o(cd.getResultSetMethod(table.getName() + "_"));
	
			ol("; /* save original PK */");
			}
				
		ol("\tif (rs.wasNull()) {");
		o("\t\tbean.__isNullInDB_");
		o(cd.getName());
		ol(" = true;");
		ol("\t\t}");
		}
	ol();
	ol(" 	/* set to true when instantiated new, false when we populate the bean from a resultset */");
	ol("	bean.setNew(false);")	; 
	ol("	/* it's not modified, just loaded from the database */");
	ol("	bean.resetModified();"); 
	ol("	return bean;");
	ol("	}");
	} //~write getfromrs



final void mgrWriteMethodGetFromRS2() throws SQLException
	{
	ol();
	ol("/** ");
	o("Creates and returns a new <tt>"); o(table.getName());	
	ol("</tt> object that represents a row from the specified ResultSet. The ResultSet is");
	ol("typically obtained via a handwritten query/PreparedStatement. The resulting ");
	ol("ResultSet should contain all of the");
	ol("column names of table, prefixed with the specified <i>prefix</i> argument."); 
	ol("a select statement that specified all fields or used a <tt>select <b>*</b>..</tt>");
	ol("clause.");
	ol("<p>");
	ol("In the select clause, we could also be selecting multiple tables. To disambiguate");
	ol("between the same field names that may exist in multiple tables, this method ");
	ol("also requires that the query should use a <font color=blue>prefix</font>");
	ol("(some arbitrary prefix) before column names, such as:");
	ol("<blockquote><pre>");
	ol("<font color=blue>foo</font>_column1");
	ol("<font color=blue>foo</font>_column2");
	ol("...etc.");
	ol("</pre></blockquote>");
	ol("This prefix will typically be the same as the table abbreviation chosen via the <b>AS</b> clause. ");
	ol("If the AS clause is not used, then it is simpler to use the {@link getFromRS(ResultSet)} method instead");
	ol("<p><b>Note:</b> the \".\" will automatically be appended between the prefix and column name");
	ol("so the prefix should not end with a \".\" or \"_\", etc<p>");
	ol("<p>");
	ol("For example:");
	ol("<blockquote>");
	ol("<code>select <font color=blue>XXX</font>.a <b>AS</b> <font color=blue>XXX</font>_a, <font color=red>YYY</font>.a <b>AS</b> <font color=red>YYY</font>_a from <font color=blue>foo as XXX</font>, <font color=red>bar as YYY</font> where foo.a = bar.a;</code>");
	ol("</blockquote>");
	ol("The {@link #columns} method conveniently returns a list of column names in fully qualified format ");
	ol("and is useful for this purpose.");
	ol("<p>Note: This method will read the <i>current</i> row from the specified result set");
	ol("and will <b>not</b> move the result set pointer to the next row after the current");
	ol("row has been read. The result set should be appropriately positioned [via <tt>rs.next()</tt>]"); 
	ol("<i>before</i> calling this method.");
	ol();
	ol("@return a new {@link ");
	o(beanClassName);
	o("} object populated with the contents of the next");
	ol("		row from the result set or <tt> null </tt> if");
	ol("		the ResultSet was empty.");
	ol("*/");
	o("public static "); 
	o(beanClassName);
	ol(" getFromRS(final ResultSet rs, String prefix) throws SQLException");
	ol("	{");
	ol("	__getfromrs_called++;");
	//decode from RS
	ol("	Argcheck.notnull(rs, \"the specified resultset parameter was null\");");
	ol("	boolean hasrow = ! rs.isAfterLast();");
	ol("	if (! hasrow) { ");
	ol("		return null; ");
	ol("		} ");
	o ("	");o(beanClassName);o(" bean = new ");o(beanClassName);ol("();"); 
	ol();
	List col_list = table.getColumnList();
	for (int n = 0; n < col_list.size(); n++)
		{
		//THIS METHOD USES COLUMN NAMES NOT COLUMN POSITIONS. THIS IS
		//-THE- MAIN DIFF BETWEEN THIS AND THE INTERNAL DECODE_FROM_RS METHOD
		//  If a column does not exist in the result set and we try to get
		//  that column by name, a sql exception (per the jdbc spec,
		//  although the spec is not totally clear on this) is allowed
		//  to be thrown (for example, the postgres drives does this)... 
		//  so we require all columns to be present while decoding a row
		//  typically a  select * must be issued. if the same column exists with
		//  the same name in more than 2 tables, we have a problem. We therefore
		//  require column names to be qualified with tablenames, such as:
		//    select foo.a as foo_a, bar.a as bar_a  from foo, bar;
		ColumnData cd = (ColumnData) col_list.get(n);	
		String setmethod_name = wrangler.getSetName(cd);
		o("\tbean.");
		o(setmethod_name);
		o("( ");

		if (! cd.useBooleanObject()) {
			o("rs.");
			}
		o(cd.getRuntimeResultSetMethod());
	
		ol(" );");	
		
		if (cd.isPK()) {
			o("\tbean.__orig_");
			o(cd.getName());
			o(" = ");

			if (! cd.useBooleanObject()) {
				o("rs.");
				}
			o(cd.getRuntimeResultSetMethod());

			ol("; /* save original PK */");
			}
				
		ol("\tif (rs.wasNull()) {");
		o("\t\tbean.__isNullInDB_");
		o(cd.getName());
		ol(" = true;");
		ol("\t\t}");
		}
	ol();
	ol(" 	/* set to true when instantiated new, false when we populate the bean from a resultset */");
	ol("	bean.setNew(false);")	; 
	ol("	/* it's not modified, just loaded from the database */");
	ol("	bean.resetModified();"); 
	ol("	return bean;");
	ol("	}");
	} //~write getfromrs2


final void mgrWriteMethodGetFromRS1Table() throws SQLException
	{
	ol();
	ol("/** ");
	o("Creates and returns a new <tt>"); o(table.getName());
	ol("</tt> object that represents a row from the specified ResultSet. For this method");
	ol("to work properly, the specified ResultSet should contain <b>all</b> (typically via <b>select *");
	o ("</b>) of the column names of table.");
	o ("<tt>");
	o (table.getName());
	ol("</tt>.");
	ol("<p>");
	ol("This method does not prepend the table name to columns when reading data from");
	ol("the result set. It is useful when writing a JDBC query by hand that uses a single table");
	ol("(no joins) and then converting the returned result set into objects of this");
	ol("class. For example:");
	ol("<p>");
	ol("<code>select a, b, c, c*2 from foo where a = 1;</code>");
	ol("<p>");
	ol("This method will expect columns to be called <code><i>a, b, c</i></code> (no column aliases) in the returned");
	ol("result set. In this example, there is only one table <code>foo</code> so qualifying the column");
	ol("names, like <code>foo.a as foo_a</code> is not necessary). Also note, for this method to work properly, the ");
	ol("column list<blockquote><code>select <i>a, b, c </i></code> ...</blockquote> should be complete, i.e., contain <i>at least</i> all the columns");
	ol("of this table (<i>additional</i> expressions like c*2 are fine). It is slightly less efficient to retrieve all columns");
	ol("especially for large tables but to construct a row into an object, we need all the fields. To be safe, use <blockquote><tt>select * ....</tt></blockquote>");
	ol("<p>");
	ol("Of course, if one needs a subset of columns, one can use the ResultSet directly and forego trying to");
	ol("convert a ResultSet row into an corresponding object");
	ol("<p>	");
	ol("See {@link getFromRS(ResultSet)} which is more useful when writing a JDBC");
	ol("query that uses multiple table joins.");
	ol("<p>Note: This method will read the <i>current</i> row from the specified result set");
	ol("and will <b>not</b> move the result set pointer to the next row after the current");
	ol("row has been read. The result set should be appropriately positioned [via <tt>rs.next()</tt>]"); 
	ol("<i>before</i> calling this method.");
	ol();
	o ("@return a new {@link ");
	o (beanClassName);
	ol("} object populated with the contents of the next");
	ol("		row from the result set or <tt> null </tt> if");
	ol("		the ResultSet was empty.");
	ol("*/");
	o ("public static "); 
	o (beanClassName);
	ol(" getFromRS1Table(final ResultSet rs) throws SQLException");
	ol("	{");
	ol("	__getfromrs_called++;");
	//decode from RS
	ol("	Argcheck.notnull(rs, \"the specified resultset parameter was null\");");
	ol("	boolean hasrow = ! rs.isAfterLast();");
	ol("	if (! hasrow) { ");
	ol("		return null; ");
	ol("		} ");
	o ("	");o(beanClassName);o(" bean = new ");o(beanClassName);ol("();"); 
	ol();
	List col_list = table.getColumnList();
	for (int n = 0; n < col_list.size(); n++)
		{
		//THIS METHOD USES COLUMN NAMES NOT COLUMN POSITIONS. THIS IS
		//-THE- MAIN DIFF BETWEEN THIS AND THE INTERNAL DECODE_FROM_RS METHOD
		//  If a column does not exist in the result set and we try to get
		//  that column by name, a sql exception (per the jdbc spec,
		//  although the spec is not totally clear on this) is allowed
		//  to be thrown (for example, the postgres driver does this)... 
		//  so we require all columns to be present while decoding a row
		//  typically a  select * must be issued. if the same column exists with
		//  the same name in more than 2 tables, we have a problem. We therefore
		//  require column names to be qualified with tablenames, such as:
		//    select foo.a as foo_a, bar.a as bar_a  from foo, bar;
		ColumnData cd = (ColumnData) col_list.get(n);	
		String setmethod_name = wrangler.getSetName(cd);
		o("\tbean.");
		o(setmethod_name);
		o("( ");

		if (! cd.useBooleanObject()){
			o("rs.");
			}
		o(cd.getResultSetMethod());
	
		ol(" );");		

		if (cd.isPK()) {
			o("\tbean.__orig_");
			o(cd.getName());
			o(" = ");
			
			if (! cd.useBooleanObject()) {
				o("rs.");
				}
			o(cd.getResultSetMethod());
	
			ol("; /* save original PK */");
			}
				
		ol("\tif (rs.wasNull()) {");
		o("\t\tbean.__isNullInDB_");
		o(cd.getName());
		ol(" = true;");
		ol("\t\t}");
		}
	ol();
	ol(" 	/* set to true when instantiated but this should be false");
	ol("	 whenever we populate the bean from a result set */");
	ol("	bean.setNew(false);")	; 
	ol("	//it's not modified, just loaded from the database");
	ol("	bean.resetModified();"); 
	ol("	return bean;");
	ol("	}");
	} //~write getfromrs
	
final void mgrWriteMethodDecodeFromRS() throws SQLException
	{
	//internal method. decode from RS. moves the result set via rs.next()
	ol();
	o("private static "); 
	o(beanClassName);
	ol(" decodeFromRS(final ResultSet rs) throws SQLException");
	ol("	{");
	ol("	Argcheck.notnull(rs, \"the specified resultset parameter was null\");");
	ol("	boolean hasrow = rs.next();");
	ol("	if (! hasrow) { ");
	ol("		return null; ");
	ol("		} ");
	ol("	" + beanClassName + " bean = new " + beanClassName + "();"); 
	ol();
	List col_list = table.getColumnList();
	for (int n = 0; n < col_list.size(); n++)
		{
		ColumnData cd = (ColumnData) col_list.get(n);	
		String setmethod_name = wrangler.getSetName(cd);
		o("\tbean.");
		o(setmethod_name);
		o("( ");

		if (! cd.useBooleanObject()){
			o("rs.");
			}
		o(cd.getResultSetMethod());

		ol(" );");	
		
		if (cd.isPK()) {
			o("\tbean.__orig_");
			o(cd.getName());
			o(" = ");

			if (! cd.useBooleanObject()){
				o("rs.");
				}
			o(cd.getResultSetMethod());

			ol("; /* save original PK */");
			}

		ol("\tif (rs.wasNull()) {");
		o("\t\tbean.__isNullInDB_");
		o(cd.getName());
		ol(" = true;");
		ol("\t\t}");
		ol();
		}

	ol();
	ol(" 	/* set to true when newly instantiated but this should be false");
	ol("	 whenever we populate the bean from a result set */");
	ol("	bean.setNew(false);")	; 
	ol("	//it's not modified, just loaded from the database");
	ol("	bean.resetModified();"); 
	ol("	return bean;");
	ol("	}");
	} //~write decodefromrs
	
final void mgrWriteMethodSave() throws SQLException	
	{
	ColumnData cd = null;
	List cols = table.getColumnList();
	//we don't insert or update auto-increment columns
	//filter those out and put the rest in filtered_cols
	List filtered_cols = new ArrayList(); 
	List autoinc_cols = new ArrayList();
	
	for (int n = 0; n < cols.size(); n++)
		{
		cd = (ColumnData) cols.get(n);	
		if (cd.isAutoIncrement()) {
			autoinc_cols.add(cd);
			if (! modifiableAutoIncrementColumns) {
				continue;   //don't add it to the filtered list
				}
			}
		filtered_cols.add(cd);
		}

	final int filtered_count = filtered_cols.size();
	ol();
	ol("/**");
	ol("Saves the specified object into the database. If the specified");
	ol("object was newly created, then it is <span style=\"font-variant:");
	ol("small-caps\">insert</span>'ed into the database, else (if it was retrieved");
	ol("earlier from the database) it is <span ");
	ol("style=\"font-variant: small-caps\">update</span>'ed. (this can be");
	ol("overriden by the {@link #update update} method). If the object is");
	ol("inserted as a new row, then after insertion, the values of");
	ol("serial/auto-incremented columns will be automatically available via the");
	ol("appropriate getXXX() methods on that object.");
	ol("<p>");
	ol("<b>NOTE 1:</b> When saving an object, only modified fields are");
	ol("saved. Do not rely on default field values (such as null) of newly");
	ol("created objects; instead explicitly set the value (including to null");
	ol("if needed) of any field that should be saved to the database.");
	ol("<p>");
	ol("<b>NOTE 2:</b> Once an object is successfully saved, it is discarded");
	ol("and cannot be saved again and any attempt to save it again will");
	ol("result in a runtime exception. Objects that need to be modified");
	ol("again must be re-instantiated or re-populated from the database");
	ol("before they can be saved again. (the serial/auto-increment data will still be");
	ol("available, discarding only affects the ability to save the object");
	ol("again).");
	ol("<p>");
	ol("<b>Note 3:</b> <font color='red'>For various reasons/flexiblity, default database values");
	ol("for columns <i>other</i> than serial columns are <b>not</b> available");
	ol("in the saved object. To get these values, retrieve the saved object again. (this is what");
	ol("we would have to do internally anyway). This is relevant, for example, when a column has");
	ol("a default value of a now() timestamp, and we need to get that timestamp after the object");
	ol("has been saved</font>");
	ol();
	ol("@return 	the number of rows inserted or updated (typically useful ");
	ol("			to see if an update succeeded)");
	ol("@throws	ValidateException 	on a validation error");
	ol("@throws SQLException		on some SQL/Database error");
	ol("@throws IOException			by the available() method if/when");
	ol("							setting a stream for longvar/text types");
	ol("*/");
	o("public static int save(final Connection con, final ");
	o(beanClassName);
	//we need IOException for the available() method when
	//setting a stream for longvar/text types in our statement
	ol(" bean) throws ValidateException, SQLException, IOException");
	ol("	{");
	ol("	__save_called++;");
	ol("	Argcheck.notnull(bean, \"the specified bean parameter was null\");");
	ol("	checkDiscarded(bean);");
	ol("	if (! bean.isModified()) { ");
	ol("		log.warn(\"bean=\", bean, \" not modified, IGNORING SAVE\\n====DEBUG STACK TRACE====\\n\", IOUtil.throwableToString(new Exception()));");
	ol("		return 0;");
	ol("		}");
	ol("	PreparedStatement ps = null;");
	ol();
	//.... insert into table_foo (col_x, col_y) values (?, ?).....
	//.... update table_foo set (col_x=? col_y=?) where ....
	ol("	boolean inserting_a_row = false;");
	ol("	if (bean.isNew() && ! bean.__force_update) ");
	//----------------- INSERT ---------------------
	ol("		{  //insert new row");
	ol("		validateBeforeSaveNew(bean);");
	ol("		int __count = 0;");
	ol("		inserting_a_row = true;");
	ol("		final StringBuilder buf = new StringBuilder(512);");
	o ("		buf.append(\"INSERT into "); 
	o(table.getName());
	ol(" (\");");

	for (int n = 0; n < filtered_count; n++) 
		{
		cd = (ColumnData) filtered_cols.get(n);	
		o ("		if (bean.");
		o(wrangler.getIsModifiedName(cd));
		ol("()) { ");
		o("			buf.append(\"");
		o(cd.getName());
		ol("\").append(\", \");");
		ol("			__count++;");
		ol("			}");
		}	
	ol();

	ol("		if (__count == 0) {");
	ol("			throw new ValidateException(\"Cannot save this bean because no column has been modified. Use JDBC directly as needed.\\n\");");
	ol("		}");

	//get rid of last ", " [ we need to do it this way
	//since we don't know when the last ',' will come since
	//any number of columns could have been modified (or not)
	ol("		buf.setLength(buf.length() - 2);"); 
	ol(" 		buf.append(\") values (\");");
	ol("		for (int n = 0; n < __count; n++) { ");
	ol("			buf.append(\"?\");");
	ol("			if ((n+1) < __count)");
	ol("				buf.append(\", \");");
	ol("			}");
	ol("		buf.append(\")\");");
	ol();
	ol("		final String insertByPKStmt = buf.toString();");
	ol("		ps = prepareStatement(con, insertByPKStmt);");
	
	ol("		/* Insert any changed values into our prepared statement */");
	utilFillPStmtFromList_IfModified(filtered_cols, "\t\t");		
	ol("		}");
	
	//-------------------- UPDATE -----------------------
	ol("	else //update existing row ");
	ol("		{");
	if (pklist.size() == 0) {
		//we write pknum to shut up the compiler about unreachable statements
		ol("		int pknum = 0; /*we write pknum to shut up the compiler about unreachable statements*/");
		ol("		if (pknum == 0) {");
		ol("			throw new ValidateException(");
		ol("				\"Cannot update this bean because it has no primary keys.\""); 
		ol("				+ \"Use JDBC directly to update values to this table.\");");
		ol("			}");
		}	
	ol("		validateBeforeSaveUpdate(bean);");
	ol("		int __count = 0;");
	ol("		final StringBuilder buf = new StringBuilder(512);");
	ol("		buf.append(\"UPDATE \");"); 
	o ("		buf.append(\"");
	o(table.getName());
	ol("\");");
	ol("		buf.append(\" SET \");");
	for (int n = 0; n < filtered_count; n++) 
		{
		cd = (ColumnData) filtered_cols.get(n);	
		o("		if (bean.");
		o(wrangler.getIsModifiedName(cd));
		ol("()) { ");
		o("			buf.append(\"");
		o(cd.getName());
		o("=?, ");
		ol("\");");	
		ol("			__count++;");
		ol("			}");
		} 
	ol();	
	
	ol("		if (__count == 0) {");
	ol("			throw new ValidateException(\"Cannot save this bean because no column has been modified. Use JDBC directly as needed.\\n\");");
	ol("		}");
	
	//get rid of last ", " [ we need to do it this way
	//since we don't know when the last ',' will come since
	//any number of columns could have been modified (or not)
	ol("		buf.setLength(buf.length() - 2);"); 
	
	ol("		buf.append(\" WHERE \");");
	o ("		buf.append(\"");
	o(Table.getPreparedStmtPlaceholders(pklist));
	ol("\");");
	
	ol("		ps = con.prepareStatement(buf.toString());");
	//ol("		log.bug(ps);");		//we write a debug below anyway
	ol();
	
	ol("		/* Insert any changed values into our prepared statement */");
	//note we don't have to worry about columns that have
	//not modified (including primitive columns for which
	//isNullInDB is true)
	utilFillPStmtFromList_IfModified(filtered_cols, "\t\t");		
	
	ol();	
	ol("		/* Set primary keys for the WHERE part of our prepared statement */");
	for (int n = 0; n < pklist.size(); n++)
		{
		cd = (ColumnData) pklist.get(n);	
		String getmethod_name = wrangler.getGetName(cd);
		String varname = cd.getName();

		o("\t\t");
		o(cd.getJavaTypeFromSQLType());
		o(" ");
		o(varname);
		o(" = ");

		//If it is a forced update, then the primary key(s) are always 
		//provided as a param (so use that to find the row to update, not
		//the cached pk, since that will be null/empty in a new object anyway)
		//If it's not a forced update, use cached orig pk value (in case pk 
		//itself was changed)
		
		o("(bean.__force_update) ? ");
		o("bean.");
		o(getmethod_name);
		o("()");
		o(" : ");
		o("bean.__orig_");
		o(cd.getName());
		ol(";");
	
		o("\t\t");
		//skip over the if_modified '?' (which are created
		//at runtime by the generated code using the 'pos')
		//integer
		ol(cd.getPreparedStmtSetMethod("ps.", "++pos", varname));
		}
	ol("\t\t} //~else update;");
	ol();
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	int result = ps.executeUpdate();");

	ol("	if (inserting_a_row) { //get auto increment info");
	int autoinc_size = autoinc_cols.size();
	if (autoinc_size > 0)
		{
		ol("		/* Retrieve values from auto-increment columns */");
		ol("		ResultSet rs = null; Statement stmt = null;");
		ol("		String query = null;");
		ol("		boolean found = false;");
		ol();	
		for (int n = 0; n < autoinc_size; n++)
			{

		//if a auto increment column is modifiable AND was modified,
		//then we cannot get a auto increment value for it (since
		//we are specifying our own value in that case, the sequence
		//on the database is not used)
		
			o ("		if (bean.");
			o(wrangler.getIsModifiedName(cd));
			ol("()) { ");
			o ("			//column: ");
			ol(cd.getName());
			ol("			//not getting auto increment value for this column");
			ol("			//since not using auto increment, a value was specified manually");
			ol("			}");		
			ol("		else{");				

			cd = (ColumnData) autoinc_cols.get(n);
			String query = dbspecific.getAutoIncrementQuery(cd);
			String setmethod_name = wrangler.getSetName(cd);
			ol("			stmt = con.createStatement();");
			o ("			query = \""); o(query); ol("\";");
			ol("			rs = stmt.executeQuery(query);");
			ol("			found = rs.next();");
			ol("			if (! found) throw new SQLException(\"No last inserted id returned\");");
			o ("			bean."); o(setmethod_name); o("( "); 

			if (! cd.useBooleanObject()){
				o("rs.");
				}
			o(cd.getResultSetMethod());
	
			ol(");");		

			ol("			if (rs.wasNull()) {");
			o ("				bean.__isNullInDB_");
			o (cd.getName());
			ol(" = true;");
			ol("				}");
			ol("			rs.close();");
			ol("			}");
			} //~for
		ol("		}");
		} //~if auto-inc size > 0
	else {
		ol("		//No auto inc columns in this table");
		ol("		}");
		} 
	ol();
	ol("	//discard after saving/updating for safety");
	ol("	bean.discard();");
	ol("	return result;");
	ol("	}");	
	}

final void mgrWriteMethodUpdate() throws SQLException	
	{
	ol();
	if (pklist.size() == 0) {
		ol("/* update() method not implemented since this bean has no primary keys. Use JDBC directly.");
		return;
		}

	ol("/**");
	ol("Uses the specified object to update existing data in the database.");
	ol("<p>");
	ol("Note, the {@link #save save} method automatically saves newly created objects");
	ol("as <i>inserts</i> in the database (and prior <i>retrieved</i> objects, when");
	ol("subsequently modified, are saved as <i>updates</i>).");
	ol("<p>");
	ol("However, sometimes it is useful to create a <i>new</i> object and then");
	ol("use its data to <i>update</i> an existing row in the database.");
	ol("This method need <b>only</b> be called to save a <u>newly</u>");
	ol("created object as an <u>update</u> into the database (overriding the");
	ol("default action of saving new objects as inserts in the database).");
	ol("<p>");
	ol("Note, also, a bean can only be updated if the corresponding table it has");
	ol("at least one primary key defined. To update tables with no primary keys,");
	ol("use JDBC directly.");
	ol("<p>");
	o("This method takes primary key(s) of {@link ");
	o(beanClassName);
	ol("} as additional arguments and sets those in the");
	ol("specified bean before updating the database (this way the row to update");
	ol("can be uniquely identified).");
	ol();
	ol("@see #save");	
	ol();
	ol("@return 	the number of rows that were updated (typically useful ");
	ol("			to see if an update succeeded)");
	ol("@throws	ValidateException 	on a validation error");
	ol("@throws SQLException		on some SQL/Database error");
	ol("*/");
	o("public static int update(final Connection con, final ");
	o(beanClassName);
	o(" bean, ");
	o(pkFormalParamString);	
	ol(") throws ValidateException, SQLException, IOException");
	ol("	{");

	for (int n = 0; n < pklist.size(); n++)
		{
		ColumnData cd = (ColumnData) pklist.get(n);	
		String varname = cd.getName(); /* used in pkFormalParamString too*/
		o("\tbean.");
		o( wrangler.getSetName(cd));
		o("(");
		o(varname);
		o(");");
		ol();
		}
	ol();
	ol("	if (bean.isNew()) { /* force update (and not insert) for new bean */");
	ol("		bean.__force_update = true;");
	ol("		}");
	ol("	return save(con, bean);");
	ol("	}");
	}
	
final void mgrWriteMethodDelete() throws SQLException
	{
	ol();
	if (pklist.size() == 0) {
		ol("/* delete() not implemented since this table does not have any primary keys defined. */");
		return;
		}
	o("static private final String deleteStmt = \"DELETE "); 
	o(" from ");
	o(table.getName());
	o(" WHERE ");
	o(Table.getPreparedStmtPlaceholders(pklist));
	ol("\";");
	ol("/** ");
	o("Deletes this object from the database. ");
	ol("<p>");
	ol("<b>NOTE 1:</b> Only objects that were retrieved from the database can be deleted. Newly");
	ol("created objects cannot be deleted since they do not yet exist in the database.");
	ol("Use {@link #deleteByKey deleteByKey} or {@link #deleteWhere deleteWhere} instead");
	ol("for arbitrary deletions. <p><b>NOTE 2:</b> Once an object is successfully");
	ol("deleted, it is discarded and cannot be deleted again and any attempt to delete");
	ol("it again will result in a runtime Exception.");
	ol("*/");
	o("public static void delete(final Connection con, ");
	o(beanClassName);
	ol(" bean) throws SQLException");
	ol("	{");
	ol("	__delete_called++;");
	ol("	if (bean.isNew()) {");
	ol("		throw new DBOException(\"Cannot delete new objects using this method. Use deleteByKey() or deleteWhere() instead\");");
	ol("		}");
	ol("	checkDiscarded(bean);");	
	ol("	final PreparedStatement ps = prepareStatement(con, deleteStmt);");
	//list of pk's for this table
	for (int n = 0; n < pklist.size(); n++)
		{
		ColumnData cd = (ColumnData) pklist.get(n);	
		String getmethod_name = wrangler.getGetName(cd);
		o("\t");
		o(cd.getJavaTypeFromSQLType());
		o(" ");
		String varname = cd.getName();
		o(varname);
		o(" = bean.");
		o(getmethod_name);
		ol("();");
		o("\t");
		String pos = String.valueOf((n+1));
		ol(cd.getPreparedStmtSetMethod("ps.", pos, varname));
		}
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	final int result = ps.executeUpdate();");
//	ol("	ps.clearParameters();");
	ol("	if (result != 1) { ");
	ol("		throw new DBOException(\"The number of deleted rows was: \" + result + \"; [Should have been 1 row exactly] \");");
	ol("		}");
	ol("	}");
	}
	
final void mgrWriteMethodDeleteByKey() throws SQLException
	{
	ol();
	if (pklist.size() == 0) {
		ol("/* deleteByKey() not implemented since this table does not have any primary keys defined */");
		return;
		}

	o("static private final String deleteByPKStmt = \"DELETE "); 
	o(" from ");
	o(table.getName());
	o(" WHERE ");
	o(Table.getPreparedStmtPlaceholders(pklist));
	ol("\";");
	ol("/** ");
	o("Deletes the rows with the specified primary key(s) from the database. ");
	ol("<p>This method uses a prepared statement and is safe from SQL injection attacks");
	ol("*/");
	o("public static void deleteByKey(final Connection con, ");
	o(pkFormalParamString);
	ol(") throws SQLException");
	ol("	{");
	ol("	__deletebykey_called++;");
	ol("	PreparedStatement ps = prepareStatement(con, deleteByPKStmt);");
	
	for (int n = 0; n < pklist.size(); n++)
		{
		ColumnData cd = (ColumnData) pklist.get(n);	
		String varname = cd.getName();
		o("\t");
		String pos = String.valueOf((n+1));
		ol( cd.getPreparedStmtSetMethod("ps.", pos, varname) );
		}
	
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	final int result = ps.executeUpdate();");
//	ol("	ps.clearParameters();");
	ol("	if (result != 1) { ");
	ol("		throw new DBOException(\"The number of deleted rows was: \" + result + \"; [Should have been 1 row exactly] \");");
	ol("		}");
	ol("	}");
	} //~write delete by key
	
final void mgrWriteMethodDeleteWhere()
	{
	ol();
	ol("/** ");
	ol("Deletes the rows with the specified where clause. <p><b>The");
	ol("where clause is sent as-is to the database and SQL injection");
	ol("attacks are possible if it is created as-is from a untrusted");
	ol("source.</b>");
	ol("(note: the string <tt>\"WHERE\"</tt> does <b>not</b> have to be");
	ol("specified in the clause. It is added automatically by this method).");
	ol();
	ol("@return the number of rows deleted by the database");
	ol("*/");	
	o("public static int deleteWhere(final Connection con, final String where) throws SQLException");
	ol("	{");
	ol("	__deletewhere_called++;");
	ol("	Argcheck.notnull(where, \"the where parameter was null (and should not be null)\");");
	ol("	final String stmt_string = \"DELETE from " + table.getName() + " WHERE \" + where ;");
	ol("	Statement stmt = con.createStatement();");
	ol("	log.bug(\"Query to run: \", stmt_string);");
	ol("	final int result = stmt.executeUpdate(stmt_string);");
	ol("	return result;");
	ol("}");
	}

final void mgrWriteMethodDeleteUsing() throws SQLException
	{	
	ol();
	ol("/** ");
	ol("Returns the rows returned by querying the table with the contents of");
	ol("the specified instance of <tt>alltypes</tt> or <tt>null</tt> if no");
	ol("rows were found. As many fields in <tt>alltypes</tt> can be set as");
	ol("needed and the values of all set fields (including fields explicitly");
	ol("set to <tt>null</tt>) are then used to perform the query. <p>Note,");
	ol("however that this method does use any primary key(s). If the ");
	ol("primary keys are known then one should use the {@link");
	ol("#deleteByKey deleteByKey} method to delete the data instead.");
	ol("<p>Likewise, to delete a previously fetched row, use the {@link ");
	ol("#delete delete} method. This method is really meant to create an new");
	ol("object, set various fields in it, and then use that to delete matching");
	ol("row(s) from the database in a type safe way.");
	ol("<p>");
	ol("This method is often convenient/safer than the {@link #deleteWhere");
	ol("deleteWhere} method (because the <tt>deleteWhere</tt> method takes");
	ol("an arbitrary query string which has to be properly escaped by the user).");
	ol();
	ol("<p>However, as a middle ground, this method also takes an <tt>clause</tt> parameter which");
	ol("is sent as is to the database. For example, a clause can be:");
	ol("<blockquote><pre>");
	ol("List beans = table_fooMgr.<font color=blue>deleteUsing(proto, <b>\"xyx > 5\"</b>)</font>;");
	ol("</pre> </blockquote>");	
	ol("This clause is optional. Specify <tt>null</tt> to not use it at all.");
	ol("If the clause is specified, do <font size='+1'><b>NOT</b></font> include the word <tt>WHERE</tt>.");
	ol();
	ol("<p>Essentially, this method is a more convenient way to use a");
	ol("PreparedStatement. Internally, a prepared statement is created and");
	ol("it's parameters are set to fields that are set in this object).");
	ol("Using PreparedStatements directly is also perfectly fine. For");
	ol("example, the following are equivalent. ");
	ol("<p> Using a PreparedStatement:");
	ol("<blockquote><pre>");
	ol("String foo = \"delete from table_foo where x = ? and y = ?\";");
	ol("PreparedStatement ps = con.prepareStatement(foo);");
	ol("ps.setString(1, \"somevalue\");");
	ol("ps.setString(2, \"othervalue\");");
	ol("int rows_deleted = ps.executeUpdate();");
	ol("</pre> </blockquote>	");
	ol("");
	ol("Using this method:");
	ol("<blockquote><pre>");
	ol("table_foo proto = new table_foo();");
	ol("proto.set_x(\"somevalue\");	//compile time safety");
	ol("proto.set_y(\"othervalue\");	//compile time safety");
	ol("int rows_deleted = table_fooMgr.<font color=blue>deleteUsing</font>(proto);");
	ol("</pre></blockquote>");
	ol("@return		the number of rows deleted");
	ol("*/");
	o("public static int deleteUsing(final Connection con, final ");
	o(beanClassName);
	ol(" bean, final String clause) throws ValidateException, SQLException");
	ol("	{");
	ol("	__deleteusing_called++;");
	ol();
	ol("	Argcheck.notnull(bean, \"the bean parameter was null (and should not be null)\");");
	ol("	if (! bean.isModified()) { ");
	ol("		throw new ValidateException(\"bean=\" + bean + \" not modified, ignoring query\");");
	ol("		}");
	ol();
	ol("	final StringBuilder buf = new StringBuilder(512);");
	o ("	buf.append(\"DELETE  from ");
	o (table.getName()); 
	ol(" WHERE \");");
	ol();
	ol("	int __count = 0;");
	List cols = table.getColumnList();
	List relevant_cols = new ArrayList();
	for (int n = 0; n < cols.size(); n++) 
		{
		ColumnData cd = (ColumnData) cols.get(n);	
		//columns that are not PK only 
		if (cd.isPK()) 
			continue;
					
		o ("	if (bean.");
		o (wrangler.getIsModifiedName(cd));
		ol("()) { ");
				
		//for the later call to utilFillPStmtFromList_IfModified		
		relevant_cols.add(cd);  

		if (! cd.usesPrimitiveJavaType()) {
			o ("		if (bean."); o(wrangler.getGetName(cd)); ol("() == null) {");
			o ("			buf.append(\""); o(cd.getName()); ol(" is NULL and \");");
			ol("			}");
			ol("		else{");
			o ("			buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
			ol("			__count++;");
			ol("			}");
			}
		else{ //primitive type
			if (cd.isNullable()) 
				{
				o ("		if (bean."); o(wrangler.getIsModifiedNullName(cd)); ol("()) {");
				o ("			buf.append(\""); o(cd.getName()); ol(" is NULL and \");");
				ol("			}");
				ol("		else{");
				o ("			buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
				ol("			__count++;");
				ol("			}");
				}
			else{
				o ("		buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
				ol("		__count++;");
				}
			}
		ol("		}");
		} //for

	if (relevant_cols.size() == 0) {
		ol("	throw new RuntimeException(\"This table contains no non-primary keys. Use the deleteByKey method instead.\");");
		ol("	}");
		return;
		}
		
	//get rid of last "and "
	ol("	buf.setLength(buf.length() - 4);"); 
	ol("	if (__count == 0) {");
	ol("		throw new ValidateException(\"No non-PrimaryKey column was modified/set in this bean. You must set at least one such column. To delete by the Primary key, use the deleteByKey method instead.\");");
	ol("		}");
	ol("	if (clause != null) {");
	ol("		buf.append(\" \");");
	ol("		buf.append(clause);");
	ol("		}");
	ol();
 	ol("	final String getUsingPKStmt = buf.toString();");
	ol("	PreparedStatement ps = prepareStatement(con, getUsingPKStmt);");

	utilFillPStmtFromList_IfModified_Object(relevant_cols, "\t");

	ol("	log.bug(\"Query to run: \", ps);");
	ol("	List list = new ArrayList();");
	ol("	int result = ps.executeUpdate();");
//	ol("	ps.clearParameters();");
	ol("	return result;");	
	ol("	}");
	} //~write deleteusing


final void mgrWriteMethodDeleteUsingNoClause() throws SQLException
	{
	ol();
	ol("/** ");
	o ("Convenience method that invokes {@link getUsing(Connection, ");
	o (beanClassName);
	o (", String) getUsing} with an empty <i><tt>clause</tt></i> parameter.");
	ol("*/");
	o("public static int deleteUsing(final Connection con, final ");
	o(beanClassName);
	ol(" bean) throws ValidateException, SQLException");
	ol("	{");
	ol("	return deleteUsing(con, bean, null);");
	ol("	}");
	} //~write deleteUsingNoClause


final void mgrWriteMethodCount()
	{
	ol();
    ol("private final static String countStmt = \"SELECT count(*) from " + table.getName() + "\";");
	ol("/**");
	ol("Returns the count of all rows in the table. <p><b>Note</b>: This may");
	ol("be an expensive operation in MVCC databases like PostgresQL, Oracle and");
	ol("others, where an entire non-optimized table scan <i>may</i> be");
	ol("required -- hence speed will typically be O(n). However, on Postgres (for");
	ol("example), this is still very fast for small values of n (on a");
	ol("mid-level test machine) as of 2004, counting 4k records was about");
	ol("15 milli-seconds(ms); this scaled almost linearly, so count(*) for 16k records was");
	ol("about 70 ms, 65k records was about 370 ms, 524k records was about");
	ol("2000 ms and 1 million records was about 4000 ms. Results will vary");
	ol("on your machine and database but the general O(n) principle will");
	ol("remain the same.");
	ol("*/");
	ol("public static int count(final Connection con) throws SQLException");
	ol("	{");
	ol("	__count_called++;");
	ol("  	int __count = -1;");
	ol("	final Statement stmt = con.createStatement();");
    ol("    final ResultSet rs = stmt.executeQuery(countStmt);");
    ol("    if (rs.next())");
    ol("    	{");
    ol("      	__count = rs.getInt(1);");
    ol("    	}");
	ol("	else { //rs returned no count, which should never happen");
	ol("		throw new DBOException(\"The COUNT query [\" + countStmt + \"] returned no rows. [Should have returned 1 row exactly] \");");
	ol("		}");
	ol("	stmt.close();");
	ol("	return __count;");
	ol("	}");
  	} //~write count
	
final void mgrWriteMethodCountWhere()
	{
	ol();
	ol("/**");
	ol("Returns the count of rows in the table using the specified <tt>where</tt> clause.");
	ol("(note: the string <tt>\"WHERE\"</tt> does <b>not</b> have to be");
	ol("specified in the clause. It is added automatically by this method).");
	ol();
	ol("@throws 	IllegalArgumentException 	if the where paramater was null");
	ol("*/");
	ol("public static int countWhere(final Connection con, final String where) throws SQLException");
    ol("	{");
	ol("	__countwhere_called++;");
    ol("	Argcheck.notnull(where, \"the where parameter was null\");");
	ol("  	int __count = -1;");															
  	ol("	final String countWhereStmt = \"SELECT count(*) from " + table.getName() + " WHERE \" + where;");
	ol("	Statement stmt = con.createStatement();");
	//the mysql driver does not print stmt.toString() properly so we need
	//to also log the statement string
	ol("	log.bug(\"Query to run: \", stmt, \" \", countWhereStmt);");
  	ol("    ResultSet rs = stmt.executeQuery(countWhereStmt);");
    ol("    if (rs.next())");
    ol("    	{");
    ol("      	__count = rs.getInt(1);");
    ol("    	}");
	ol("	else { //rs returned no count, which should never happen");
	ol("		throw new DBOException(\"The COUNT query [\" + countWhereStmt + \"] returned no rows. [Should have returned 1 row exactly] \");");
	ol("		}");
	ol("	stmt.close();");
	ol("	return __count;");
	ol("	}");
	} //~write countwhere


final void mgrWriteMethodCountUsingNoClause() throws SQLException
	{
	ol();
	ol("/** ");
	o ("Convenience method that invokes {@link getUsing(Connection, ");
	o (beanClassName);
	o (", String) getUsing} with an empty <i><tt>clause</tt></i> parameter.");
	ol("*/");
	o("public static int countUsing(final Connection con, final ");
	o(beanClassName);
	ol(" bean) throws ValidateException, SQLException");
	ol("	{");
	ol("	return countUsing(con, bean, null);");
	ol("	}");
	} //~write deleteUsingNoClause

	
final void mgrWriteMethodCountUsing() throws SQLException
	{
	ol();
	ol("/** ");
	ol("Returns the rows count by querying the table with the contents of the");
	o ("specified instance of <tt>");
	o (beanClassName);
	ol("</tt> As many fields in <tt>alltypes</tt> can be set as needed and the");
	ol("values of all set fields (including fields explicitly set to");
	ol("<tt>null</tt>) are then used to perform the query. If the primary");
	ol("key(s) are known then one can also use the {@link #exists} method to");
	ol("see if that row exists in the database.");
	ol("<p>");
	ol("This method is often convenient/safer than the {@link #countWhere");
	ol("countWhere} method (because the <tt>countWhere</tt> method takes an");
	ol("arbitrary query string which has to be properly escaped by the");
	ol("user). ");
	ol("<p>Essentially, this method is a more convenient way to use a");
	ol("PreparedStatement (with parameters set to fields that are set in");
	ol("this object). Using PreparedStatements directly is also perfectly");
	ol("fine. For example, the following two are equivalent. <p>");
	ol("Using a PreparedStatement:");
	ol("<blockquote><pre>");
	ol("String foo = \"select <i>count(*)</i> from table_foo where x = ? and y = ?\";");
	ol("PreparedStatement ps = con.prepareStatement(foo);");
	ol("ps.setString(1, \"somevalue\");");
	ol("ps.setString(2, \"othervalue\");");
	ol("ResultSet rs  = ps.executeUpdate();");
	ol("rs.next();");
	ol("int count = rs.getInt(1);");
	ol("</pre> </blockquote>");
	ol("");
	ol("Using this method:");
	ol("<blockquote><pre>");
	ol("table_foo proto = new table_foo();");
	ol("proto.set_x(\"somevalue\");	//compile time safety");
	ol("proto.set_y(\"othervalue\");	//compile time safety");
	ol("int count = table_fooMgr.<font color=blue>countUsing</font>(proto);");
	ol("</pre> </blockquote>");	
	ol("The clause is optional. Specify <tt>null</tt> to not use it at all.");
	ol("If the clause is specified, do <font size='+1'><b>NOT</b></font> include the word <tt>WHERE</tt>.");
	ol("*/");
	o("public static int countUsing(final Connection con, final ");
	o(beanClassName);
	ol(" bean, final String clause) throws ValidateException, SQLException");
	ol("	{");
	ol("	__countusing_called++;");
	o ("	Argcheck.notnull(bean, \"the bean parameter was null (and should not be null)\");");
	ol("	if (! bean.isModified()) { ");
	ol("		throw new ValidateException(\"bean=\" + bean + \" not modified, ignoring query\");");
	ol("		}");
	ol();
	ol("	int __count = 0;");
	ol("	final StringBuilder buf = new StringBuilder(512);");
	o ("	buf.append(\"SELECT count(*)  from " );
	o (table.getName());
	ol(" WHERE \");");

	List cols = table.getColumnList();
	List relevant_cols = new ArrayList();
	for (int n = 0; n < cols.size(); n++) 
		{
		ColumnData cd = (ColumnData) cols.get(n);	
		
		//for the later call to utilFillPStmtFromList_IfModified		
		relevant_cols.add(cd);  
			
		o ("	if (bean.");
		o (wrangler.getIsModifiedName(cd));
		ol("()) { ");
		if (! cd.usesPrimitiveJavaType()) {
			o ("		if (bean."); o(wrangler.getGetName(cd)); ol("() == null) {");
			o ("			buf.append(\""); o(cd.getName()); ol(" is NULL and \");");
			ol("			}");
			ol("		else{");
			o ("			buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
			ol("			__count++;");
			ol("			}");
			}
		else{ //primitive type
			if (cd.isNullable()) 
				{
				o ("		if (bean."); o(wrangler.getIsModifiedNullName(cd)); ol("()) {");
				o ("			buf.append(\""); o(cd.getName()); ol(" is NULL and \");");
				ol("			}");
				ol("		else{");
				o ("			buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
				ol("			__count++;");
				ol("			}");
				}
			else{
				o ("		buf.append(\""); o (cd.getName()); o ("=? and "); ol("\");");	
				ol("		__count++;");
				}
			}
		ol("		}");
		}		
	ol();
	
	//get rid of last "and "
	ol("	buf.setLength(buf.length() - 4);"); 
	ol("	if (clause != null) {");
	ol("		buf.append(\" \");");
	ol("		buf.append(clause);");
	ol("		}");
	ol();
	ol("	final String countUsingStmt = buf.toString();");
	ol("	PreparedStatement ps = prepareStatement(con, countUsingStmt);");

	utilFillPStmtFromList_IfModified_Object(relevant_cols, "\t");

	ol("	log.bug(\"Query to run: \", ps);");
	ol("	ResultSet rs = ps.executeQuery();");
	ol("	if (! rs.next()) {");
	ol("		throw new DBOException(\"The COUNT query [\" + countUsingStmt + \"] returned no rows. [Should have returned 1 row exactly] \");");
	ol("		}");
	ol("	int rows = rs.getInt(1);");
	ol("	rs.close();");
//	ol("	ps.clearParameters();");
	ol("	return rows;");	
	ol("	}");
	} //~write countusing
	
	
final void mgrWriteMethodExists() throws SQLException
	{	
	if (pklist.size() == 0) {
		ol("/* exists() not implemented since this table does not have any primary keys defined */");
		return;
		}
		
	ol();
	o("static private final String existsStmt = \"SELECT count(*) from "); 
	o(table.getName());
	o(" WHERE ");
	o(Table.getPreparedStmtPlaceholders(pklist));
	ol("\";");
	
	ol("/**");
	ol("Returns <tt>true</tt> if a row with the specified primary keys exists, <tt>false</tt> otherwise.");
	ol("<p>This method uses a prepared statement and is safe from SQL injection attacks");
	ol("*/");
	o("public static boolean exists(final Connection con, ");	
	//example: int col_a, String col_b ....
	o(pkFormalParamString);  
	ol(") throws SQLException");
	ol("	{");
	ol("	__exists_called++;");
	ol("	PreparedStatement ps = prepareStatement(con, existsStmt);");
	for (int n = 0; n < pklist.size(); n++)
		{
		ColumnData cd = (ColumnData) pklist.get(n);	
		String varname = cd.getName(); //example: col_a
		o("\t");
		String pos = String.valueOf((n+1));
		ol(cd.getPreparedStmtSetMethod("ps.", pos, varname));
		}
	ol("	log.bug(\"Query to run: \", ps);");
	ol("	ResultSet rs = ps.executeQuery();");
	ol("	int __count = -1;");
    ol("    if (rs.next())");
    ol("    	{");
    ol("      	__count = rs.getInt(1);");
    ol("    	}");
	ol("	else { //rs returned no count, which should never happen");
	ol("		throw new DBOException(\"The COUNT query [\" + existsStmt + \"] returned no rows. [Should have returned 1 row exactly] \");");
	ol("		}");
	ol("	rs.close();");
//	ol("	ps.clearParameters();");
	ol("	return (__count > 0); //exists if __count > 0");
	ol("	}");
	}
	
final void mgrWriteMethodExistsUsing() throws SQLException
	{	
	ol("/**");
	o ("A thin wrapper around {@link getUsing(Connection,"); o(beanClassName);
	ol(") getUsing}");
	ol("that returns <tt>false</tt> if no rows are returned, <tt>true</tt> otherwise.");
	ol("*/");
	o("public static boolean existsUsing(final Connection con, final ");
	o(beanClassName);
	ol(" bean) throws ValidateException, SQLException");
	ol("	{");
	ol("	final List list = getUsing(con, bean, null);"); 
	ol("	return (list.size() > 0);");
	ol("	}");
	}
	
final void mgrWriteMethodPrepareStatement()
	{
	ol();
	ol("/**");
	ol("Returns a prepared statement given it's variable name.");
	ol();
	ol("Essentially speeds up the creation of prepared statements perhaps");
	ol("using a per connection cache -- which makes sense for pooled");
	ol("connections since they are not closed but returned to the pool");
	ol("and hence don't need to \"prepare\" statements every time (the");
	ol("prepareStatement call is seperate from actually filling in the");
	ol("placeholders in a already created prepared statement -- which");
	ol("does need to be done every time). <p>");
	ol("Prepared statements are unique per connection, so multiple threads");
	ol("using different connections won't stomp over each other's prepared");
	ol("statements. Multiple threads using the SAME connection will cause");
	ol("bizarre errors but multiple threads won't get the same connection");
	ol("from the connection manager -- ever :-), so that should never happen.");
	ol("*/");
	o("private static final PreparedStatement prepareStatement (");
	ol(" final Connection con, final String sql) throws SQLException");
	ol("	{");
	ol("	if (! (con instanceof fc.jdbc.PooledConnection) ) { ");
	ol("		return con.prepareStatement(sql);");
	ol("		}");
	ol("	final PooledConnection pc = (PooledConnection) con;");
	ol("	return pc.getCachedPreparedStatement(sql);");
	ol("	}");
	}

final void mgrWriteCheckDiscarded()
	{
	ol();
	ol("private static final void checkDiscarded(final DBO bean) throws DBOException");
	ol("	{");
	ol("	if (bean.isDiscarded()) {");
	ol("		throw new DBOException(\"===== Attempt to save a discarded object === \" + bean);"); 	
	ol("		}");
	ol("	}");
	}
	
final void mgrWriteMethodStats()
	{	
	ol();
	ol("private static final java.util.Date __loadDate = new java.util.Date();");
	ol("/** Returns usage statistics for this class */");
	ol("public static String stats() ");
	ol("	{");
	ol("	//locally created _numberFormat for thread safety");
	ol("	final java.text.NumberFormat _numberFormat = java.text.NumberFormat.getInstance();");
	ol("	final String nl = fc.io.IOUtil.LINE_SEP;");
	ol("	StringBuffer buf = new StringBuffer(256);");
	o ("	buf.append(\"Class Name: [");
	o (mgrClassName);
	ol("]; Class loaded on: \");"); 
	ol("	buf.append(__loadDate);");
	ol("	buf.append(nl);");
	o ("	buf.append(\"---- Start Usage Statistics ----\")"); ol(".append(nl);");
	ol();
	ol("	ByteArrayOutputStream out = new ByteArrayOutputStream(512);");
	ol("	TablePrinter.PrintConfig config = new TablePrinter.PrintConfig();");
	ol("	config.setPrintBorders(false);");
	ol("	config.setCellSpacing(1);");
	ol("	config.setCellPadding(0);");
	ol("	config.setAutoFit(true);");
	ol("	TablePrinter p = new TablePrinter(2, new PrintStream(out), config);");
	ol("	p.startTable();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"Method\");");
	ol("	p.printCell(\"# times called\");");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"getAll()\");");
	ol("	p.printCell(_numberFormat.format(__getall_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"getLimited()\");");
	ol("	p.printCell(_numberFormat.format(__getlimited_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"getByKey()\");");
	ol("	p.printCell(_numberFormat.format(__getbykey_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"getWhere()\");");
	ol("	p.printCell(_numberFormat.format(__getwhere_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"getUsing()\");");
	ol("	p.printCell(_numberFormat.format(__getusing_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"getUsing(prepared_stmt)\");");
	ol("	p.printCell(_numberFormat.format(__getusing_ps_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"getFromRS()\");");
	ol("	p.printCell(_numberFormat.format(__getfromrs_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"save()\");");
	ol("	p.printCell(_numberFormat.format(__save_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"delete()\");");
	ol("	p.printCell(_numberFormat.format(__delete_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"deleteByKey()\");");
	ol("	p.printCell(_numberFormat.format(__deletebykey_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"deleteWhere()\");");
	ol("	p.printCell(_numberFormat.format(__deletewhere_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"deleteUsing()\");");
	ol("	p.printCell(_numberFormat.format(__deleteusing_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"count()\");");
	ol("	p.printCell(_numberFormat.format(__count_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"countWhere()\");");
	ol("	p.printCell(_numberFormat.format(__countwhere_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"countUsing()\");");
	ol("	p.printCell(_numberFormat.format(__countusing_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.startRow();");
	ol("	p.printCell(\"exists()\");");
	ol("	p.printCell(_numberFormat.format(__exists_called));");
	ol("	p.endRow();");
	ol();
	ol("	p.endTable();");
	ol("	buf.append(out.toString());");
	ol("	return buf.toString();");
	ol("	}");
	}
	
final void mgrWriteMethodToString()
	{	
	ol();
	ol("public String toString() ");
	ol("	{");	
	ol("	return getClass().getName() + \" [call stats() for more info]\";");
	ol("	}");
	}

void mgrWriteValidators() throws SQLException
	{
	ol();
	ol("// ================ Validation  ==================== ");
	ol();
	ol("/** ");
	ol("Creates and attaches validators for all the fields in the");
	ol("specified {@link fc.web.forms.Form}. These fields should");
	o("<i>have the same name</i> in the form as in {@link "); o(beanClassName);
	ol("}. If this is not the case, then the then the differences can be specifed");
	ol("as follows. <p>");
	ol("<dl>");
	ol("<dt>with a prefix</dt>");
	o ("	<dd><tt>(prefix + ");o(beanClassName);ol(" column)</tt> should equal <tt>form fieldname</tt></dd>");
	ol("<dt>with a suffix</dt> ");
	o ("	<dd><tt>(");o(beanClassName);ol(" column + suffix)</tt> should equal <tt>form fieldname</tt></dd>");
	ol("<dt>with both a prefix/suffix</dt> ");
	o ("	<dd><tt>(prefix + ");o(beanClassName);ol(" + suffix)</tt> should equal <tt>form fieldname</tt></dd>");
	ol("<dt>with a arbitrary map</dt> ");
	o ("	<dd>[key] <tt>");o(beanClassName);ol(" column</tt> -> [value] <tt>form fieldname</tt>");
	ol("	<u>If a map is specified, then the prefix/suffix are not used.</u>");
	ol("	</dd>");
	ol("</dl>");
	ol("<p>These validators are for database constraints such as <i>nullability</i> & <i>column length</i>.");
	ol("These validators save a lot of grunt-work in adding such schema");
	ol("constraints to the front-end {@link fc.web.forms.Form}. <p><b>However, <i>business and");
	ol("other validation constraints</i> still need to be manually added to");
	ol("the application code/front-end forms as/when needed</b>.");
	ol("<p>");
	ol("");
	ol("The following table shows the kind of validators added by this method");
	ol("<table border=1 width=90%>");
	ol("<tr bgcolor='#CCCCCC'>");
	ol("	<td>Database SQL Type</td>");
	ol("	<td><b>Nullable</b>validator</td>");
	ol("	<td><b>Length</b> validator</td>");
	ol("	<td><b>Digits only</b> input validator ({@link VText#allowIntegersOnly})</td>");	
	ol("</tr>");
	ol("	<tr>");
	ol("		<td><tt>CHAR</tt>, <tt>VARCHAR</tt></td>");
	ol("		<td>Yes (maximum length constraint).<br><font size='-1' color=red>This");
	ol("		only applies to form fields that are subclasses of {@link ");
	ol("		fc.web.forms.MaxSizable} </font></td>");
	ol("		<td>-NO-</td>");
	ol("	</tr>");
	ol("	<tr>");
	ol("		<td><tt>TINYINT, MEDIUMINT, INT, BIGINT (integral types)</tt></td>");
	ol("		<td>Yes</td>");
	ol("		<td>-NO-</td>");
	ol("		<td>Yes to integer columns displayed using form fields that are subclasses of {@link fc.web.forms.AbstractText}<br> Note: <b>not</b> added non-<i>integral</i> number types such as <tt>FLOAT, REAL, DOUBLE, NUMERIC/DECIMAL</tt></td>");
	ol("	</tr>");
	ol("	<tr>");
	ol("		<td>All other SQL types</td>");
	ol("		<td>Yes</td>");
	ol("		<td>-NO-</td>");
	ol("	</tr>");
	ol("</table>");
	ol("<p>Automatic validators are very useful but can be very tricky to understand. It is");
	ol("suggested to invoke this method, print the form using it's <tt>toString</tt>");
	ol("method and then examine the output to see what validators were added If those");
	ol("automatic validators are too little, too many or too hard to understand, <u>then");
	ol("simply enoough, do NOT invoke this method and simply add validators by");
	ol("hand</u>. In particular, do <i>not</i> add automatic validators for");
	ol("<b>tables</b> in which a row is optional but <i>if</i> some column is filled in");
	ol("the front end form, <i>then</i> all columns must be filled.");
	ol();
	ol("@param	form	the form containing fields (some or all) representing");
	ol("				this and possible other tables. These field");
	ol("				objects must have been added to the form prior");
	ol("				to calling this method");
	ol("@param	prefix  an optional (null allowed) prefix to this table's column name with which the");
	ol("				corresponding column was added to the form.");
	ol("				A <tt>*</tt> specifies all possible prefixes");
	ol("@param	suffix  an optional suffix (null allowed) to this table's column name with which the ");
	ol("				corresponding column was added to the form.");
	ol("				A <tt>*</tt> specifies all possible suffixes");
	ol("@param	map		an optional map (null allowed) that maps this table's column name with which the ");
	ol("				corresponding column was added to the form. ");
	ol("				[key] <tt>table's column_name</tt> -> [value] <tt>form's fieldname</tt>");
	ol("*/");	
	ol("public static void addValidators(final fc.web.forms.Form form, final String prefix, final String suffix, final Map map) ");
	ol("	{");
	ol("	addValidators(form, prefix, suffix, map, false);");
	ol("	}");
	ol();
	
	ol("private static void addValidators(fc.web.forms.Form form, String prefix, String suffix, final Map map, final boolean onlyOnFilled) ");
	ol("	{");
	ol("	List list = null;");	
	ol("	Argcheck.notnull(form, \"How can I add validators to the form when the form parameter was null ?\");");
	ol("	Field field = null;");
	ol("	FieldValidator fv = null;");
	ol("	String colname_in_form = null;");
	ol("	//fields can be null if they are not being used in the html form");
	List cols = table.getColumnList();
	for (int n = 0; n < cols.size(); n++)
		{
		ColumnData cd = (ColumnData) cols.get(n);
		String colname = cd.getName();
		String sqltypename = cd.getSQLTypeName().intern();
		ol();
		o ("\t");ol(getBeanComment(cd));
		o ("	list = getFieldFromForm(form, \"");
		o (colname); 
		o ("\", prefix, suffix, map, ");
	
		//false=no warn message if this field not in form
		if (cd.isAutoIncrement()) 
			o("false"); 
		else
			o("true");
			
		ol(");");
		ol("	if (list.size() > 0) ");
		ol("		{ //add applicable automatic validators, empty if n/a");
		ol("		for (int n = 0; n < list.size(); n++)");
		ol("			{");
		ol("			field = (Field) list.get(n);");

		//nullable
		if (! cd.isNullable()) 
			{
			if (cd.hasDefaultValue()) {
				o ("			/* field is non-nullable but has a default value [");
				o (cd.getDefaultValue()); 
				ol("], skipping non-nullability validation */");
				}
			else{
				ol("			if (field instanceof Choice) {");
				ol("				//choice fields are ignored because they can");
				ol("				//mean false even when NOT selected/filled");
				ol("				continue;");
				ol("				}");
				ol("			/* field is non-nullable */");
				o ("			fv = new VFilled(field, \"");
				o(validateNull_ErrorMsg);
				ol("\");"); 				
				}
			}
		else{
			ol("			/* field is nullable, skipping non-nullability validation */");
			}

		//numbers only	
		if (sqltypename.toLowerCase().indexOf("int") >= 0)
			{
			ol();
			ol("			/* database type for this field is integral */");
			ol("			if (field instanceof AbstractText) {");
			o ("				fv = new VText((AbstractText)field, \"");
			o(validateIntegerOnly_ErrorMsg);
			ol("\")");
			ol("					.allowIntegersOnly();");
			ol("				}");
			}
		
		//length
		if ((sqltypename == "CHAR" || sqltypename  == "VARCHAR"))
			/*
			colsize IS NOT accurate/reliable for non text types.
			(therefore we only set maxsize for char/varchar)
			
			text(longvarchar) size is not knowable with postgres and postgres
			returns -1 for LONGVARCHAR Except now we have another bug, the
			current driver returns VARCHAR even for text columns (instead of
			LONGVARCHAR). What the fuck ? They are fucking up the best free
			database on the planet with a shitty jdbc driver. god, it's almost
			5am and i'm tired. p.s: postgres jdbc driver + support is still
			waaay better than all other competing db's though.
			*/
			{
			//bug workaround
			int colsize = cd.getSize();
			if (colsize < 0)
				colsize = Integer.MAX_VALUE;
			//end workaround

			ol("			if (! (field instanceof MaxSizable)) {");
			o ("				log.warn(\"Skipping maximum length validator for field '\" + field.getName() + \"'; [database type='");o(sqltypename);ol("', field.type='\" +  field.getType() + \"' is not MaxSizable]\"); ");
			ol("				}");
			ol("			else{");
			o ("				VText vt = new VText((MaxSizable) field, \"");
			o(validateText_ErrorMsg_MaxSize);
			ol("\");"); 				
			o ("				vt.setMaxSize(");
			o (String.valueOf(colsize));
			ol(");");
			ol("				}");
			}
		ol("			}");  //generated for
		ol("		}");	
		}
	ol("	}");
	ol();

	ol("/** ");
	ol("Convenience method that calls {@link #addValidators(Form, String, String, Map)} with a ");
	ol("<tt>null</tt> prefix/suffix and the specified map");
	ol("*/");
	ol("public static void addValidators(fc.web.forms.Form form, Map map) ");
	ol("	{");
	ol("	addValidators(form, null, null, map);");
	ol("	}");
	ol();

	ol("/** ");
	ol("Convenience method that calls {@link #addValidators(Form, String, String, Map)} with a ");
	ol("<tt>null</tt> prefix/suffix/map");
	ol("*/");
	ol("public static void addValidators(fc.web.forms.Form form) ");
	ol("	{");
	ol("	addValidators(form, null, null, null);");
	ol("	}");
	ol();

	ol("/** ");
	ol("Convenience method that calls {@link #addValidators(Form, String, String, map)} with the ");
	ol("specified prefix and a <tt>null</tt> suffix/map");
	ol("*/");
	ol("public static void addValidators(fc.web.forms.Form form, String prefix) ");
	ol("	{");
	ol("	addValidators(form, prefix, null, null);");
	ol("	}");
	ol();

	ol("/** ");
	ol("Validates a form field <i>if</i> it is filled by the user. Leaves empty fields alone.");
	ol("This is very useful for fields that are optional but must have the correct value when");
	ol("filled by the user");
	ol("*/");
	ol("public static void addIfFilledValidators(Form form, String prefix) ");
	ol("	{");
	ol("	addValidators(form, prefix, null, null, true);");
	ol("	}");
	ol();
	
	ol("/** implementation helper method -- not for public use */");
	ol("static List getFieldFromForm(Form form, String colname, String prefix, String suffix, Map map, boolean warn)");
	ol("	{");
	ol("	Field field = null;");
	ol("	List list = Form.empty_list;");
	ol("	boolean getwhere = false;");
	ol("	getwhere = false;");
	ol("	String colname_in_form = colname;");
	ol();
	ol("	if (map != null) {");
	ol("		String str = (String) map.get(colname);");
	ol("		if (str != null) {");
	ol("			prefix = null;  /*ignored when there is a mapping*/");
	ol("			suffix = null;  /*ignored when there is a mapping*/");
	ol("			colname_in_form = str; /* else if not in map, colname remains as-is*/");
	ol("			}");
	ol("		}");
	ol();
	ol("	if (prefix != null) ");
	ol("		{ ");
	ol("		if (prefix.equals(\"*\")) { ");
	ol("			getwhere = true;");
	ol("			}");
	ol("		else{");
	ol("			colname_in_form = prefix + colname_in_form;");
	ol("			}");
	ol("		}");
	ol();
	ol("	if (suffix != null) ");
	ol("		{");
	ol("		if (suffix.equals(\"*\")) { ");
	ol("			getwhere = true;");
	ol("			}");
	ol("		else{");
	ol("			colname_in_form = colname_in_form + suffix;");
	ol("			}");
	ol("		}");
	ol();
	ol("	if (getwhere) { ");
	ol("		list = form.getContaining(colname_in_form);");
	ol("		if (list.size() == 0 && warn) warn(form, colname_in_form, suffix, prefix, map);");
	ol("		return list;");
	ol("		}");
	ol("	else{");
	ol("		//containsField() check prevents an un-necessary warning with form.get()");
	ol("		if (! form.containsField(colname_in_form)) {");
	ol("			if (warn) warn(form, colname_in_form, suffix, prefix, map);");
	ol("			return list;");
	ol("			}");
	ol("		field = form.get(colname_in_form);" );
	ol("		list = new ArrayList();");
	ol("		list.add(field);");
	ol("		}");
	ol("	return list;");
	ol("	}");

	ol();
	ol("private static final void warn(Form form, String name, String suffix, String prefix, Map map) {");
	ol("	log.warn(form.getName(),\": No automatic validators will be added for Field [\",name,\"]. This field does not exist in the front-end form. (this could be normal). suffix=[\"+suffix+\"] prefix=[\"+prefix+\"] map=\", map);");
	ol("	}");
	
	//ValidateBeforeSaveNew	
	ol();
	ol("/** ");
	ol("Validates the bean before saving it to the database. This");
	ol("method is called internally by the {@link save()} method");
	ol("before saving a new bean (i.e., inserting a new row) to the");
	ol("database.");
	ol("<p>");
	ol("The validation is somewhat basic and there can exist many");
	ol("constraints and conditions on the database that might results in a");
	ol("insert/update error anyway. But by doing some basic validation");
	ol("against some known constraints, we save a needless trip to the");
	ol("database.");
	ol("We check to see that: <ol>");
	ol("<li><i>non-nullable and non auto-increment</i> columns [and with no default");
	ol("column value in the database] are modified (via a set method) since these");
	ol("columns must have a explicitly set value.</li>");
	ol("<li>for non-nullable columns that hold non-primitive (i.e., Object)");
	ol("java types, the modified value for non-nullable columns is a");
	ol("non-null object. </li>");
	ol("</ol>");
	ol("*/");
	o ("protected static void validateBeforeSaveNew(");
	o (beanClassName);
	ol(" bean) throws ValidateException");
	ol("	{");
	ol("	boolean error = false;");
	ol("	final StringBuffer buf = new StringBuffer(\"The following validation errors were found\").append(IOUtil.LINE_SEP);");
	for (int n = 0; n < cols.size(); n++)
	{
	ColumnData cd = (ColumnData) cols.get(n);
	String colname = cd.getName();
	ol();
	if (cd.isNullable()) {
		o ("	//["); o (colname);
		ol("]=>nullable, no modification check necessary, skipping...");
		}
	else if ( ! cd.isNullable() && cd.hasDefaultValue()) {
		o ("	//["); o (colname);
		o ("]=>is not nullable but has a default column value [");
		o (cd.getDefaultValue()); 
		ol("], no modification check necessary, skipping...");
		}
	else if (cd.isAutoIncrement()) {
		o ("	//["); o (colname);
		ol("]=>auto-increment, no modification check necessary, skipping...");
		}		
	else { //not nullable and value required
		o("	if (! bean.");
		o(wrangler.getIsModifiedName(cd));
		ol("()) {");
		ol("		error = true;");
		o ("		buf.append(\"");
		o (colname);
		o (" was not set (this field is required in the database)\").append(\";current value=\"");
		o (").append(bean.");
		o (wrangler.getGetName(cd));
		ol("()).append(IOUtil.LINE_SEP);");
		ol("		}");
		
		if (! cd.usesPrimitiveJavaType())
			{
			ol("	else { //was modified but to null");
			o ("		if (bean.");
			o (wrangler.getGetName(cd));
			ol("() == null) {");
			ol("			error = true;");
			o ("			buf.append(\"");
			o (colname);
			o (" was set to null (but is non-nullable)\").append(\";current value=\"");
			o (").append(bean.");
			o (wrangler.getGetName(cd));
			ol("()).append(IOUtil.LINE_SEP);");
    		ol("			}");
    		ol("		}");
    		}
    	else{
    		o ("	//");
			o (cd.getJavaTypeFromSQLType());
			ol(" is primitive, skipping null test");			
			}
		}
	} //for

	ol("	if (error) { ");
	ol("		throw new ValidateException(buf.toString());");
	ol("		}");
	ol("	}");

	//ValidateBeforeSaveUpdate
	ol();
	ol("/** ");
	ol("Validates the bean before saving it to the database. This method is");
	ol("called internally by the {@link save()} method before updating an");
	ol("existing bean (i.e., updating a row) in the database.");
	ol("<p>");
	ol("For <i>each modified column</i>, if that column is non-nullable in");
	ol("the database, then it must have a non-null value in the bean before");
	ol("it is saved. This check is only done for fields of");
	ol("<i>non</i>-primitive (Object) java types. [There is no way to ensure");
	ol("a non-null value for <i>primitive</i> types since all values");
	ol("[including 0] are non-null for those types].");
	ol("*/");
	o ("protected static void validateBeforeSaveUpdate(");
	o (beanClassName);
	ol(" bean) throws ValidateException");
	ol("	{");
	ol("	boolean error = false;");
	ol("	final StringBuffer buf = new StringBuffer(\"The following validation errors were found\").append(IOUtil.LINE_SEP);");
	ol();

	for (int n = 0; n < cols.size(); n++)
		{
		ColumnData cd = (ColumnData) cols.get(n);
		String colname = cd.getName();

		if (cd.isNullable()) {
			o ("	//["); o (colname);
			ol("]=>nullable, no modification check necessary, skipping...");
			}
		/* 
		dont check for existence of default column value here, if it's an
		update, prolly a good idea for the bean to have a updated value and not
		rely on column defaults 
		*/
		else if (cd.isAutoIncrement()) {
			o ("	//["); o (colname);
			ol("]=>auto-increment, no modification check necessary, skipping...");
			}
		/*
		else if (cd.isPK()) { //must be able to find the row to update
			o("	if (! bean.");
			o(wrangler.getIsModifiedName(cd));
			ol("())	{");
			ol("		error = true;");
			o ("		buf.append(\"");
			o (colname);
			ol(" was not set (this field is a primary key and is needed to find the row to update.\");");
			ol("		}");	
			}
		*/
		/* unlike validatebeforesave, we don't check to see if every 
		required column has been modified (an update allows updates
		whatever is modified, and leaves the rest (required or non required)
		alone
		*/
		else{
			o("	if (bean.");
			o(wrangler.getIsModifiedName(cd));
			ol("()) {");		
			if (! cd.usesPrimitiveJavaType())
				{
				o ("		if (bean.");
				o (wrangler.getGetName(cd));
				ol("() == null) {");
				ol("			error = true;");
				o ("			buf.append(\"");
				o (colname);
				o (" was set to null (but is non-nullable)\").append(\";current value=\"");
				o (").append(bean.");
				o (wrangler.getGetName(cd));
				ol("()).append(IOUtil.LINE_SEP);");
				ol("			}");
				}
			else{
				o ("		//");
				o (cd.getJavaTypeFromSQLType());
				ol(" is primitive, skipping null test");			
				}
				ol("		}");
			}
		} //for

	ol("	if (error) { ");
	ol("		throw new ValidateException(buf.toString());");
	ol("		}");
	ol("	}");

	}//~mgrWriteValidators()

void o(String str) {
	out.print(str);
	}

void o(String str, String str2) {
	out.print(str);
	out.print(str2);
	}

void ol(String str) {
	out.println(str);	
	}

void ol() {
	out.println();
	}

void utilFillPStmtFromList_IfModified (
	final List list, final String tabprefix) 
throws SQLException
	{
	final String tabprefix2 = tabprefix + "\t";
	final String tabprefix3 = tabprefix + "\t\t";

	o(tabprefix);
	ol("int pos = 0;");

	for (int n = 0; n < list.size(); n++)
		{
		ColumnData cd = (ColumnData) list.get(n);	
		String colname = cd.getName();

		o(tabprefix);
		o("if (bean.");
		o(wrangler.getIsModifiedName(cd));
		ol("()) {");

		o(tabprefix2);
		ol("pos++;");

		/* 
		if it's modified, then we save it, whether it was originally null in the
		db is irrelevant. of course, if it's set to null, we save it as null,
		else a real value otherwise 
		*/
		
		if (cd.usesPrimitiveJavaType() && cd.isNullable()) 
			{
			o(tabprefix2);
			o ("if (bean.");
			o (wrangler.getIsModifiedNullName(cd)); 
			ol("()) { /* nullable in db */");
			o(tabprefix3);
			o("ps.");
			o(cd.getPreparedStmtSetNullMethod("pos", colname));
			ol(";");
			o(tabprefix3);
			ol("}");
			o(tabprefix2);
			ol("else{");
			o(tabprefix3); 
			o(cd.getJavaTypeFromSQLType()); o(" "); o(colname); 
			o(" = bean."); o(wrangler.getGetName(cd)); ol("();");
			o(tabprefix3);
			ol(cd.getPreparedStmtSetMethod("ps.", "pos", colname));
			o(tabprefix3);
			ol("}");
			o(tabprefix2);
			ol("}");
			} 
		else{		
			o(tabprefix2);
			o(cd.getJavaTypeFromSQLType()); o(" "); o(colname);
			o(" = bean."); o(wrangler.getGetName(cd)); ol("();");
			o(tabprefix2);
			ol(cd.getPreparedStmtSetMethod("ps.", "pos", colname));
			//ol("log.bug(ps);");				
			o(tabprefix2);
			ol("}");
			}
		} //~for
	}


void utilFillPStmtFromList_IfModified_Object (final List list, final String tabprefix) 
throws SQLException
	{
	final String tabprefix2 = tabprefix + "\t";
	final String tabprefix3 = tabprefix + "\t\t";

	o(tabprefix);
	ol("int pos = 0;");

	for (int n = 0; n < list.size(); n++)
		{
		ColumnData cd = (ColumnData) list.get(n);	
		String colname = cd.getName();

		o(tabprefix);
		o("if (bean.");
		o(wrangler.getIsModifiedName(cd));
		ol("()) {");

		if (! cd.usesPrimitiveJavaType() || cd.isNullable()) 
			{
			o(tabprefix2);
			o ("if (bean.");
			if (! cd.usesPrimitiveJavaType()) {
				o (wrangler.getGetName(cd));
				ol("() == null) { ");
				}
			else{
				o (wrangler.getIsModifiedNullName(cd)); 
				ol("()) { ");
				}
			o (tabprefix3);
			ol("/* no value to set here, uses [xxx IS NULL] syntax*/");
			o (tabprefix3);
			ol("}");
			o(tabprefix2);
			ol("else{");
			}
			
		o(tabprefix3);
		ol("pos++;");	
		o(tabprefix3);
		o(cd.getJavaTypeFromSQLType());
		o(" ");
		o(colname);
		o(" = bean.");
		o(wrangler.getGetName(cd));
		ol("();");
		o(tabprefix3);
		ol(cd.getPreparedStmtSetMethod("ps.", "pos", colname));
		o(tabprefix3);
		ol("}");

		if (! cd.usesPrimitiveJavaType() || cd.isNullable()) {
			o(tabprefix2);
			ol("}");
			}
		} //~for
	}


public boolean isPostgres() {
	return dbname == DBName.postgres;
	}

public boolean isMysql() {
	return dbname == DBName.mysql;
	}


/**  
Usage: 
java fc.jdbc.dbobjects.Generate -conf
<path-to-configuration-file> No flags will produce a list of
options and usage information.
**/
public static void main(String[] args) throws Exception
	{
	Generate gen = new Generate(args);
	}

}		
