// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc.dbo;

import java.sql.*;
import java.util.*;

import fc.io.*;
import fc.util.*;

/** 
Represents each column in a table, used by generate.java

@author hursh jain
**/
class ColumnData 
{
static boolean 		nullIfNullableUnknown;
static SystemLog	log;
static Types 		types;

static ToString.Style style = new ToString.Style();
static {
	style.reflectVisibleLevel = ToString.Style.VisibleLevel.PRIVATE;
	style.startString = IOUtil.LINE_SEP;
	style.endString = IOUtil.LINE_SEP;
	}

/**
This method must be called before this class is instantiated
*/
static void init(SystemLog log, PropertyMgr props)
	{		
	ColumnData.log = log;
	types = new Types(log, props);
	}
	
private		Table		table; 		//reference to the containing table
private 	String 		colname;  
private 	int			colnum;
private 	int 		sqltype;  	//java.sql.type
private 	String  	typename;   //db/driver specific type name (in pg, we can get json, jsonb, enum, etc this way)
private 	int			colsize;
private 	boolean 	nullable;
private 	boolean		isAutoIncrement;
private 	String  	remarks;
private 	boolean 	hasRemarks;
private		String		default_val;
private 	boolean 	isPK;
//not null if isFK
private 	FKData		fkdata;  

ColumnData(Table table,
		String colname, int colnum, int datatype, String typename,
		int colsize, int nullable, boolean auto_increment,
		String remarks, String default_val)
	{
	this.table				= table;
	this.colname 			= colname;
	this.colnum				= colnum;
	this.sqltype 			= datatype;
	this.typename 			= typename;
	this.colsize 			= colsize;
	this.isAutoIncrement	= auto_increment;
	hasRemarks = ! (remarks == null || remarks.equals(""));
	this.remarks = remarks;
	this.default_val		= default_val;

	//System.out.println(table + "." + colname + ".default_val->"+default_val);
	
	switch (nullable) {
		case DatabaseMetaData.columnNoNulls: 
			this.nullable = false; break;
		case DatabaseMetaData.columnNullable:
			this.nullable = true; break;
		case DatabaseMetaData.columnNullableUnknown:
			this.nullable = nullIfNullableUnknown ? true : false;
		default:
			throw new IllegalArgumentException("received unknown value for NULLABLE field from DBMetaData.getColumns(), value=" + nullable);
		}		
	}

/**
Returns the table to which this column belongs.
*/
public Table getTable()
	{
	return this.table;
	}

/** 
This method must be called to set this column as being a
primary key.
@param val	<tt>true</tt> to set as PK
**/
public void setPK(boolean val) {
	isPK = val;
	}

/** 
This method must be called to set this column as being a
foreign key.
**/
public void setFK (String fk_colname, String pk_table, String pk_colname) 
	{
	fkdata = new FKData(fk_colname, pk_table, pk_colname);
	}

public String getName() { 
	return colname; 
	}
	
public int getSize() {
	return colsize;
	}
	
public int getNumber() {
	return colnum;
	}
	
public boolean isNullable() {
	return nullable;
	}

public boolean isAutoIncrement() {
	return isAutoIncrement;
	}
	
public boolean hasDefaultValue() {
	return default_val != null;
	}

public String getDefaultValue() {
	return default_val;
	}
	
/** 
returns the fkdata or null if this column is
not a fk
*/
public FKData getFK()  {
	return fkdata;
	}

/* 
Returns the name of the <b>position</b>-based result set method to get the value
for a specific column, for example: <tt>getString(1)</tt>. The result set
must contain the list of columns in the expected order.
*/
public String getResultSetMethod() throws SQLException {
	return types.getResultSetMethod(sqltype, String.valueOf(colnum), this);
	}

/* 
Returns the name of the <b>position</b>-based result set method to get the
value for a specificied position, for example, <tt>getString(pos)</tt> where
pos is the position parameter.
*/
public String getResultSetMethod(int pos) throws SQLException {
	//used by mgrWriteMethodSave, when getting a auto inc value. The
	//auto inc value resultset only has 1 column, so we are passed '1'
	//by mgrWriteMethodSave (we don't use the ordinal number of this column
	//itself, since that's not valid for the autoinc query)

	return types.getResultSetMethod(sqltype, String.valueOf(pos), this);
	}


/**
Returns the name of the <b>name</b>-based result set method to get the value for a
specific column. If the specified <tt>prependTableName</tt> argument is
non-null, then the table name is prepended to the column name. For example:
<tt>getString("tablename_colname")</tt>.
<p>
Specify <tt>null</tt> (or an empty string) for the prependValue parameter to
get the method with the column name, for example, <tt>getString("colname")</tt>.
*/
public String getResultSetMethod(String prependValue) throws SQLException 
	{
	final String name = (prependValue != null) ? 
		"\"" + prependValue + colname + "\"": 
		"\"" + colname + "\"";
	return types.getResultSetMethod(sqltype, name, this);
	}


/**
Returns code that creates at runtime, the <b>name</b>-based result set method to 
get the value for a specific column. This is used for when the column prefix is 
not known until runtime (assumes the runtime prefix argument is called "prefix");
*/
public String getRuntimeResultSetMethod() throws SQLException 
	{
	return types.getRuntimeResultSetMethod(sqltype, colname, this);
	}
	
/**
We require the caller to specify the position (and DO NOT use the database
colnum of this colnum) because we are filling in some '?' of our prepared
statement and this position of '?' in our statement is independent of the
position of this ordinal column pos in our database -- this will happen for
example when certain columns in the middle of our table are not part of the
prepared statement.
*/
public String getPreparedStmtSetMethod(String psvar, String position, String varname) 
throws SQLException 
	{
	return types.getPreparedStmtSetMethod(sqltype, psvar, position, varname, this);
	}

public String getPreparedStmtSetNullMethod(String position, String varname) 
throws SQLException 
	{
	return types.getPreparedStmtSetNullMethod(sqltype, position, varname, this);
	}

//returns true if this column uses a corresponding java type
//that is primitive (int, short etc), false if uses an object
//type
public boolean usesPrimitiveJavaType() throws SQLException {
	return types.usesPrimitiveJavaType(sqltype, this);
	}

//returns true if this column uses a simple integral type
//(for inc/dec methods)
public boolean usesSimpleIntegralJavaType() throws SQLException {
	return types.usesSimpleIntegralJavaType(sqltype);
	}

public String getJavaTypeFromSQLType() throws SQLException {
	return types.getJavaTypeFromSQLType(sqltype, this);
	}

//integer constant from java.sql.Type	
public int getSQLType() {
	return sqltype;
	}
	
//name (TEXT,DATE etc) for a java.sql.Type constant
public String getSQLTypeName() {
	return types.getSQLTypeName(sqltype);
	}
	
public String getSQLTypeDriverSpecificName() {
	return typename;
	}	

public String getRemarks() {
	return remarks;
	}


public boolean hasRemarks() {
	return hasRemarks;
	}

public boolean isPK() {
	return isPK;
	}

public boolean isFK() {
	return (fkdata != null);
	}
	
boolean useBooleanObject() 
	{
	if (sqltype == java.sql.Types.BIT ||
		sqltype == java.sql.Types.BOOLEAN)
		{
		if (isNullable() && types.booleanObjectForNullableBooleans) {
			return true;
			}
		}
		
	return false;
	}
	
public String toString() {
	return new ToString(this, style).reflect().render();
	}

//we do not track update/cascade rules since this framework
//does not need that info
public static class FKData 
	{
	String fk_colname, pk_table, pk_colname;
	
	FKData(
		String fk_colname,		
		String pk_table, 
		String pk_colname
		)
		{
		this.fk_colname = fk_colname;
		this.pk_table = pk_table;
		this.pk_colname = pk_colname;
		}
	
	public String getFKColName() { return fk_colname; }
	public String getPKColName() { return pk_colname; }
	public String getPKTableName() { return pk_table;}

	static final ToString.Style style = new ToString.Style();
	static {
		style.reflectVisibleLevel = ToString.Style.VisibleLevel.DEFAULT;
		}
	
	public String toString() {
		return new ToString(this, style).reflect().render();
		}
	
	}	

}          //~class ColumnData