// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc;

import java.sql.*;
import java.util.*;
import java.math.*;
import java.io.*;

import fc.io.*;
import fc.util.*;

/**  
Wrapper class for a {@link java.sql.ResultSet ResultSet} that allows the
value of specified columns to be transformed. Transforms <b>only</b> work
the <b><tt>get<u>Object</u>(..)</tt></b> and
<b><tt>get<u>String</u>(..)</tt></b> methods, all other get methods return
an unmodified result obtained from the underlying result set.

@author hursh jain
**/

public final class ResultSetTransform implements ResultSet
{
static final boolean dbg = false;

ResultSet 			rs;
ResultSetMetaData 	rsmd;
Map 				transformsByIndex;

// cache of column names mapped to their index since resultset.findColumn
// is probably an expensive operation (we need this later when retrieving
// values by column names)
Map ColNameIndex; 

public ResultSetTransform(ResultSet rs) throws SQLException
	{
	Argcheck.notnull(rs, "specified resultset argument was null");
	this.rs = rs;
	this.rsmd = rs.getMetaData();
	transformsByIndex = new HashMap();
	ColNameIndex = new HashMap(); 
	}

/** 
Adds a transform rule for the specified column. When this column's
data is later retrieved via the {@link getObject(int)} or the
{@link getString(int)} methods, this transform rule will be invoked.
(note, the transform rule is not invoked for other <tt>getXXX(..)</tt>
methods).
**/		
public void addColumnTransform(int columnIndex, Rule rule) 
throws SQLException
	{
	Argcheck.istrue(columnIndex > 0, "column index must be greater than zero");
	Argcheck.notnull(rule, "specified rule cannot be null");
	if (dbg) System.out.println("ResultSetTransform: addColumnTransform("+columnIndex+","+rule+")");
	transformsByIndex.put(new Integer(columnIndex), rule);
	}

/** 
Adds a transform rule for the specified column. When this column's
data is later retrieved via the {@link getObject(int)} or the
{@link getString(int)} methods, this transform rule will be invoked.
(note, the transform rule is not invoked for other <tt>getXXX(..)</tt>
methods).

@throws SQLException	if the specified column name was not
						found in the result set or if some other
						underlying SQL error occurs
**/		
public void addColumnTransform(String columnName, Rule rule) 
throws SQLException
	{
 	Argcheck.notnull(columnName, "column name must not be null");
 	Argcheck.notnull(rule, "specified rule cannot be null");
 	if (dbg) System.out.println("ResultSetTransform: addColumnTransform('"+columnName+"',"+rule+")");
   	
   	Integer index = new Integer(rs.findColumn(columnName));
   	if (dbg) System.out.println("'" + columnName + "' is located at column:" + index);
   	ColNameIndex.put(columnName, index);
   	transformsByIndex.put(index, rule);
	}

/** 
A transform rule, added to a specific column index or name via
the {@link addColumnTransform} methods. To instantiate this
class, use the <tt>foo.new Rule()</tt> syntax, where foo is
an instance of the outer ResultSetTransform class.
**/
public abstract class Rule {
	/** Should modify the given column value as needed. **/
   	protected abstract Object transform(int columnIndex, Object value);
	public ResultSet getResultSet() { return rs; }
	public ResultSetMetaData getMetaData() { return rsmd; }
	}
	
Object applyRule(Integer key, Object value) 
	{
	if (dbg) System.out.println("Enter: applyRule("+key+","+value+"), Transforms: " + transformsByIndex);

	if (transformsByIndex.containsKey(key)) {
		Rule r = (Rule) transformsByIndex.get(key);
		return r.transform(key.intValue(), value);
		}
	else  {
		return value;
		}
	}

public static void main(String[] args) throws Exception
	{
	String propfile = "./ConnectionMgrTestProps.txt";

	Args myargs = new Args(args);
	myargs.setUsage("java fc.jdbc.ResultSetTransform -query query-to-execute -colname columnName");

	String 	query  = myargs.getRequired("query");
	String 	colname = myargs.getRequired("colname");
	SimpleConnectionMgr cmgr = 
			new SimpleConnectionMgr( 
				new FilePropertyMgr(new File(propfile)));

	ResultSet rs = cmgr.getConnection().createStatement().executeQuery(query);	

	ResultSetTransform rst = new ResultSetTransform(rs);	
	Rule r = rst.new Rule() {
		public Object transform(int index, Object v) {
			return "transformed[col("+index+")]: " + v;
			}
		};
	
	rst.addColumnTransform(colname, r);
	QueryUtil.printResultSetTable(rst, System.out);
	}
	
//== WRAPPED AND OVERRIDDEN =======================================

public Object getObject(int columnIndex) throws SQLException
	{
	Integer key =  new Integer(columnIndex);
	return applyRule(key, rs.getObject(columnIndex));
	}
      
public Object getObject(String columnName) throws SQLException
	{
	Integer key = (Integer) ColNameIndex.get(columnName); //null ok, will passthru
	return applyRule(key, rs.getObject(columnName));
	}	

public String getString(int columnIndex) throws SQLException 
	{ 
	Integer key =  new Integer(columnIndex);
	//cast works for null values too
	String result = (String) applyRule(key, rs.getString(columnIndex));
	return result; 
	}
	
public String getString(String columnName) throws SQLException 
	{
	//cast works for null values too
	Integer key = (Integer) ColNameIndex.get(columnName); //null ok, will passthru
	String result = (String) applyRule(key, rs.getString(columnName));
	return result;
	}


//== WRAPPED AND PASSTHRU ========================================

public boolean getBoolean(int columnIndex) throws SQLException { return rs.getBoolean(columnIndex); }
public byte getByte(int columnIndex) throws SQLException { return rs.getByte(columnIndex); }
public short getShort(int columnIndex) throws SQLException { return rs.getShort(columnIndex); }
public int getInt(int columnIndex) throws SQLException { return rs.getInt(columnIndex); }
public long getLong(int columnIndex) throws SQLException { return rs.getLong(columnIndex); }
public float getFloat(int columnIndex) throws SQLException { return rs.getFloat(columnIndex); }
public double getDouble(int columnIndex) throws SQLException { return rs.getDouble(columnIndex); }
public BigDecimal getBigDecimal(int columnIndex, int scale) throws SQLException { return rs.getBigDecimal(columnIndex, scale); }
public byte[] getBytes(int columnIndex) throws SQLException { return rs.getBytes(columnIndex); }
public java.sql.Date getDate(int columnIndex) throws SQLException { return rs.getDate(columnIndex); }
public java.sql.Time getTime(int columnIndex) throws SQLException { return rs.getTime(columnIndex); }
public java.sql.Timestamp getTimestamp(int columnIndex) throws SQLException { return rs.getTimestamp(columnIndex); }
public java.io.InputStream getAsciiStream(int columnIndex) throws SQLException { return rs.getAsciiStream(columnIndex); }
public java.io.InputStream getUnicodeStream(int columnIndex) throws SQLException { return rs.getUnicodeStream(columnIndex); }
public java.io.InputStream getBinaryStream(int columnIndex) throws SQLException { return rs.getBinaryStream(columnIndex); }
//======================================================================
// Methods for accessing results by column name
//======================================================================
public boolean getBoolean(String columnName) throws SQLException { return rs.getBoolean(columnName); }
public byte getByte(String columnName) throws SQLException { return rs.getByte(columnName); }
public short getShort(String columnName) throws SQLException { return rs.getShort(columnName); }
public int getInt(String columnName) throws SQLException { return rs.getInt(columnName); }
public long getLong(String columnName) throws SQLException { return rs.getLong(columnName); }
public float getFloat(String columnName) throws SQLException { return rs.getFloat(columnName); }
public double getDouble(String columnName) throws SQLException { return rs.getDouble(columnName); }
public BigDecimal getBigDecimal(String columnName, int scale) throws SQLException { return rs.getBigDecimal(columnName, scale); }
public byte[] getBytes(String columnName) throws SQLException { return rs.getBytes(columnName); }
public java.sql.Date getDate(String columnName) throws SQLException { return rs.getDate(columnName); }
public java.sql.Time getTime(String columnName) throws SQLException { return rs.getTime(columnName); }
public java.sql.Timestamp getTimestamp(String columnName) throws SQLException { return rs.getTimestamp(columnName); }
public java.io.InputStream getAsciiStream(String columnName) throws SQLException { return rs.getAsciiStream(columnName); }
public java.io.InputStream getUnicodeStream(String columnName) throws SQLException { return rs.getUnicodeStream(columnName); }
public java.io.InputStream getBinaryStream(String columnName) throws SQLException { return rs.getBinaryStream(columnName); }
public boolean next() throws SQLException { return rs.next(); }
public void close() throws SQLException { rs.close(); }
public boolean wasNull() throws SQLException { return rs.wasNull(); }
//=====================================================================
// Advanced features:
//=====================================================================
public SQLWarning getWarnings() throws SQLException { return rs.getWarnings(); }
public void clearWarnings() throws SQLException { rs.clearWarnings(); }
public String getCursorName() throws SQLException { return rs.getCursorName(); }
public ResultSetMetaData getMetaData() throws SQLException { return rs.getMetaData(); }
//----------------------------------------------------------------
public int findColumn(String columnName) throws SQLException { return rs.findColumn(columnName); }
//--------------------------JDBC 2.0-----------------------------------
//---------------------------------------------------------------------
// Getters and Setters
//---------------------------------------------------------------------
public java.io.Reader getCharacterStream(int columnIndex) throws SQLException { return rs.getCharacterStream(columnIndex); }
public java.io.Reader getCharacterStream(String columnName) throws SQLException { return rs.getCharacterStream(columnName); }
public BigDecimal getBigDecimal(int columnIndex) throws SQLException { return rs.getBigDecimal(columnIndex); }
public BigDecimal getBigDecimal(String columnName) throws SQLException { return rs.getBigDecimal(columnName); }
//---------------------------------------------------------------------
// Traversal/Positioning
//---------------------------------------------------------------------
public boolean isBeforeFirst() throws SQLException { return rs.isBeforeFirst(); }
public boolean isAfterLast() throws SQLException { return rs.isAfterLast(); }
public boolean isFirst() throws SQLException { return rs.isFirst(); }
public boolean isLast() throws SQLException { return rs.isLast(); }
public void beforeFirst() throws SQLException { rs.beforeFirst(); }
public void afterLast() throws SQLException { rs.afterLast(); }
public boolean first() throws SQLException { return rs.first(); }
public boolean last() throws SQLException { return rs.last(); }
public int getRow() throws SQLException { return rs.getRow(); }
public boolean absolute( int row ) throws SQLException { return rs.absolute(row); }
public boolean relative( int rows ) throws SQLException { return rs.relative(rows); }
public boolean previous() throws SQLException { return rs.previous(); }
//---------------------------------------------------------------------
// Properties
//---------------------------------------------------------------------
public void setFetchDirection(int direction) throws SQLException { rs.setFetchDirection(direction); }
public int getFetchDirection() throws SQLException { return rs.getFetchDirection(); }
public void setFetchSize(int rows) throws SQLException { rs.setFetchSize(rows); }
public int getFetchSize() throws SQLException { return rs.getFetchSize(); }
public int getType() throws SQLException { return rs.getType(); }
public int getConcurrency() throws SQLException { return rs.getConcurrency(); }
//---------------------------------------------------------------------
// Updates
//---------------------------------------------------------------------
public boolean rowUpdated() throws SQLException { return rs.rowUpdated(); }
public boolean rowInserted() throws SQLException { return rs.rowInserted(); }
public boolean rowDeleted() throws SQLException { return rs.rowDeleted(); }
public void updateNull(int columnIndex) throws SQLException { rs.updateNull(columnIndex); }  
public void updateBoolean(int columnIndex, boolean x) throws SQLException { rs.updateBoolean(columnIndex, x); }
public void updateByte(int columnIndex, byte x) throws SQLException { rs.updateByte(columnIndex, x); }
public void updateShort(int columnIndex, short x) throws SQLException { rs.updateShort(columnIndex, x); }
public void updateInt(int columnIndex, int x) throws SQLException { rs.updateInt(columnIndex, x); }
public void updateLong(int columnIndex, long x) throws SQLException { rs.updateLong(columnIndex, x); }
public void updateFloat(int columnIndex, float x) throws SQLException { rs.updateFloat(columnIndex, x); }
public void updateDouble(int columnIndex, double x) throws SQLException { rs.updateDouble(columnIndex, x); }
public void updateBigDecimal(int columnIndex, BigDecimal x) throws SQLException { rs.updateBigDecimal(columnIndex, x); }
public void updateString(int columnIndex, String x) throws SQLException { rs.updateString(columnIndex, x); }
public void updateBytes(int columnIndex, byte x[]) throws SQLException { rs.updateBytes(columnIndex, x); }
public void updateDate(int columnIndex, java.sql.Date x) throws SQLException { rs.updateDate(columnIndex, x); }
public void updateTime(int columnIndex, java.sql.Time x) throws SQLException { rs.updateTime(columnIndex, x); }
public void updateTimestamp(int columnIndex, java.sql.Timestamp x) throws SQLException { rs.updateTimestamp(columnIndex, x); }
public void updateAsciiStream(int columnIndex, java.io.InputStream x, int length) throws SQLException { rs.updateAsciiStream(columnIndex, x, length); }
public void updateBinaryStream(int columnIndex, java.io.InputStream x,int length) throws SQLException { rs.updateBinaryStream(columnIndex, x, length); }
public void updateCharacterStream(int columnIndex,java.io.Reader x,int length) throws SQLException { rs.updateCharacterStream(columnIndex, x, length); }
public void updateObject(int columnIndex, Object x, int scale) throws SQLException { rs.updateObject(columnIndex, x, scale); }
public void updateObject(int columnIndex, Object x) throws SQLException { rs.updateObject(columnIndex, x); }
public void updateNull(String columnName) throws SQLException { rs.updateNull(columnName); }  
public void updateBoolean(String columnName, boolean x) throws SQLException { rs.updateBoolean(columnName, x); }
public void updateByte(String columnName, byte x) throws SQLException { rs.updateByte(columnName, x); }
public void updateShort(String columnName, short x) throws SQLException { rs.updateShort(columnName, x); }
public void updateInt(String columnName, int x) throws SQLException { rs.updateInt(columnName, x); }
public void updateLong(String columnName, long x) throws SQLException { rs.updateLong(columnName, x); }
public void updateFloat(String columnName, float x) throws SQLException { rs.updateFloat(columnName, x); }
public void updateDouble(String columnName, double x) throws SQLException { rs.updateDouble(columnName, x); }
public void updateBigDecimal(String columnName, BigDecimal x) throws SQLException { rs.updateBigDecimal(columnName, x); }
public void updateString(String columnName, String x) throws SQLException { rs.updateString(columnName, x); }
public void updateBytes(String columnName, byte x[]) throws SQLException { rs.updateBytes(columnName, x); }
public void updateDate(String columnName, java.sql.Date x) throws SQLException { rs.updateDate(columnName, x); }
public void updateTime(String columnName, java.sql.Time x) throws SQLException { rs.updateTime(columnName, x); }
public void updateTimestamp(String columnName, java.sql.Timestamp x) throws SQLException { rs.updateTimestamp(columnName, x); }
public void updateAsciiStream(String columnName, java.io.InputStream x, int length) throws SQLException { rs.updateAsciiStream(columnName, x, length); }
public void updateBinaryStream(String columnName, java.io.InputStream x,int length) throws SQLException { rs.updateBinaryStream(columnName, x, length); }
public void updateCharacterStream(String columnName, java.io.Reader reader,int length) throws SQLException { rs.updateCharacterStream(columnName, reader, length); }
public void updateObject(String columnName, Object x, int scale) throws SQLException { rs.updateObject(columnName, x, scale); }
public void updateObject(String columnName, Object x) throws SQLException { rs.updateObject(columnName, x); }
public void insertRow() throws SQLException { rs.insertRow(); }
public void updateRow() throws SQLException { rs.updateRow(); }
public void deleteRow() throws SQLException { rs.deleteRow(); }
public void refreshRow() throws SQLException { rs.refreshRow(); }
public void cancelRowUpdates() throws SQLException { rs.cancelRowUpdates(); }
public void moveToInsertRow() throws SQLException { rs.moveToInsertRow(); }
public void moveToCurrentRow() throws SQLException { rs.moveToCurrentRow(); }
public Statement getStatement() throws SQLException { return rs.getStatement(); }
public Object getObject(int i, java.util.Map map) throws SQLException { return rs.getObject(i, map); }
public Ref getRef(int i) throws SQLException { return rs.getRef(i); }
public Blob getBlob(int i) throws SQLException { return rs.getBlob(i); }
public Clob getClob(int i) throws SQLException { return rs.getClob(i); }
public Array getArray(int i) throws SQLException { return rs.getArray(i); }
public Object getObject(String colName, java.util.Map map) throws SQLException { return rs.getObject(colName, map); }
public Ref getRef(String colName) throws SQLException { return rs.getRef(colName); }
public Blob getBlob(String colName) throws SQLException { return rs.getBlob(colName); }
public Clob getClob(String colName) throws SQLException { return rs.getClob(colName); }
public Array getArray(String colName) throws SQLException { return rs.getArray(colName); }
public java.sql.Date getDate(int columnIndex, Calendar cal) throws SQLException { return rs.getDate(columnIndex, cal); }
public java.sql.Date getDate(String columnName, Calendar cal) throws SQLException { return rs.getDate(columnName, cal); }
public java.sql.Time getTime(int columnIndex, Calendar cal) throws SQLException { return rs.getTime(columnIndex, cal); }
public java.sql.Time getTime(String columnName, Calendar cal) throws SQLException { return rs.getTime(columnName, cal); }
public java.sql.Timestamp getTimestamp(int columnIndex, Calendar cal) throws SQLException { return rs.getTimestamp(columnIndex, cal); }
public java.sql.Timestamp getTimestamp(String columnName, Calendar cal) throws SQLException { return rs.getTimestamp(columnName, cal); }
//-------------------------- JDBC 3.0 ----------------------------------------
public java.net.URL getURL(int columnIndex) throws SQLException { return rs.getURL(columnIndex); }
public java.net.URL getURL(String columnName) throws SQLException { return rs.getURL(columnName); }
public void updateRef(int columnIndex, java.sql.Ref x) throws SQLException { rs.updateRef(columnIndex, x); }
public void updateRef(String columnName, java.sql.Ref x) throws SQLException { rs.updateRef(columnName, x); }
public void updateBlob(int columnIndex, java.sql.Blob x) throws SQLException { rs.updateBlob(columnIndex, x); }
public void updateBlob(String columnName, java.sql.Blob x) throws SQLException { rs.updateBlob(columnName, x); }
public void updateClob(int columnIndex, java.sql.Clob x) throws SQLException { rs.updateClob(columnIndex, x); }
public void updateClob(String columnName, java.sql.Clob x) throws SQLException { rs.updateClob(columnName, x); }
public void updateArray(int columnIndex, java.sql.Array x) throws SQLException { rs.updateArray(columnIndex, x); }
public void updateArray(String columnName, java.sql.Array x) throws SQLException { rs.updateArray(columnName, x); }
//-------------------------- JDBC 4.0 ----------------------------------------
public void updateNClob(String columnName, java.io.Reader r) throws SQLException { rs.updateNClob(columnName, r); }
public void updateNClob(int columnIndex, java.io.Reader r) throws SQLException { 	rs.updateNClob(columnIndex, r); }
public void updateNClob(String columnName, java.io.Reader r, long l) throws SQLException { rs.updateNClob(columnName, r, l); }
public void updateNClob(int columnIndex, java.io.Reader r, long l) throws SQLException { 	rs.updateNClob(columnIndex, r, l); }
public void updateNClob(String columnName, NClob x) throws SQLException { rs.updateNClob(columnName, x); }
public void updateNClob(int columnIndex, NClob x) throws SQLException { 	rs.updateNClob(columnIndex, x); }
public void updateNString(int columnIndex, String s) throws SQLException { rs.updateNString(columnIndex, s); }
public void updateNString(String columnLabel, String s) throws SQLException { rs.updateNString(columnLabel, s); }
public void updateClob(String columnName, java.io.Reader r) throws SQLException { rs.updateClob(columnName, r); }
public void updateClob(int columnIndex, java.io.Reader r, long l) throws SQLException { rs.updateClob(columnIndex, r); }
public void updateClob(String columnName, java.io.Reader r, long l) throws SQLException { rs.updateClob(columnName, r, l); }
public void updateClob(int columnIndex, java.io.Reader r) throws SQLException { rs.updateClob(columnIndex, r); }
public void updateBlob(int columnIndex, java.io.InputStream x) throws SQLException { rs.updateBlob(columnIndex, x); }
public void updateBlob(String columnName, java.io.InputStream x) throws SQLException { rs.updateBlob(columnName, x); }
public void updateBlob(int columnIndex, java.io.InputStream x, long l) throws SQLException { rs.updateBlob(columnIndex, x, l); }
public void updateBlob(String columnName, java.io.InputStream x, long l) throws SQLException { rs.updateBlob(columnName, x, l); }
public void updateCharacterStream(String columnName, java.io.Reader x) throws SQLException { rs.updateCharacterStream(columnName, x); }
public void updateCharacterStream(int columnIndex, java.io.Reader x) throws SQLException { rs.updateCharacterStream(columnIndex, x); }
public void updateCharacterStream(String columnName, java.io.Reader x, long l) throws SQLException { rs.updateCharacterStream(columnName, x, l); }
public void updateCharacterStream(int columnIndex, java.io.Reader x, long l) throws SQLException { rs.updateCharacterStream(columnIndex, x, l); }
public void updateNCharacterStream(String columnName, java.io.Reader x) throws SQLException { rs.updateNCharacterStream(columnName, x); }
public void updateNCharacterStream(int columnIndex, java.io.Reader x) throws SQLException { rs.updateNCharacterStream(columnIndex, x); }
public void updateNCharacterStream(String columnName, java.io.Reader x, long l) throws SQLException { rs.updateNCharacterStream(columnName, x, l); }
public void updateNCharacterStream(int columnIndex, java.io.Reader x, long l) throws SQLException { rs.updateNCharacterStream(columnIndex, x, l); }
public void updateBinaryStream(String columnName, java.io.InputStream x) throws SQLException { rs.updateBinaryStream(columnName, x); }
public void updateBinaryStream(int columnIndex, java.io.InputStream x) throws SQLException { rs.updateBinaryStream(columnIndex, x); }
public void updateBinaryStream(String columnName, java.io.InputStream x, long l) throws SQLException { rs.updateBinaryStream(columnName, x, l); }
public void updateBinaryStream(int columnIndex, java.io.InputStream x, long l) throws SQLException { rs.updateBinaryStream(columnIndex, x, l); }
public void updateAsciiStream(String columnName, java.io.InputStream x) throws SQLException { rs.updateAsciiStream(columnName, x); }
public void updateAsciiStream(int columnIndex, java.io.InputStream x) throws SQLException { rs.updateAsciiStream(columnIndex, x); }
public void updateAsciiStream(String columnName, java.io.InputStream x, long l) throws SQLException { rs.updateAsciiStream(columnName, x, l); }
public void updateAsciiStream(int columnIndex, java.io.InputStream x, long l) throws SQLException { rs.updateAsciiStream(columnIndex, x, l); }
public void updateRowId(int columnIndex, RowId x) throws SQLException { rs.updateRowId(columnIndex, x); }
public void updateRowId(String columnLabel, RowId x) throws SQLException { rs.updateRowId(columnLabel, x); }
public void updateSQLXML(int columnIndex, SQLXML x) throws SQLException { rs.updateSQLXML(columnIndex, x); }
public void updateSQLXML(String columnLabel, SQLXML x) throws SQLException { rs.updateSQLXML(columnLabel, x); }
public int getHoldability() throws SQLException { return rs.getHoldability(); }
public RowId getRowId(int columnIndex) throws SQLException { return  rs.getRowId(columnIndex); }
public RowId getRowId(String columnLabel) throws SQLException { return  rs.getRowId(columnLabel); }
public NClob getNClob(int columnIndex) throws SQLException { return rs.getNClob(columnIndex); }
public NClob getNClob(String columnLabel) throws SQLException { return rs.getNClob(columnLabel); }
public String getNString(int columnIndex) throws SQLException { return rs.getNString(columnIndex); }
public String getNString(String columnLabel) throws SQLException { return rs.getNString(columnLabel); }
public Reader getNCharacterStream(int columnIndex) throws SQLException { return rs.getNCharacterStream(columnIndex); }
public Reader getNCharacterStream(String columnLabel) throws SQLException { return rs.getNCharacterStream(columnLabel); }
public SQLXML getSQLXML(int columnIndex) throws SQLException { return rs.getSQLXML(columnIndex); }
public SQLXML getSQLXML(String columnLabel) throws SQLException { return rs.getSQLXML(columnLabel); }             
public boolean isClosed()throws SQLException { return rs.isClosed(); }
public boolean isWrapperFor(Class iface) throws SQLException { return rs.isWrapperFor(iface); }
public Object unwrap(Class iface) throws SQLException { return rs.unwrap(iface); }

//== 1.7+ ===============================================
public Object getObject(String str, Class c) throws SQLException {
	return rs.getObject(str, c);
	}
public Object getObject(int i, Class c) throws SQLException {
	return rs.getObject(i, c);
	}

//== END WRAPPED ===============================================
}
