// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc;

import java.sql.*;
import java.io.*;
import java.util.*;
import java.util.concurrent.*;

import fc.io.*;
import fc.util.*;

//import EDU.oswego.cs.dl.util.concurrent.*;

/**  
A pooled connection manager. Connections must be closed when done
with, which automatically returns them back to the pool. 

@author hursh jain
**/
public final class PooledConnectionMgr extends ConnectionMgr
{
/*
we don't really have any way to get all connections in
the buffer and close them all - the bounderbuffer class
doesn't allow this. So we need this class
*/
class MyBoundedBuffer extends ArrayBlockingQueue
	{
	int bufsize;
	
	MyBoundedBuffer(int size) {
		super(size);
		this.bufsize = size;
		}
		
	//closes all connections, even ones that are checked out
	void closeAllConnections() 
		{
		Object[] items = toArray();
		for (int n = 0; n < items.length; n++)
			{
			try {
				PooledConnection con = (PooledConnection) items[n];
				if (con != null) {
					if (! con.isClosed() ) {  
						log.warn("PooledConnectionMgr shutdown: closing open database connection #" + con.id);
						}
					con.closeReally();
					}
				}
			catch (SQLException e)
				{
				log.warn(e);
				}
			}
		}
}

public static final int defaultSize = 100;

MyBoundedBuffer 	buffer;
boolean 			dbgCheckOut;
Map					checkedOutInfo; //useful for debugging pool leaks

/** If a connection cannot be obtained within this time,
then a warning will be logged. The value is in milliseconds.
*/
public static final long ConnectionTimeoutWarning = 1500;

/** 
Constructs a new connection pool with the specified size.
{@see ConnectionMgr} 
**/
public PooledConnectionMgr(
		String jdbc_url, String jdbc_driver, 
		String jdbc_user, String jdbc_password, 
		String jdbc_catalog,
		int size) 
throws Exception
	{
	super(jdbc_url, jdbc_driver, jdbc_user, jdbc_password, jdbc_catalog);
	init(size);
	}

/** 
Constructs a new connection pool with the specified size. The
size is the number of connections that this pool holds.
**/
public PooledConnectionMgr(PropertyMgr props, int size) 
throws Exception
	{ 
	super(props);
	init(size);
	}

/** 
Constructs a new connection pool with the specified size. The
size is the number of connections that this pool holds.
**/
public PooledConnectionMgr(PropertyMgr props, String prefix, int size) 
throws Exception
	{ 
	super(props, prefix);
	init(size);
	}
	
/** 
Constructs a new connection pool with the specified size. The
size is the number of connections that this pool holds. 
**/
public PooledConnectionMgr(
		SystemLog log, PropertyMgr props, int size) 
throws Exception
	{ 
	super(log, props);
	init(size);
	}

/** 
Constructs a new connection pool with the specified size. The
size is the number of connections that this pool holds. 
**/
public PooledConnectionMgr(
		SystemLog log, PropertyMgr props, String prefix, int size) 
throws Exception
	{ 
	super(log, props, prefix);
	init(size);
	}
	
private void init(int size) throws Exception
	{
	if (size <= 0) {
		throw new IllegalArgumentException("size must be greater than 0");
		}
	
	buffer = new MyBoundedBuffer(size);
	log.bug("PooledConnectionMgr.constructor: Constructed new connection pool buffer: ", buffer);

	for (int n = 0; n < size; n++) 
		{
		Connection con = DriverManager.getConnection(url,user,password);
		if (con == null) {
			throw new Exception("PooledConnectionMgr: Cannot connect to database [" + url + "]");
			}
		PooledConnection pooled_con = new PooledConnection(this, con);
		buffer.put(pooled_con);
		}

	checkedOutInfo = new Hashtable(); 
	log.bug("PooledConnectionMgr.constructor: added ", buffer.size(), " connections to the connection pool");
	}

/** 
Helps to debug connection checkouts. By default, this is 
<tt>false</tt> (since there is a stack trace overhead per
connection when debugging is enabled).

@param	dbg 	<tt>true</tt> to enable tracking of checked out
				connections from this pool. <tt>false</tt>
				stops tracking and clears any existing tracking
				info.
**/
public void setDebugCheckout(boolean dbg) {
	dbgCheckOut = dbg;
	if (! dbg) {
		checkedOutInfo.clear();
		}
	}

/** 
Returns a iterator over a collection of Exception objects, each
containing a stack trace for code that has currently checked out
a connection from this pool. For this to work, make sure that
{@link #setDebugCheckout} has first been invoked with 
<tt>true</tt>.
**/
public Iterator getCheckoutInfo() {
	return checkedOutInfo.values().iterator();
	}

/**
Gets a list of connections in the pool such that each connection has a
transaction that was started more than the specified milliseconds ago but
has not yet been aborted or commited (this is a good way to see if there
is a transaction leak somewhere).
*/
public List getHungTransactions(long milliseconds) 
	{
	long now = System.currentTimeMillis();
	Object[] items = buffer.toArray();
	List list = new ArrayList();
	for (int n = 0; n < items.length; n++) {
		PooledConnection con = (PooledConnection) items[n];
		if (con.transaction_start_time == 0 
			|| (con.transaction_start_time + milliseconds) > now) {
			continue;
			}
		list.add(con);
		}
	return list;
	}

/** 
Returns a connection from the pool possibly waiting
indefinitely until a connection is available. However, if a
connection is not obtained in {@link
#ConnectionTimeoutWarning} milliseconds, then a warning is
logged.
**/
protected Connection getConImpl() throws SQLException 
	{
	try {
		PooledConnection con = null;
		con = (PooledConnection) buffer.poll(ConnectionTimeoutWarning, 
												TimeUnit.MILLISECONDS);
	
		if (con == null) 
			{
			log.warn(this, 
				"Could not obtain connection in ", 
				String.valueOf(ConnectionTimeoutWarning), 
				" milliseconds..\n", 
				"Possibly waiting forever for connection here...\n",
				IOUtil.throwableToString(new Exception("Debug Stack Trace"))
				);
	
			con = (PooledConnection) buffer.take();
			log.info(this, "...Connection obtained successfully.");
			}
			
		con.setClosed(false);
		
		if (dbgCheckOut) {
			checkedOutInfo.put(new Integer(con.getID()), new Exception("==PooledConnectionMgr/Checkout Debug Info=="));
			}
			
		return con;
		}
	catch (InterruptedException e) {
		e.printStackTrace();
		throw new SQLException(e.getMessage());
		}
	}

protected boolean handleMgrShutdown() {
	buffer.closeAllConnections();
	return true;
 	}	
	
void put(PooledConnection con) throws SQLException
	{
	//we know this won't block because our pool only gets stuff
	//put in it that was previously taken from it
	try {	
	  	buffer.put(con);
		if (dbgCheckOut) {
			checkedOutInfo.remove( new Integer(con.getID()) );
			}
		}
	catch (InterruptedException e) 
		{
		e.printStackTrace();
		throw new SQLException(e.getMessage());
		}
	}

public String toString() {
	return myname + " [capacity=" + buffer.bufsize + "; available=" + buffer.size() + "]";
	}
	
//== testing =================================================	
	
public static void main(String[] args) throws Exception
	{
	Args myargs = new Args(args);
	Log.getDefault().setLevel(SystemLog.DEBUG);
	myargs.setUsage("java fc.jdbc.PooledConnectionMgr -conf conf-file -query query-to-execute");
	
	String query  	= myargs.getRequired("query");
	String propfile = myargs.getRequired("conf");
	final int size = 5;
	
	//test 1
	FilePropertyMgr fmgr = new FilePropertyMgr(new File(propfile));

	PooledConnectionMgr pool = new PooledConnectionMgr(fmgr, size);

	pool.setDebugCheckout(true);
	
	List list = new ArrayList();
	for (int n = 0; n < size/2; n++) {
		Connection con = pool.getConnection();
		list.add(con);	
		}
		
	System.out.println("Got connections, connection pool="+pool);
	System.out.println("Now using connections...");
	
	int num = 1;
	for (Iterator it = list.iterator(); it.hasNext(); /*empty*/) {
		Connection item = (Connection) it.next();
		Statement stmt = item.createStatement();
		ResultSet rs = stmt.executeQuery(query);
		System.out.println("=== query using connection [# " + num++ + "]" + item + "=====");
		QueryUtil.printRS(rs);
		System.out.println("=============================================================");
		item.close();		
		}	
	
	System.out.println("Used and closed all connections in pool.");
	Connection con = pool.getConnection();
	System.out.println("Got new connection:" + con);
	
	System.out.println("==== [debug] Checkout Info: ====");
	for (Iterator it = pool.getCheckoutInfo(); it.hasNext(); /*empty*/) {
		Throwable item = (Throwable) it.next();
		item.printStackTrace();
		}	
	System.out.println("==== [debug end] Checkout Info: ====");
			
	System.out.println("Closed all connections, connection pool="+pool);	
	
	pool.close();
	
	System.out.println("Trying to get connection after closing connection mgr");
	try {
		con = pool.getConnection();
		}
	catch (Exception e) {
		e.printStackTrace();		
		}
		
	//test 2
	System.out.println("TEST 2....");
	
	final PooledConnectionMgr pool2 = 
					new PooledConnectionMgr(fmgr, size);

	Thread t = new Thread() 
		{
		public void run() 
			{
			try {
				List list = new ArrayList();
				for (int n = 0; n < size; n++) {
					Connection con = pool2.getConnection();
					list.add(con);	
					System.out.println("Got connection: " + con);
					}
				System.out.println("sleeping 10 seconds");
				Thread.currentThread().sleep(5000);
				for (int n = 0; n < list.size(); n++)
					{
					System.out.println(
						"Releasing connection: " + 
						 (n+1) + " " + list.get(n) );
					((Connection) list.get(n)).close();
					}
				}
			catch (Exception e) {
				e.printStackTrace();
				}
			}
		};
		
	t.start();
	
	for (int n = 0; n < size; n++) {
		System.out.println("Main Thread: Trying to get connection..." + (n+1));
		con = pool2.getConnection();
		System.out.println("Main Thread: Got connection: " + con);
		con.close();
		}

	pool2.close();	
	} //main
}          //~class PooledConnectionMgr