// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import fc.app.*;
import java.io.*;
import EDU.oswego.cs.dl.util.concurrent.*;

/**
Manages command line input/output. The default action is to read a line
and echo it back to the command line. Subclasses should do something more
useful by overriding the <tt>run</tt> method.

@author hursh jain	
@version 1.0 12/15/2001
**/
public class CmdLine implements Runnable
{
public static final String DEFAULT_PROMPT = "JavaCMD> ";
public static final String DEFAULT_INPUT_PROMPT = "Enter> ";
public static final String DEFAULT_QUIT_COMMAND = "exit";

static final int PROMPT_NORMAL = 0; //show normal prompt
static final int PROMPT_ASK = 1; //show PS2

BufferedReader in; 
PrintWriter out;
PrintWriter err;

final String version = "Command Line v1.0";
volatile protected String 	prompt = DEFAULT_PROMPT;
volatile protected String 	inputPrompt = DEFAULT_INPUT_PROMPT;
volatile protected int		promptState = PROMPT_NORMAL;

volatile protected String 	quitCmd = DEFAULT_QUIT_COMMAND;
volatile protected boolean 	finished = false; //set to true to stop Cmd Line
volatile protected boolean 	localecho = false;

Object start_lock = new Object();
protected String line = null; //the current input line
Semaphore nextLineHasBeenRead; //signals a new line

/**
Constructs the object using the specified input and output streams.
@param in	the BufferedReader to use for input
@param out	the PrintWriter to use for output
@param err	the PrintWriter to use for errors
**/
public CmdLine(BufferedReader in, PrintWriter out, PrintWriter err)
	{
	Argcheck.notnull(in); 
	Argcheck.notnull(out); 
	Argcheck.notnull(err);
	this.in = in;
	this.out = out;
	this.err = err;
	try {
		nextLineHasBeenRead = new Semaphore(1);
		nextLineHasBeenRead.acquire();
		}
	catch (Exception e) { e.printStackTrace();}
	}

/**
Constructs a default object, using <tt>System.in, System.out</tt>
for the input, output and error streams.
**/
public CmdLine()
	{
	this(	
		new BufferedReader(new InputStreamReader(System.in)), 
		new PrintWriter(System.out), 
		new PrintWriter(System.err)
		);
	}

/**
Sets the prompt to show the user in the input stream. Set to
<tt>null</tt> for no prompt. By default, the <tt>CmdLine.DEFAULT_PROMPT</tt>
is shown.
@param 	prompt	the prompt string 
@see	#DefaultPrompt
**/
public void setPrompt(String prompt)
	{
	this.prompt = prompt;
	}

public String getPrompt() 
	{
	if (promptState == PROMPT_NORMAL)
		return prompt;
	else if (promptState == PROMPT_ASK) {
		promptState = PROMPT_NORMAL;   //hack. all of this needs to be rewritten at some point
		return inputPrompt;
		}
	else 
		throw new IllegalStateException("CmdLine.getPrompt(): illegal prompt state");
	}

/**
Sets the input prompt to show the user when user input is expected. This
is similar to PS2 in bash. Set to <tt>null</tt> for no prompt. By 
default, the <tt>CmdLine.DEFAULT_INPUT_PROMPT</tt>is shown.
@param 	prompt	the prompt string 
@see	#DefaultPrompt
**/
public void setInputPrompt(String prompt)
	{
	this.inputPrompt = prompt;
	}

/**
Sets the default command to quit the CmdLine. It is upto the actual
implementation to use this, if needed.
@param 	quitstring	the string that needs to be typed in to quit the Cmd Line.
**/
public void setQuitCmd(String quitstring)
	{
	Argcheck.notnull(quitstring);
	this.quitCmd = quitstring;
	}

/** starts the command line **/
public void start()
	{
	new Thread(this, version).start();
	synchronized (start_lock) 
		{
		try {
			start_lock.wait();
			}
		catch (Exception e) {
			e.printStackTrace();
			}
		}
	}

/**
Shows the specified string to the user shows the input prompt
to the user (to indicate the user's input is expected). Keeps
looping and asking the same question until a non-empty response
has been typed by the user, then returns that response.
*/
public String ask(String str) 
	{
	String result = "";
	
	while (result.equals("")) {
		out.println(str); //the string
		out.flush();
		promptState = PROMPT_ASK; //prompt for next line

		try {
			result = getNextLine();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
		}

	return result;
	}

	
/**
Immediately writes a string to the output. The prompt is written
before the string.
*/
public void write(String str) 
	{
	out.print(str);
	out.flush();	
	}

/**
Immediately writes the specified string as a new line, to the output.
The prompt is written before the string.
*/
public void writeln(String str) 
	{
	out.print(this.prompt);	
	out.println(str);
	out.flush();	
	}

/**
Immediately writes the specified string to the output. The prompt
is not written.
**/
public void writeWithoutPrompt(String str)
	{
	out.print(str);
	out.flush();	
	}

/**
Immediately writes the specified string as a new line. The prompt
is not written.
**/
public void writelnWithoutPrompt(String str)
	{
	out.println(str);
	out.flush();	
	}


/**
Turns the local echo on or off. This can be honored by some
implementations of the {@link #handleInput] method to echo
what was typed in.
@boolean	val		<tt>true</tt> to turn local echo on
**/
public void setLocalEcho(boolean val) {
	localecho = val;
	}


/**
Returns <tt>true</tt> if the command line is still running,
<tt>false</tt> otherwise.
**/
public boolean isRunning() {
	return ! finished;
	}


/**
A programmatic way to stop the command line.
*/	
public void stop()
	{
	this.finished = true;
	}

/**
Gets the latest input line that has been read. Can return the
same line multiple times if no new lines have been read since
the last time this method was called. Returns <tt>null</tt>
if no line has been read so far.
**/
public String getLastLine() {
	return line;
	}

/**
Returns the input line read *after* the last time this
method was called. If no input line has been read since
then, this method will block until a new input line has
been read. Throws a <tt>InterruptedException</tt> if
the thread calling this method was interrupted.
**/
public String getNextLine() throws InterruptedException
	{
	//calling thread block until the next line is read and this flag 
	//is released
	nextLineHasBeenRead.acquire();
	return line;
	}


/** 
The main command line loop. Calls the template methods 
{@link #readLine}, {@link #checkForExit} and {@link #handleInput}.
Any of these template methods can be overriden in subclasses, as 
needed. However, this method itself should <b>not</b> be 
overridden in subclasses.
**/
public void run() 
	{
	System.out.println("Starting " + this.version);
	
	//started 
	synchronized (start_lock) {
		start_lock.notifyAll();
		}
	
	while (! finished) 
		{	
		write(getPrompt()); 
		readLine();
		checkForExit();			
		//we only want to allow clients to call into this class
		//if the CmdLine is still running - checkForExit() determines
		//that and therefore has to come before releasing any flags
		nextLineHasBeenRead.release(); //we have a new line			
		handleInput();
		}
	try {
		//do NOT do this ! (otherwise System.in etc. are permanently
		//closed !
		//in.close();
		//out.close();
		//err.close();
		}
	catch (Exception e) {
		}
	}

/**
Sets the next line of input without the trailing newline
character. A line of input is signalled when a newline character
is encountered. This method will block until the input has a 
newline.<br>
Note: Unfortunately, as of JDK 1.4 there seems to be no way to
get the input buffer unless a newline is also entered. (Stream/Reader
methods like <tt>available</tt> and <tt>ready</tt> return 0/false
until a newline is entered). This means <tt>System.in</tt> always
appears to work in "cooked" mode (both on windows and an linux).
This sucks.
<br>
Note: This method is <tt>protected</tt>. It cannot be
called directly by clients but used by subclasses to read
the next line that was typed in. 
**/
protected void readLine() 
	{
	try {
	  	line = in.readLine();
		//System.out.println(">>>>[" + line + "]");
		} 
	catch(java.io.IOException e) {
		e.printStackTrace();
		}
	}

/**
Quits when the {@link #DEFAULT_QUIT_COMMAND} is encountered. This method
must call {@link #stop} to quit the command line (in the default
implementation of <tt>run</tt>)
**/
protected void checkForExit() {
	if (line.equalsIgnoreCase(quitCmd)) { 
		System.out.println("Stopping command line..");
		stop();
		}
	}

/**
The default implementation simply prints the line that was entered
if local echo is <tt>on</tt>, otherwise does nothing.
**/
protected void handleInput() {
	if (localecho)
		writeln("I read the line:'" + line + "'");
	}


public static void main(String args[]) throws Exception
	{
	new Test();	
	}

/**	
Unit Test History 	
<pre>
Class Version	Tester	Status		Notes
1.0				hj		passed		
</pre>
*/	
static private class Test 
{
Test() throws Exception
	{
	final CmdLine cline = new CmdLine();
	cline.setLocalEcho(true);
	cline.setPrompt("$> ");
	cline.start();
	cline.ask("what is your name ?");
	}
} //~Test



}           //~class CmdLine