// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import java.io.*;
import java.util.*;
import java.util.regex.*;

import fc.util.*;

/** 
Prints a table formatted using plain text/ascii characters.
Useful for console based output and/or formatted output to a
file.

@author hursh jain
**/
public class TablePrinter
	{
	private static final boolean dbg = false;
	private static final Object Extra_Content_Marker = new Object();
	
	String  		linesep = IOUtil.LINE_SEP;
	PrintConfig 	config;
	PrintWriter		out;
	int				columncount;	
	String  		rowborder;	
	int				cellwidth;
	String			spacing_str;
	String			padding_str;
	boolean			usingSpacing;		
	boolean			printingHeader;	 //printing a header line ?
	int				current_linenum; //not including headers
	int 			current_cellnum;
	String[]		wrappedcells;
	int				wrapcells_left;

	//use for max cell widths when using autofit
	int[]			largest_width;
	//used to cache table data when using autofit
	//list of String[]
	List			autoFitTableData = new ArrayList(); 
	
	int				currentAutoFitRow;
	int				currentAutoFitCell;
	
	
	/**
	Constructs a new table printer.
	
	@param	columnCount		the number of columns in the table.
							Attemps to print more than these
							number of columns will result in 
							a runtime exception.
	@param	pw				the destination print stream
	@param	config			the printing configuration
	**/
	public TablePrinter (int columnCount, PrintStream ps, PrintConfig config) 
		{
		this(columnCount, new PrintWriter(ps), config);	
		}

	/**
	Constructs a new table printer.
	
	@param	columnCount		the number of columns in the table.
							Attemps to print more than these
							number of columns will result in 
							a runtime exception.
	@param	pw				the destination print writer
	@param	config			the printing configuration
	**/
	public TablePrinter (int columnCount, PrintWriter pw, PrintConfig config) 
		{
		Argcheck.notnull(config, "config was null");
		Argcheck.notnull(pw, "printwriter/stream was null");
		this.columncount= columnCount;
		this.out = pw;
		this.config = config;
		cellwidth = config.cellwidth;	
		padding_str = StringUtil.repeatToWidth(
			config.cellpadding_glyph, config.cellpadding);
		
		spacing_str = StringUtil.repeatToWidth(
				config.cellspacing_glyph, config.cellspacing);

		usingSpacing = config.cellspacing > 0;
		printingHeader = false;
		wrappedcells = new String[columncount];
		initRowBorder();
		}


	/******************************************************* 	

	Layout rules:
	upper and lower cell borders look like:
	
	+-----+------+-----+------+
	|	  |		 |	   |	  |	 
	|	  |		 |	   |	  |
	+-----+------+-----+------+

	or:
	
	+-----+ +-----+ +-----+
	| 	  | |	  | |	  |
	|	  | |	  | |	  |
	+-----+ +-----+ +-----+

			
	We don't support cell specific borders and know the 
	number of columns in the table beforehand.
	
	0) startTable
		- prints the initial table header is applicable
	
	1) startRow - 
		- initializes some variables
		- if this is a new row, 
			- writes spacing
			- writes entire top border
	
	2) printcell 
		- if spacing
			- write spacing, left border, content, right border
		- no spacing
			- write left border, content		
		- tracks whether row contents need to be wrapped
		if wrapping needed, sets a flag ( we have to wait
		until entire row is written before writing a new
		row containing the wrapped contents )
	
	3) endRow
		- if no spacing 
			writes last most right border (not needed if we
			are using spacing, since in that case each cell
			writes it's right border)
		- write spacing (if using spacing)
		- if wrapping is needed, calls startRow again
		specifying that this is *not* a new row (and
		hence the new row border will not be written)
		- if wrapping not needed, and using spacing, writes
		the bottom border (needed to delimit rows with spacing)
		- write header row again if applicable (if header rows are 
		repeated every x number of lines)
			
	4) endTable
		- write border and spacing after the last row (we don't know
		beforehand the # of rows, so we wait until the last
		row is written before writing the ending spacing)
		
	********************************************************/
			
	/** This method should be called to start the table **/
	public void startTable() 
		{
		if (config.autoFit) {
			startTableAutoFit();
			return;
			}
		
		if (config.header != null) 
			printHeader();
		}
	
	/** This method should be called to finish the table **/
	public void endTable() 
		{
		if (config.autoFit) {
			endTableAutoFit();
			return;
			}
	
		if (! usingSpacing) {
			printRowBorder();
			}
			
		if (usingSpacing) {
			printVerticalSpacing();	
			}
			
		//if prinstream->printwriter, then unless we flush,
		//writes to underlying printstream may not appear
		//as expected (and the sequence can be screwy if
		//we are also writing to the underlying printstream
		//independently
		out.flush();
		}
	
	/** This method should be called to start a new row **/
	public void startRow() 
		{
		if (config.autoFit)  {
			startRowAutoFit();
			return;
			}

		current_cellnum = 0;
			
		//logical non header lines
		if (! printingHeader && rowFinished()) 
			current_linenum++;  //first line = 1
		
		if (rowFinished()) 
			{
			if (usingSpacing) {
				printVerticalSpacing();	
				}
			printRowBorder();
			}	
		}
		
	void endRowCommon() 
		{
		if (! usingSpacing) {
			if (config.printborder) {
				out.print(config.cellborder_v); //right most border
				}
			}
		else	
			out.print(spacing_str);	//right most spacing
		
		out.print(linesep);	
		}	
	
	/** This method should be called to finish the existing row **/
	public void endRow() 
		{	
		if (config.autoFit)  {
			endRowAutoFit();
			return;
			}
						
		endRowCommon();
				
		if (config.cellwrap) 
			{
			while (! rowFinished() ) 
				{
				startRow(); 
				for (int n = 0; n < columncount; n++) {
					printCell(wrappedcells[n]);
					}
				endRowCommon();  	
				}
			}		
	
		if (usingSpacing) {
			printRowBorder();		
			}
		
		if (! printingHeader && repeatHeader()) {
			printHeader();
			}
		}
		
	boolean repeatHeader() 
		{
		int pagesize = config.pageSize;
		
		if (pagesize <= 0 || ! config.headerEveryPage)
			return false;
		
		if (current_linenum == 0)
			return false;
		
		//System.out.println("current_linenum[" + current_linenum + "] % pagesize[" + pagesize + "] = " + (current_linenum % pagesize));
	
		if ((current_linenum % pagesize) == 0) {
			return true;
			}
			
		return false;
		}

	/** 
	This method should be called to print a new cell in the
	current row. This method should not be invoked for more
	columns than the table was instantiated with, otherwise a
	runtime exception will be thrown.
	**/
	public void printCell(String str) 
		{		
		if (current_cellnum == columncount)
			throw new IllegalArgumentException("Cannot add more cells than number of columns in this table. (max columns=" + columncount + ")");
		
		str = removeEmbeddedSpaces(str);
				
		if (config.autoFit)  {
			printCellAutoFit(str);
			return;
			}
	
		//left spacing
		if (usingSpacing) {
			out.print(spacing_str);
			}
				
		//left border
		if (config.printborder) {
			out.print(config.cellborder_v); 
			}

		printCellContent(str);
	
		//right border if spaced
		if (usingSpacing) {
			if (config.printborder) {
				out.print(config.cellborder_v); 
				}
			}		
	
		current_cellnum++;
		}
			
	public String toString() {
		return "TablePrinter. Config = " + config;
		}	
	
	//== impl. methods ===================================
					
	void printCellContent(String str) 
		{		
		int width = config.getCellWidthForColumn(current_cellnum);

		int strlen = (str == null) ? 0 : str.length();
		
		//align cell contents appropriately
		String content = StringUtil.fixedWidth(str, width, config.align);	
		
		out.print(padding_str);
		out.print(content);
		out.print(padding_str);
		
		if (dbg) System.out.println("current cell=" + current_cellnum);
	
		if (config.cellwrap)
			{
			if (strlen > width) 
				{
				//don't double count wrapped cells
				if (wrappedcells[current_cellnum] == null)
					wrapcells_left++;

				//this cell needs to be wrapped,save the extra portion
				wrappedcells[current_cellnum] = str.substring(width, strlen); 
				}
			else {
				if (wrappedcells[current_cellnum] != null)
					wrapcells_left--;
				wrappedcells[current_cellnum] = null;
				}
			}		
		if (dbg) System.out.println("Wrapped cells left=" + wrapcells_left + "; Wrapped:" + Arrays.asList(wrappedcells));
		}

	boolean rowFinished() {
		return wrapcells_left == 0;
		}

	void printHeader() 
		{
		if (config.header == null)
			return;
			
		printingHeader = true;
		startRow();
		for (int n = 0; n < config.header.length; n++)
			printCell(config.header[n]);
		endRow();
		printingHeader = false;
		}
	
	void printRowBorder() 
		{
		if (! config.printborder)
			return;
			
		out.print(rowborder);	
		}	

	private void initRowBorder() 
		{
		//new Exception().printStackTrace();
		int default_padded_cellwidth = 
				cellwidth + 2 * config.cellpadding;
		
		if (! config.printborder)
			return;
		
		StringBuffer buf = new StringBuffer(
				columncount * (1+default_padded_cellwidth+config.cellspacing));
			
		for (int n = 0; n < columncount; n++)
			{
			if (usingSpacing)
				buf.append(spacing_str);
			buf.append(config.cellcorner);
			int padded_cellwidth = config.getCellWidthForColumn(n) 
									+ 2 * config.cellpadding;
			for (int k = 0; k < padded_cellwidth; k++) 
				{
				buf.append(config.cellborder_h);
				}
			if (usingSpacing)
				buf.append(config.cellcorner);
			}
		if (! usingSpacing)
			buf.append(config.cellcorner);
		buf.append(linesep);
		rowborder = buf.toString();
		}

void printVerticalSpacing() 
	{
	//1 lesser for aesthetics
	for (int k = 1; k < config.cellspacing; k++) { 
		out.print(linesep);
		}
	}			

//== autofit methods ================

void startTableAutoFit() {
	largest_width = new int[columncount];
	}
	
void endTableAutoFit() 
	{
	if (dbg) {
		for (Iterator it = autoFitTableData.iterator(); it.hasNext(); /*empty*/) {
			String[] item = (String[]) it.next();
			System.out.println(Arrays.asList(item));
			}	
		}
		
	// == 2nd pass ==================================	
	int rowcount = autoFitTableData.size();
    if (rowcount == 0)
    	return;

    //we are already handling autofit rendering, so terminate further recursion
    config.setAutoFit(false); 

    for (int n = 0; n < largest_width.length; n++) {
		if (dbg) System.out.println("largest_width["+n+"]="+largest_width[n]);
	    config.setCellWidthForColumn(n, largest_width[n]);
    	}

    //TablePrinter printer = new TablePrinter(columncount, out, config);
	TablePrinter printer = this;
	printer.initRowBorder();
	
    printer.startTable();
    int n = 0;
    while (n < rowcount) 
    	{
	    printer.startRow();
    	for (int k = 0; k < columncount; k++) {
   		 	printer.printCell(
    		((String[])autoFitTableData.get(n))[k] );
    		}
    	printer.endRow();
    	n++;
    	}
		
    printer.endTable();	
	}
	
void startRowAutoFit() {
	currentAutoFitCell = 0;
	autoFitTableData.add(currentAutoFitRow, new String[columncount]);
	}
	
void endRowAutoFit() {
	currentAutoFitRow++;
	}

void printCellAutoFit(String str) 
	{
	int len = (str != null) ? str.length() : "null".length();
	
	if (largest_width[currentAutoFitCell] < len)
		 largest_width[currentAutoFitCell] = len;

	String[] row = (String[]) autoFitTableData.get(currentAutoFitRow);
	row[currentAutoFitCell++] = str;	
	}

final private Pattern newlines = Pattern.compile("\\r|\\n|\\r\\n");
String removeEmbeddedSpaces(String str)
	{
	/*
	we convert embedded newlines in the string to spaces
	otherwise the embedded newline will cause printing
	to start at the left most margin. i.e.,
	
	baz\n\baz2 will print like
			
	A    B  	C
	x	 y		baz
	baz2
	
	but we want
				<---> 
	A    B    	C
	x	 y		baz
				baz2
	
	*/

	if (str != null)
		{
		str = newlines.matcher(str).replaceAll(" ");
		}
	return str;
	}

/** 
Configuration object containing for table printing object.

Some methods in this class return <tt>this</tt> object for method
chaining convenience. 
<p>
Note: Alas ! TablePrinter does not support cellspans across
columns or rows. That would make things too complicated for
this implementation.
**/
public static class PrintConfig
	{
	static final ToString.Style style = new ToString.Style();
	static {
		style.reflectVisibleLevel = ToString.Style.VisibleLevel.PRIVATE;
		}
			
	private int		 cellwidth 	 		 = 	20;
	
	//individual widths if specified
	private Map		 cellwidthMap		 = new HashMap();
	 
	private String 	 cellcorner 	 	 = 	"+";
	private String 	 cellborder_h 		 = 	"-";
	private String 	 cellborder_v 		 = 	"|";
	private String	 cellpadding_glyph	 =  " ";
	private String	 cellspacing_glyph	 =  " ";
	private boolean  cellwrap			 =  true;
	private boolean  autoFit			 =  false;
	private boolean	 printborder  		 =  true;
	private	int		 cellspacing  		 =	0;
	private	int		 cellpadding  		 =	0;
	private	HAlign	 align		 		 =  HAlign.LEFT;
	private int		 pageSize	 		 = -1;		
	private boolean	 headerEveryPage	 = true;
	private String[] header;
	
	/** 
	Constructs a new PrintConfig with the following default options. 
	<ul>
		<li>cell corner: the <tt>+</tt> character 
		<li>horizontal cell border: the <tt>-</tt> character 
		<li>vertical cell border, the <tt>|</tt> character 
		<li>width of each column: 20 chars 
		<li>horizontal alignment of each cell: {@link HAlign#LEFT}
		<li>The character used for cellpadding and cellspacing is a
		blank space.
	</ul>
	**/
	public PrintConfig() {
		}
	
	/** 
	Optionally sets the header row for the table **/
	public void setHeader(String[] header) {
		this.header = header;
		}	

	/** 
	Gets the header row for the table if set. If not
	set, return null
	**/
	public String[] getHeader() {
		return this.header;
		}	

	
	/** 
	Sets the number of lines on each page. A page is a
	logical unit that typically shows some number of lines
	on the screen without the need to scroll the page. This
	value is useful in conjunction with other page specific
	settings like {@link showPageHeading()}
	
	@param	lines	number of lines on the page, specify a zero or
					negative quantity for a single page. (number of
					lines <b>not</b> including lines occupied by the table 
					header itself, if the header is printed).
	**/
	public PrintConfig setPageSize(int lines) {
		this.pageSize = lines;
		return this;
		}
		
	/** 
	Specifies that page heading (if set) on each separate
	page. This may be ignored if no heading has been set.
	<tt>true</tt> by default, although headers will still
	not be printed until the {@link setPageSize} method is
	invoked.
	**/
	public PrintConfig headerEveryPage(boolean show) {
		this.headerEveryPage = show;
		return this;
		}
	
	/** Sets the alignment of each cell. By default: 
	<tt>{@link HAlign#LEFT left}</tt>
	**/
	public PrintConfig setAlign(HAlign align) {
		this.align = align;
		return this;
		}

	/** 
	Sets the string (typically a single character) that makes up a 
	cell corner. Defaults to <tt>+</tt>
	**/
	public PrintConfig setCellCorner(String str) {
		this.cellcorner = str;
		return this;
		}

	/** 
	Sets the string (typically a single character) that makes up a 
	horizontal cell border. Defaults to <tt>-</tt>
	**/
	public PrintConfig setCellBorderHorizontal(String str) {
		this.cellborder_h = str;
		return this;
		}

	/** 
	Sets the string (typically a single character) that makes up a 
	vertical cell border. Defaults to <tt>|</tt>
	**/
	public PrintConfig setCellBorderVertical(String str) {
		this.cellborder_v = str;
		return this;
		}

	/** 
	Sets the string (typically a single character) that makes up
	the cellpadding. Defaults to <tt>" "</tt>
	**/
	public PrintConfig setCellPaddingGlyph(String str) {
		this.cellpadding_glyph = str;
		return this;
		}

	/** 
	Sets the string (typically a single character) that makes up
	the cellspacing. Defaults to <tt>" "</tt>
	**/
	public PrintConfig setCellSpacingGlyph(String str) {
		this.cellspacing_glyph = str;
		return this;
		}

	/** 
	Sets the width of each cell. Defaults to <tt>20</tt>. 
	This width is common to all cells.

	@param	width	the width of each cell
	**/
	public PrintConfig setCellWidth(int width) {
		if (width <= 0) {
			throw new IllegalArgumentException("cell width must be greater than zero");
			}
		this.cellwidth = width;
		return this;
		}	

	/** 
	Sets the cell width of the the specified column. cells
	are numbered starting from 0 
	
	@param	column	the cell number
	@param	width	the desired character width
	**/
	public PrintConfig setCellWidthForColumn(int column, int width) {
		cellwidthMap.put(Integer.valueOf(column), Integer.valueOf(width));
		return this;
		}
	
	/** 
	convenience method for internal use:
	returns the cellwidth if set otherwise the default cellwidth
	**/
	int getCellWidthForColumn(int column) 
		{
		Object obj = cellwidthMap.get(Integer.valueOf(column));
		if (obj != null) {
			return ((Integer) obj).intValue();
			}
		return cellwidth;
		}
	
	
	/** 
	Cell wrapping is on by default and the contents of any column that 
	exceed the width are wrapped within each cell (using the current platforms line 
	seperator for newlines within the cell). 
		
	@param	wrap <tt>true</tt>	To turn cell wrapping on, <tt>false</tt> 
								to turn cell wrapping off and show
								fixed width contents only.
	**/	
	public PrintConfig setCellWrap(boolean wrap) {
		cellwrap = wrap;	
		return this;
		}
	
	/** 
	Sets each cell to expand to the size needed for the maximum
	sized cell in that column. By default, this is <tt>false</tt>.
	Setting this to <tt>true</tt> automatically turns <b>off</b> 
	cell wrapping.
	<p>
	Note: using autofit may result in slower performance
	because the table almost always has to be rendered in
	two passes.
	
	@param	autofit		<tt>true</tt> to turn autofit on, 
						<tt>false</tt> otherwise.
	**/
	public PrintConfig setAutoFit(boolean autofit) 
		{
		autoFit = autofit;
		if (autofit)
			cellwrap = false;
		return this;
		}	
	
	/**  
	Specifies whether table and cell borders are printed. By
	default this is true.

	@param	print	<tt>true</tt> to print borders, <tt>false</tt> otherwise
	**/
	public PrintConfig setPrintBorders(boolean print) {
		printborder = print;
		return this;
		}
	
	/**  
	Specifies the cell spacing between cells. This is useful
	for tables with no borders. By default, this value is
	zero.
	**/
	public PrintConfig setCellSpacing(int width) {
		cellspacing = width;
		return this;
		}
		
	/**
	Specifies the cell padding for each cell. This is useful for
	tables with no borders. By default, this value is zero.
	**/
	public PrintConfig setCellPadding(int width) {
		cellpadding = width;
		return this;
		}
	
	/** 
	Prints a short description of this object. 
	**/
	public String toString() 
		{ 
		return new ToString(this, style).reflect().render();
		}
		
	} //~PrintConfig

public static void main(String[] args)
	{	
	System.out.println("Table with borders and *NO* wrapping, (truncated to fixed cell width)");
	//borders
	PrintConfig config = new PrintConfig();
	config.setCellWrap(false);
	printTest(config, null);


	System.out.println("Table with borders and autofit");
	//borders
	config = new PrintConfig();
	config.setAutoFit(true);
	printTest(config, null);


	System.out.println("Table with borders and wrapping");
	//borders
	config = new PrintConfig();
	printTest(config, null);
	
	System.out.println("");	
	System.out.println("Table with borders and padding & spacing");
	//borders + padding
	config = new PrintConfig();
	config.setAutoFit(true);
	config.setCellSpacing(2);
	config.setCellPadding(1);
	config.setCellWidth(5);
	printTest(config, null);

	System.out.println("");	
	System.out.println("Table with borders and padding & spacing and AUTOFIT");
	//borders + padding
	config = new PrintConfig();
	config.setCellSpacing(2);
	config.setCellPadding(1);
	config.setCellWidth(5);
	printTest(config, null);

	System.out.println("");	
	System.out.println("Table with borders, spacing=2 page size = 2 lines and headers = true");
	config = new PrintConfig();
	config.setCellSpacing(2);
	config.setCellWidth(5);
	config.setPageSize(2);
	config.headerEveryPage(true);
	printTest(config, new String[] {"Header", "A", "helloworld", "d", "End Header"});

	System.out.println("");	
	System.out.println("Table with borders, no padding and default alignment of CENTER");
	//borders + padding
	config = new PrintConfig();
	config.setCellSpacing(0);
	config.setCellPadding(0);
	config.setCellWidth(15);
	config.setAlign(HAlign.CENTER);
	printTest(config, null);

	System.out.println("");
	System.out.println("Table with no borders and padding & spacing");
	//no borders + padding
	config = new PrintConfig();
	config.setPrintBorders(false);
	config.setCellWidth(5);
	config.setCellSpacing(2);
	config.setCellPadding(3);
	printTest(config, null);

	}


static void printTest(PrintConfig config, String[] header) 
	{
	TablePrinter printer = new TablePrinter(5, System.out, config);

	if (header != null)
		config.setHeader(header);
		
	printer.startTable();

	//row 1
	printer.startRow();
	printer.printCell("abcdef");
	printer.printCell("4324");
	printer.printCell("Q");
	printer.printCell("");
	printer.printCell("abc");
	printer.endRow();
		
	//2
	printer.startRow();
	printer.printCell("row2: abc");
	printer.printCell("the quick brown \n-fox jumps over the lazy dog");
	printer.printCell("hello");
	printer.printCell("world");
	printer.printCell(null);
	printer.endRow();


	//3
	printer.startRow();
	printer.printCell("row3");
	printer.printCell("a");
	printer.printCell(null);
	printer.printCell(null);
	printer.printCell(null);
	printer.endRow();
	
	printer.endTable();
	
	System.out.println("TablePrinter.toString()=" + printer);
	System.out.println("");
	}
	
} //~TablePrinter	 
