// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import java.io.*;

import fc.util.*;
/**
Prompts a user for input.
<p>
Thread safety: This class is not thread safe and higher level
synchronization (or multiple instances of this class) should
be used.

@author hursh jain	
@version 1.0 12/15/2001
**/
public class Prompt 
{
			BufferedReader 	in; 
			PrintWriter 	err;
			PrintWriter 	out;
protected 	String 			line = null; //the current input line

/**
Constructs the object using the specified input and output writers.
@param in	the BufferedReader to use for input
@param out	the PrintWriter to use for output
@param err	the PrintWriter to use for errors
**/
public Prompt(Reader in, PrintWriter out, PrintWriter err)
	{
	Argcheck.notnull(in); 
	Argcheck.notnull(out); 
	Argcheck.notnull(err);
	this.in = IOUtil.bufferReader(in);
	this.out = out;
	this.err = err;
	}

/**
Constructs the object using the specified input and output streams.
@param in	the BufferedReader to use for input
@param out	the PrintWriter to use for output
@param err	the PrintWriter to use for errors
**/
public Prompt(InputStream in, PrintStream out, PrintStream err)
	{
	Argcheck.notnull(in); 
	Argcheck.notnull(out); 
	Argcheck.notnull(err);
	this.in = new BufferedReader(new InputStreamReader(in));
	this.out = new PrintWriter(out);
	this.err = new PrintWriter(err);
	}


/**
Constructs a fallback object, using <tt>System.in, System.out</tt>
for the input, output and error streams.
**/
public Prompt()
	{
	this(	
		new BufferedReader(new InputStreamReader(System.in)), 
		new PrintWriter(System.out), 
		new PrintWriter(System.err)
		);
	}

/**
Shows the specified string to the user shows the input prompt
to the user (to indicate the user's input is expected). Keeps
looping and asking the same question until a non-empty response
has been typed by the user, then returns that response.

@param	str			the string to show to the user
@param	newline		whether to print a newline after the string
*/
public String ask(String str, boolean newline) 
	{
	String result = "";	
	while (result.equals("")) 
		{
		out.print(str); //the string
		
		if (newline)
			out.println();
			
		out.flush();
		readLine();
		result = line;
		}

	return result;
	}

/**
Calls {@link ask(String, boolean)} so that a newline
is not shown after the question shown to the user.
**/
public String ask(String str) {
	return ask(str, false);
	}

/**
Shows the specified string to the user shows the input prompt
to the user (to indicate the user's input is expected). Returns
the fallback string if a empty response has been typed by the user, 
else returns the user response.

@param	str			the string to show to the user
@param	fallback		response to return if the user types in an 
					empty response (typically by simply hitting a
					newline)
@param	newline		whether to print a newline after showing the
					specified string to the user
**/
public String ask(String str, String fallback, boolean newline) 
	{
	out.print(str); 
	if (newline)
		out.println();
	
	out.flush();
	readLine();
	if (line.equals(""))
		return fallback;
	
	return line;
	}

/** 
Calls {@link ask(String, String, boolean)} so that a newline
is not shown after the question shown to the user.
**/
public String ask(String str, String fallback) {
	return ask(str, fallback,  false);
	}

	
/**
Immediately writes a string to the output. The prompt is written
before the string.
*/
public void write(String str) 
	{
	out.print(str);
	out.flush();	
	}

/**
Immediately writes the specified string as a new line, to the output.
The prompt is written before the string.
*/
public void writeln(String str) 
	{
	out.println(str);
	out.flush();	
	}

/**
Gets the latest input line that has been read. Can return the
same line multiple times if no new lines have been read since
the last time this method was called. Returns <tt>null</tt>
if no line has been read so far.
**/
public String getLastLine() {
	return line;
	}

/**
Sets the next line of input without the trailing newline
character. A line of input is signalled when a newline character
is encountered. This method will block until the input has a 
newline.<br>
Note: Unfortunately, as of JDK 1.4 there seems to be no way to
get the input buffer unless a newline is also entered. (Stream/Reader
methods like <tt>available</tt> and <tt>ready</tt> return 0/false
until a newline is entered). This means <tt>System.in</tt> always
appears to work in "cooked" mode (both on windows and an linux).
This sucks.
**/
protected void readLine() 
	{
	try {
	  	line = in.readLine();
		} 
	catch(java.io.IOException e) {
		e.printStackTrace();
		}
	}

//-- stream/writer sets ----

public void setInput(Reader in) {
	this.in = IOUtil.bufferReader(in);
	}

public void setInput(InputStream in) {
	this.in = new BufferedReader(new InputStreamReader(in)); 
	}

public void setOutput(PrintStream out) {
	this.out = new PrintWriter(out);
	}

public void setOutput(PrintWriter out) {
	this.out = out;
	}

public void setError(PrintStream err) {
	this.err = new PrintWriter(err);
	}

public void setError(PrintWriter err) {
	this.err = err;
	}


public static void main(String args[]) throws Exception
	{
	new Test();	
	}

/**	
Unit Test History 	
<pre>
Class Version	Tester	Status		Notes
1.0				hj		passed		
</pre>
*/	
static private class Test 
{
Test() throws Exception
	{
	Prompt prompt = new Prompt();
	String name = prompt.ask("What is your name ? ");
	System.out.println("You entered: " + name);
	
	name = prompt.ask("What is your age [default 1] ? ", "1");
	System.out.println("You entered: " + name);
	}
} //~Test

}           //~class Prompt