// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.page;

import java.io.*;
import java.util.*;

import fc.io.*;
import fc.util.*;

/**
A Reader suitable for lexing. Supports all of: <code>peek</code>,
<code>read</code> and <code>unread</code>. (no JDK 1.5 reader class has
all of those). Mark/reset is not supported because it's too complex to
implement given the current <i>fixed-buffer</i> implementation of this
class. (on the flip-side this implementation does allow to read
<i>very</i> large files without risk of running out of JDK memory).
<p>
<xmp>1 4m 31337 h4x0r! ph33r |\/|y m4d sk1llz !!11!</xmp>:-]
<p>
Note 1: If this class is invoked from the command line, setting the dbg
flag in the code to <code>true</code> is useful.

@author hursh jain
*/
public final class PageReader extends Reader
{
/*
Old implemenation had a rolling buffer window and a previous spillover buffer.
It was too complicated, so that implemenation is pastured.

It is MUCH simpler to read the entire file into one character array...(unread,
read, mark, reset) etc, become trivial. By contrast, if using a
limited-size-read-buffer implementation (as before) all of those become much
harder since we loose the existing buffer contents when the buffer is *refilled*
(so how do you go back in the buffer if the buffer is gone ?).
*/
static final boolean dbg = false;

//our own buf/pos because most/all reader subclasses dont have mark/reset/unread  
char[]			buf 			= null;
int				pos				= 0;
int				count 			= 0;
int				markpos			= 0;
//line, col and other tracking 
int				line			= 1;
int				col				= 0;
int				lastcol			= 1; 	//for unread past a newline
boolean			pushBackNL		= false;
boolean			lastWasCR		= false;
String			encoding;
static String	DEFAULT_ENCODING = "UTF-8";

/** 
Creates a new PageReader wrapping the specified reader
*/
public PageReader(Reader r) throws IOException
	{
	Argcheck.notnull(r, "specified reader was null");
	buf = IOUtil.readerToCharArray(r);
	this.encoding = DEFAULT_ENCODING;
	}

/**
Creates a reader with the specified non-null encoding.
*/
public PageReader(File file, String encoding)  throws IOException
	{
	Argcheck.notnull(file, "specified file was null");
	Argcheck.notnull(encoding, "specified encoding was null");
	this.encoding = encoding;
	buf = IOUtil.fileToCharArray(file, encoding);
	}

/**
Creates a reader using the UTF-8 encoding.
*/
public PageReader(File file)  throws IOException
	{
	this(file, DEFAULT_ENCODING);
	}

public void close() throws IOException
	{
	//no underlying stream since everything read into buffer. not much to do.
	}
	
public int read() throws IOException
	{
	if (pos == buf.length) {
		return -1;
		}
		
	char c = buf[pos++];	
	
	if (dbg) System.out.println(">>>>>>>> DEBUG: read() from BUF, c=" + StringUtil.viewableAscii(c));
	adjustReadLineNum(c);

	return c;
	}

public int read(char[] buf, int start, int len) throws IOException
	{
	throw new IOException("not implemented, use the read() method instead");
	}

/**
Unreads the current character (which could be EOF) so that the next read will 
return the current character (or EOF) again.
*/
public void unread() throws IOException
	{
	char c = 0;
	
	if (pos == 0) 
		{
		throw new IOException("I am at the beginning of the stream. Cannot unread anything because nothing has been read so far");
		}
	else{ 
		c = buf[--pos];
		if (dbg) System.out.println(">>>>>>>> DEBUG: unread() from BUF, c=" + StringUtil.viewableAscii(c));
		}
	
	adjustUnreadLineNum(c);
	}

/**
Unreads the specified number of characters
*/
public void unread(int count)  throws IOException
	{
	for (int n = 0; n < count; n++) {
		unread();
		}
	}

/**
Useful for inserting included files into the stream and then parsing that content in-line
with the rest of the file.
*/
public void insertIntoStream(File file) throws IOException
	{
	char[] insert = IOUtil.fileToCharArray(file, encoding);

	char[] result = new char[buf.length + insert.length];
	System.arraycopy(buf, 0, result, 0, pos);
	System.arraycopy(insert, 0, result, pos, insert.length);
	System.arraycopy(buf, pos, result, pos+insert.length, buf.length-pos);
	
	buf = result;
	}

/**
Useful for inserting included files into the stream and then parsing that content in-line
with the rest of the file.
*/
public void insertIntoStream(Reader r) throws IOException
	{
	char[] insert = IOUtil.readerToCharArray(r);

	char[] result = new char[buf.length + insert.length];
	System.arraycopy(buf, 0, result, 0, pos);
	System.arraycopy(insert, 0, result, pos, insert.length);
	System.arraycopy(buf, pos, result, pos+insert.length, buf.length-pos);
	
	buf = result;
	}


void adjustReadLineNum(char c)
	{
	// we can read: \r, \r\n , \n all of which increase line count by exactly 1
	switch (c) 
		{
		case '\n': 
			if (! lastWasCR) {
				line++;
				lastcol=col;
				col=1;
				}
			else {
				lastWasCR = false;
				}
			break;
		
		case '\r': 
			line++;
			lastcol=col;
			col=1;
			lastWasCR = true;
			break;
		
		case '\t':
			col = col + 4;
			break;
		
		default:
			col++;
		}
	}


void adjustUnreadLineNum(char c)
	{
	// we can unread: \r, \r\n , \n all of which reduce line count by exactly 1
	switch (c) {
		case '\n': 
			pushBackNL = true;
			line--;
			col=lastcol;
			break;
		case '\r': 
			if (! pushBackNL) { 
				line--;
				col=lastcol;
				}
			else{
				pushBackNL = false;
				}
			break;
		case '\t':
			col = col - 4;
			break;
		default:
			col--;
		}
	}

public int peek() throws IOException
	{
	return buf[pos];
	}


/**
Skips all whitespace characters such that the next {@link read} will
return the <b>next</b> non-whitespace character (or EOF if there are no
more characters).
*/
public void skipWhitespace() throws IOException
	{
	int c = -1;
	while (true)
		{
		c = read();
		
		if (c == -1) {
			break;
			}
		
		if (! Character.isWhitespace(c)) {
			unread();
			break;
			}
		}
	}
	

/**
Tries to read/consumes the specified char and returns true
if successful. If the specified char is not found, does not
consume anything and returns false.
*/
public boolean match(int target) throws IOException
	{
	int c = read();
	
	if (c == target)
		return true;
	else
		unread();
	
	return false;
	}

/**
Tries to read/consumes the specified non-null string and returns true
if successful. If the specified string is not found, does not
consume anything and returns false.
*/
public boolean match(String target) throws IOException
	{
	if (target == null)
		throw new IllegalArgumentException("Specified target string was null");
	
	int c = -1;
	for (int i = 0; i < target.length(); i++)
		{
		c = read();
		
		if ( c == -1 || c != target.charAt(i)) {
			unread(i+1);
			return false;
			}
		}
	
	return true;
	}

public boolean matchIgnoreCase(String target) throws IOException
	{
	if (target == null)
		throw new IllegalArgumentException("Specified target string was null");
	
	int c = -1;
	for (int i = 0; i < target.length(); i++)
		{
		c = read();
		
		if ( c == -1 || c != Character.toLowerCase(target.charAt(i))) {
			unread(i+1);
			return false;
			}

		}
	
	return true;
	}

public boolean markSupported()
	{
	return false;
	}

public 	int getLine() { 
	return line; 
	}
	
public 	int getCol() { 
	return col; 
	}

char[] getBuf() { return buf; }
int getPos() { return pos; }

//other utility methods

public static void main (String args[]) throws IOException
	{
	//CHANGE CHAR BUFFER TO A SMALL VALUE FOR TESTING */
	StringReader sr = null;
	PageReader lex = null;
	int c = -1;
	
	System.out.println("Reading an empty string.....");	
	sr = new StringReader("");	
	lex = new PageReader(sr);
	while ( (c = lex.read()) != -1) {
		testprint(lex, c);
		}
	
	System.out.println("----------------- TEST 2 --------------");
	sr = new StringReader("abc");		
	lex = new PageReader(sr);
	while ( (c = lex.read()) != -1) {
		testprint(lex, c);
		//System.out.print(c + " ");
		}

	System.out.println("----------------- TEST 3 --------------");
	sr = new StringReader("abcde");		
	lex = new PageReader(sr);
	try	{
		c = lex.read();
		testprint(lex, c);
		lex.unread();
		testprint(lex, -10);
		lex.unread();
		testprint(lex, -10);
		c = lex.read();
		testprint(lex, c);
		}
	catch (Exception e) {
		e.printStackTrace();
		}

	System.out.println("----------------- TEST 4 --------------");
	sr = new StringReader("abcd\ne");		
	lex = new PageReader(sr);
	try	{
		c = lex.read();
		testprint(lex, c);
		lex.unread();
		testprint(lex, -10);

		for (int i = 0; i < 5; i++) {
			c = lex.read();
			testprint(lex, c);
			}

		for (int i = 0; i < 5; i++) {
			lex.unread();
			testprint(lex, -10);
			}
		
		for (int i = 0; i < 5; i++) {
			c = lex.read();
			testprint(lex, c);
			}
		
		c = lex.read();
		testprint(lex, c);
		}
	catch (Exception e) {
		e.printStackTrace();
		}

	System.out.println("----------------- TEST 5 --------------");
	sr = new StringReader("abcd\r\ne");		
	lex = new PageReader(sr);
	try	{
		c = lex.read();
		testprint(lex, c, lex.peek());
		lex.unread();
		testprint(lex, -10, lex.peek());

		for (int i = 0; i < 5; i++) {
			c = lex.read();
			testprint(lex, c, lex.peek());
			}

		for (int i = 0; i < 5; i++) {
			lex.unread();
			testprint(lex, -10, lex.peek());
			}
		
		for (int i = 0; i < 5; i++) {
			c = lex.read();
			testprint(lex, c, lex.peek());
			}
		
		c = lex.read();
		testprint(lex, c, lex.peek());
		}
	catch (Exception e) {
		e.printStackTrace();
		}

	System.out.println("--------- TEST 6 ---(insert into stream middle)-------");
	sr = new StringReader("abc");	
	lex = new PageReader(sr);
	
	try	{
		c = lex.read();
		testprint(lex, c);
	
		StringReader insert = new StringReader("123");
		System.out.println("inserting \"123\" into the stream\n");
		lex.insertIntoStream(insert);

		while ( (c = lex.read()) != -1) {
			testprint(lex, c);
			}
		}
	catch (Exception e) {
		e.printStackTrace();
		}


	System.out.println("--------- TEST 7 ---(insert into stream begin)-------");
	sr = new StringReader("abc");	
	lex = new PageReader(sr);
	
	try	{
		StringReader insert = new StringReader("123");
		System.out.println("inserting \"123\" into the beginning of stream\n");
		lex.insertIntoStream(insert);

		while ( (c = lex.read()) != -1) {
			testprint(lex, c);
			}
		}
	catch (Exception e) {
		e.printStackTrace();
		}

	System.out.println("--------- TEST 8 ---(insert into stream end)-------");
	sr = new StringReader("abc");	
	lex = new PageReader(sr);
	
	try	{
		while ( (c = lex.read()) != -1) {
			testprint(lex, c);
			}
		StringReader insert = new StringReader("123");
		System.out.println("inserting \"123\" into the end of the stream\n");
		lex.insertIntoStream(insert);

		while ( (c = lex.read()) != -1) {
			testprint(lex, c);
			}
		}
	catch (Exception e) {
		e.printStackTrace();
		}

	}

private static void testprint(PageReader lex, int c, int peek)
	{
	if (c == -1) {
		System.out.println("====> recieved EOF (-1) from read().......");
		}

	System.out.format(
		"buf=%s, pos=%d, buflen=%d\nline=%d, col=%d, char=[%s]",
 		StringUtil.arrayToString(lex.getBuf()), lex.getPos(), lex.getBuf().length,
		lex.getLine(), lex.getCol(), 
		(c == -10) ? "N/A" : StringUtil.viewableAscii((char)c));	

	if (peek != -2)  
		System.out.format(", peek=[%s]", StringUtil.viewableAscii((char)peek));

	System.out.print("\n\n");
	}

private static void testprint(PageReader lex, int c)
	{
	testprint(lex, c, -2);
	}

}
