// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.page;

import java.io.*;
import java.util.*;
import java.net.URL;

import jakarta.servlet.*;
import jakarta.servlet.http.*;

import fc.util.*;
import fc.io.*;
import fc.io.fileselectors.*;

/** 
Regression testing since eyeballing parser output just ain't cutting it. 
For developers only, useful when the molly parser is hacked/changed and
we want to ensure that the new parser does not break any existing behavior.
<p>
All regression tests are in the <pre>test</pre> subdirectory (relative to the java
source of this class). These tests exist as *.mp files. This program will
run the parser on each file and a) either check actual output with expected
output and/or b) see if an expected error occurs.
<p>
The expected/canonical output (generated by the good working parser) also
always exists in the <pre>test</pre> subdirectory. The <i>actual</i>
output generated by the latest hacked parser is output in a temporary
directory (the location of which is specified on the command line). The
output are then compared and should be identical.
<p>
The expected output can be regenerated via the <pre>generateExpected</pre>
argument but this should be used by caution. Future/actual output of the
changed/hacked parser is compared with the last good generated expected
output so the expected output should only be updated when the parser is 
in a known working state.

@author hursh jain
*/
public class PageParserTest
{
static 	List results = new ArrayList();

public static void main (String args[]) throws Exception
	{	
	Args myargs = new Args(args);
	Log log = Log.getDefault();
	myargs.setUsage("java " + myargs.getMainClassName() 
        + "\n---- Specify ---"
        + "\n   -test   <if specified, run all unit tests with the latest parser>"
        + "\n   -tmpdir <location to a tmp dir used for generated files"
        + "\n----- or ----"
        + "\n   -generateExpected if specified, expected/canonical parser output "
        + "\n                     is generated. Use with caution. The servlet API"
        + "\n                     along with all the molly classes must be on the"
        + "\n                     system classpath for this option to work fully."
		);

	boolean testing	 = myargs.flagExists("test");
	boolean generate = myargs.flagExists("generateExpected");
	File 	tmpdir = null;
	
	if (! testing && ! generate) 
		myargs.showError();

	if (testing) 
		{
		tmpdir = new File(myargs.getRequired("tmpdir"));
		if (! tmpdir.exists()) {
			log.error("The specified temp directory: " + tmpdir.getAbsolutePath() + " does not exist");
			return;
			}
		}

	Class c = new PageParserTest().getClass();
	URL url = c.getResource("test");
	//System.out.println(url + ";" + url.getFile());	
	File testdir = new File(url.toURI());
	if (! testdir.isDirectory()) {
		log.error(testdir + " is not a directory. This test does not work from within jars. For development, you need to explode the jar");
		return;
		}
		
	log.bug("Regression test directory: " + testdir.getAbsolutePath());

	File[] files = testdir.listFiles(new SuffixSelector(".mp"));
	Arrays.sort(files);
	
	if (files.length == 0)
		log.warn("No *.mp test files found in directory: ", testdir.getAbsolutePath());
		
	Watch wp = new Watch("parse");
	Watch wc = new Watch("compile");
	
	int crap_count = 0;
	int generate_count = 0;
	for (int n = 0; n < files.length; n++) 
		{
		File test = files[n];
		log.bug("Processing file: ", test.getName()); 
				
		//some files are not expected to compile (as part of the test)
		//those files start with "crap..."
		boolean crap = test.getName().toLowerCase().startsWith("crap");
		if (crap) {
			crap_count++;
			}
			
		String result_file_str = test.getName();
		int i = result_file_str.lastIndexOf(".");
		String outbase = result_file_str.substring(0, i);
		String outname = outbase + ".java";
		
		File expected = new File(testdir, outname);
		File out  	  = new File(tmpdir, outname);
		
		PageParser p;
		PageCompiler pc;
		CharArrayWriter buf = new CharArrayWriter();
		
		//generate
		if (generate) 
			{
			try {
				buf.reset();
				buf.append("Generate: ").append(
					String.format("%-20s",test.getName()));

				wp.start();
				p = new PageParser(testdir, test, expected, outbase);
				p.parse();
				wp.stop();
				
				buf.append(" [parsed in:").
					append(
						String.format("%3s",
						String.valueOf(wp.getTime())))
					.append( " ms, "); 
				//successful parse, no parse exceptions
				//there may be compile time errors though (expected for
				//crap files)
				wc.start();
				pc = new PageCompiler(expected, System.getProperty("java.class.path"), null);
				boolean compiled = pc.compile();
				wc.stop();

				if (! compiled) {
					buf.append(" compile error...possibly expected]"); 
					log.bug(pc.getError());
					}
				else{
					buf.append(" compiled in: ").
						append(
							String.format("%5s",
							String.valueOf(wc.getTime())))
						.append(" ms]"); 
					//remove class file
					File f = new File(testdir, outbase + ".class");
					if (f.exists()) {
						f.delete();
						}
					//compiled ok too.
					generate_count++;
					}
				}
			catch (Exception e) 
				{
				buf.append(" [parse error...possibly expected]"); 
//				if (! crap) { //crap files are expected to have errors
					msg(test.getName(),  "Parse Error: " + e.toString());
//					}
				}

			log.info(buf.toString());
			continue;
			} //end if (generate)
	
		//test
		
		//we need an expected file to compare with
		if (! expected.exists()) 
			{
			if (! crap) {  //wont exist for crap files cause parse error
				msg(test.getName(), "FAILED [expected file: " +  expected.getAbsolutePath() + " does not exist]");
				continue;
				}
			}
			
		p = new PageParser(testdir, test, out, outbase);
		
		try {
			p.parse();
			//crap files will cause a parse exception
			//if here, then parse successful
			if (! compare(expected, out)) {
				msg(test.getName(), "FAIL [expected != actual]");
				}
			else{
				msg(test.getName(), "OK");
				}
			}
		catch (Exception e) {
			if (crap) {
				//we wanted the expected parse exception
				msg(test.getName(), "OK");
				}
			else{
				msg(test.getName(), "FAIL [ParseException] " + e.toString());
				}
			}
		
		} //end for loop
	
	if (generate) {
		System.out.println();
		System.out.println((files.length - crap_count) + " total (non-crap) *.mp files in " + testdir.getAbsolutePath());
		System.out.println(generate_count + " expected files were generated.");
		}
	
	//print results
	System.out.println();
	TablePrinter.PrintConfig pconfig = new TablePrinter.PrintConfig();
	pconfig.setAutoFit(false).setPrintBorders(true).setCellPadding(1);
	if (generate) {
		pconfig.setCellWidthForColumn(1,55);
		System.out.println("The following errors are expected!");
		}
	TablePrinter printer = new TablePrinter(2, System.out, pconfig);
	printer.startTable();
	for (int n = 0; n < results.size(); n++) {
		printer.startRow();
		Object[] arr = (Object[]) results.get(n);
		printer.printCell(arr[0].toString());
		printer.printCell(arr[1].toString());
		printer.endRow();
		}
	printer.endTable();
	}

static void msg(Object testName, Object result)
	{
	results.add(new Object[] {testName, result});
	}

static boolean compare(File expected, File output) throws Exception
	{
	if (! output.exists())
		return false;
		
	byte[] expected_b = IOUtil.fileToByteArray(expected);
	byte[] output_b   = IOUtil.fileToByteArray(output);
	
	return Arrays.equals(expected_b, output_b);
	}

}		
