// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.page;

import java.io.*;
import java.util.*;

import fc.util.*;
import fc.io.*;

/**
Manages pages. Pages are found below a web document root directory. Pages
are compiled as needed and the resulting class file is loaded/run. If a
page is changed, it is automatically recompiled, reloaded and rerun. If the
page has a compilation error, that page remains unloaded until the error is
fixed.
<p>
A new PageMgr should be instantiated for each unique root directory (for
example with multiple virtual hosts or users, each having their own root
directory).

@author hursh jain
*/
public final class PageMgr
{
private static final boolean dbg = false;

File 		docroot;
File 		scratchdir;    //for compiled pages.
Map			pagemap = new HashMap();
Log			log;
String 		classpath;
PageServlet servlet;      //added this so a page can refer back to the parent servlet

//used by performance hack in getPage(), remove if hack removed.
String 		docrootStr;
String 		scratchdirStr;    

/**
Constructs a new Page manager. The page manager will use the system
classpath and <tt>/WEB-INF/classes</tt>, <tt>/WEB-INF/lib</tt> as the
classpath for complilation. Pages can refer to any class found in those
locations.

@param	servlet		the Molly Servlet. This is optional and can be
					<tt>null</tt> when creating/testing the PageMgr from
					the command line.
@param	docroot		absolute path to the document root directory within
					which the pages are found. This is the directory that
					correspond to the "/" location of the webapp.
@param	scratchdir	absolute path to a scratch dirctory where intermediate
					and temporary files can be written. This is where the
					translated <tt>page-->java</tt> file will be created.
@param	log			a logging destination.

*/
public PageMgr(PageServlet servlet, File docroot, File scratchdir, Log log) 
	{
	Argcheck.notnull(docroot, "docroot parameter was null");
	
	this.log = log;
	this.servlet = servlet;
	
	if (! docroot.exists())
		throw new IllegalArgumentException("The specified docroot" + docroot + "] does not exist. How am I supposed to load pages, eh ?");

	if (! docroot.isDirectory())
		throw new IllegalArgumentException("The specified docroot [" + docroot + "] is not a directory. Give me your webpage directory, fool !");

	this.docroot = docroot;	
	this.scratchdir = scratchdir;

	this.docrootStr = docroot.getPath();	
	this.scratchdirStr = scratchdir.getPath();

	//we need to put /WEB-INF/classes, /WEB-INF/lib in the classpath too
	StringBuilder buf = new StringBuilder(1028);
	buf.append(System.getProperty("java.class.path"));
	buf.append(File.pathSeparator);
	File webinf =  new File(docroot, "WEB-INF");

	buf.append(new File(webinf,"classes").getAbsolutePath());
	File lib = new File(webinf, "lib");
	if (lib.exists()) {
		File[] list = lib.listFiles(new FilenameFilter() {
			public boolean accept(File f, String name) {
				return name.endsWith("zip") || name.endsWith("jar");
				}
			});
		for (int n = 0; n < list.length; n++) {
			buf.append(File.pathSeparator);
			buf.append(list[n].getAbsolutePath());
			}
		}
	classpath = buf.toString();

	log.info("Created new PageMgr. Using: \n\t\t\tdocroot:     ", docroot, 
			"\n\t\t\tscratchroot: ", scratchdir, 
			"\n\t\t\tclasspath:   ", classpath);
	}

/*
Internally called by pageservlet when it's unloaded. ensures that the
destroy() method of every loaded page is called.
*/
void destroy()
	{
	Iterator i = pagemap.values().iterator();
	while (i.hasNext()) {
		Page p = (Page) i.next();
		p.destroy();
		}
	}

/**
Returns the {@link Page} corresponding the the page path.

@args	contextRelativePagePath  path relative to the servlet context
								 (e.g.: "/foo/bar/baz.mp"), the leading
								 '/' is optional.
*/
public Page getPage(String contextRelativePath) throws Exception
	{
	Argcheck.notnull(contextRelativePath, "Internal error: the contextRelativePath parameter was null");
	Page page = null;

	final CharArrayWriter fbuf = new CharArrayWriter(128);

	//File docroot_pagefile = new File(docroot, contextRelativePagePath);
	//->micro-optimization
	fbuf.append(docrootStr).append(File.separator).append(contextRelativePath);	
	final File docroot_pagefile = new File(fbuf.toString());
	fbuf.reset();
	//->end mo

	if (! docroot_pagefile.exists()) {
		if (dbg) System.out.println("page: " + docroot_pagefile + " does not exist. Returning null page");
		return null;
		}

	final String pagefile  = StringUtil.fileName(contextRelativePath);

	/* this can be '/'  if page = /foo.mp or '' if page = foo.mp */
	final String pagedir   = StringUtil.dirName(contextRelativePath);
	
	if (!  (pagedir.equals("/") || pagedir.equals("") )) {
		//File pagedirFile = new File(scratchdir, pagedir);
		//->micro-optimization
		fbuf.append(scratchdirStr).append(File.separator).append(pagedir);
		File pagedirFile = new File(fbuf.toString());
		fbuf.reset();
		//->end mo
		
		//we need to create this directory otherwise print/file writers
		//that try to write a file within that directory will crap out.
		if (! pagedirFile.exists()) {
			if (dbg) System.out.println("Creating directory: " + pagedirFile);
			pagedirFile.mkdirs();
			}
		}
	
	String classname = getClassNameFromPageName(pagedir, pagefile);
	
	/*
	File javafile 	= new File(scratchdir, 
							pagedir + File.separator + classname + ".java");

	File classfile 	= new File(scratchdir, 
							pagedir + File.separator + classname + ".class");
	*/
	//->micro-optimization
	fbuf.append(scratchdirStr).append(pagedir).append(File.separator)
									.append(classname).append(".java");
	final File javafile = new File(fbuf.toString());
	fbuf.reset();

	fbuf.append(scratchdirStr).append(pagedir).append(File.separator)
									.append(classname).append(".class");
	final File classfile = new File(fbuf.toString());
	fbuf.reset();
	//->end mo

	if (dbg) {
		System.out.println(
			String.format("contextRelativePath=%s, pagedir=%s, pagefile=%s, javafile=%s, classfile=%s\n",
					contextRelativePath, pagedir, pagefile, javafile, classfile));
		}

	synchronized (this) 
		{		
		String src_encoding = null;
		long page_modified 	= docroot_pagefile.lastModified();
		long java_modified 	= javafile.lastModified();  //returns 0 if !exist
		long class_modified = classfile.lastModified();	//returns 0 if !exist

		if (dbg)
			{
			System.out.format(
			" %-20s %10d\n %-20s %10d\n %-20s %10d\n", 
					"Modified: page:", page_modified, 
					"java:", java_modified, 
					"class:", class_modified);
			}
		
		if ( java_modified == 0L || page_modified > java_modified) 
			{
			if (dbg) System.out.format("page_mod > java_mod, parsing the page.........\n");
			PageParser parser = new PageParser(
							docroot, docroot_pagefile, javafile, classname);

			log.info("PARSING page:", javafile.getPath());

			try {
				parser.parse();
				src_encoding = parser.getSourceEncoding();
				if (src_encoding == null) {
					src_encoding = Page.DEFAULT_SRC_ENCODING;
					}
				}
			catch (IOException e) {
				//the parser may write a partially/badly written file
				//if the parse failed.
				if (javafile.exists()) {
					javafile.delete();
					}
				throw e; //rethrow the parse exception
				}
				
			java_modified 	= javafile.lastModified();  //since newly parsed
			}
	
		boolean forceReload = false;
		//Java source could be generated or hacked by hand
		// if nothing needs compiling, then we still need to load the
		// page the first time it's accessed
		if ( class_modified == 0L || java_modified > class_modified) 
			{
			if (dbg) System.out.format("java_mod > class_mod, compiling the java source.........\n");
			
			log.info("COMPILING page:", javafile.getPath());
			
			//src_encoding can be null, that's fine.
			PageCompiler pc = new PageCompiler(javafile, classpath, src_encoding);
			
			if (! pc.compile())
				throw new ParseException(pc.getError());
			
			forceReload = true; //since we recompiled, we reload even if
								//page exists in the page cache
			}

		final boolean page_in_map = pagemap.containsKey(contextRelativePath);
		
		if (forceReload || ! page_in_map)
			{
			PageClassLoader loader = new PageClassLoader(/*scratchdir*/);
			Class c = loader.loadClass(
				Page.PACKAGE_NAME + "." + classfile.getCanonicalPath());
			
			if (dbg) System.out.println("class = " + c);
			page = (Page) c.newInstance();
			if (dbg) System.out.println("page = " + page);
			page.init(servlet, contextRelativePath);	
			
			//the pagemap uses contextRelativePath so that the
			//we store /foo/bar.mp and /bar.mp differently.
			//Also there should be a separate instance of 
			//PageServlet per context/WEB-INF/web.xml, so
			//different contexts will get different pageservlets
			//and hence different pagemgr's (and the pagemap
			//within each pagemgr will be different, hence
			//same path names within each context won't stomp
			//over each other).
			
			if (page_in_map) {
				Page oldpage = (Page) pagemap.get(contextRelativePath);
				oldpage.destroy();
				}
			//replace old page always
			pagemap.put(contextRelativePath, page);
			}
		else{
			page = (Page) pagemap.get(contextRelativePath);
			}
		}		
	
	if (dbg) log.bug("Returning PAGE=", page);
 	return page;
	}

/*
a.mp 	-> 	a_mp
b.mp 	-> 	b_mp
c.mp 	-> 	c_mp
p/q.mp  ->  p_q.mp
5!#.mp -> 	5__mp   ---> name clash 
5!!.mp ->   5__mp   ---> name clash

So a simplistic mapping(any bad classname chars--> '_') will not work.
We therefore hex-encode every special char.
*/
static String getClassNameFromPageName(String dir, String page) throws IOException
	{
	if (dbg) System.out.println("getClassNameFromPageName(): dir=["+dir+"]; page=["+page+"]");
		
	StringBuilder buf = new StringBuilder();
	char c;
	
	/*
	url=/$/y.mp  dir=/$/ or possibly $/  --> name HH_y.mp  [HH=hex]
 	url=/y.mp	 dir=/   				 --> name y.mp 
	we don't want our names to always start with _ because that's hokey
	*/
	if (dir.length() > 0)
		{
		boolean skip = false;
		c = dir.charAt(0);
		
		if (c == '/' || c == File.separatorChar)
			skip = true;			
		
		if (! skip)
			{
			if (! Character.isJavaIdentifierStart(c))
				buf.append(Integer.toHexString(c));
			else 
				buf.append(c);
			}
			
		if (dbg) System.out.println("buf3="+buf.toString());
		
		for (int n = 1; n < dir.length(); n++) 
			{
			c = dir.charAt(n);
			
			if (c == '/' || c == File.separatorChar)
				c = '_';
				
			if (! Character.isJavaIdentifierPart(c))
				buf.append(Integer.toHexString(c));
			else 
				buf.append(c);	
			
			if (dbg) System.out.println("buf4="+buf.toString());
			}
		}
		
	int dotpos = page.indexOf(".");
	if (dotpos != -1)
		page = page.substring(0,dotpos);
	
	c = page.charAt(0);

	if (! Character.isJavaIdentifierPart(c))
		buf.append(Integer.toHexString(c));
	else 
		buf.append(c);	
	
	for (int n = 1; n < page.length(); n++) 
		{
		c = page.charAt(n);
		if (Character.isJavaIdentifierPart(c)) {
			buf.append(c);
			}
		else{
			if (dbg) System.out.println(">>>>>>> " + c + " -> " + Integer.toHexString(c));
			buf.append(Integer.toHexString(c));
			}
		}
	
	buf.append("_mp");
	return buf.toString();
	}

		
/**
Interactive page manager use/testing.
*/
public static void main (String args[]) throws Exception
	{
	Args myargs = new Args(args);
	myargs.setUsage("java fc.web.page.PageMgr -docroot path-to-docroot-dir (. for cwd)] [-scratchroot path-to-scratchdir  (default .)]");
	String docroot = myargs.getRequired("docroot");
	String scratchroot = myargs.get("scratchroot", ".");
	PageMgr pagemgr = new PageMgr(null,
		new File(docroot), new File(scratchroot), Log.getDefault());
	
	//from a jdk techtip
    BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
  	String pagename = null;
	while (true) 
		{
		try {
			System.out.print("Enter LOAD <path-to-page>, RELOAD, GC, or QUIT: ");    
			String cmdRead = br.readLine();
			String[] toks = cmdRead.split("\\s+");
			String cmd = toks[0].toUpperCase();
			
			if (cmd.equals("QUIT")) {
				return;
				} 
			else if (cmd.equals("LOAD")) {
				pagename = toks[1];
				testLoad(pagemgr, pagename);
				}		
			else if (cmd.equals("RELOAD")) {
				if (pagename == null)
					System.out.println("Load a page first.....");
				else
					testLoad(pagemgr, pagename);
				}		
			else if (cmd.equals("GC")) {
				System.gc();
				}
			} //try
		catch (Throwable e) {
			e.printStackTrace();
			}
		} //while
	} //main

private static void testLoad(PageMgr pagemgr, String pagename) throws Exception
	{
	Page p = pagemgr.getPage(pagename);
	if (p != null) 
		System.out.println(ClassUtil.getClassLoaderInfo(p));
	else
		System.out.println("Could not load page. getPage() returned null");
	}

}
