// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import jakarta.servlet.*;
import jakarta.servlet.http.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;


/**
Validates that a group of fields is filled by the user. By default, every
field must be filled. However, optionally it can be specified that every
field in the group is not filled (all empty), only 1 field is filled or
1 or more fields (greater than zero) are filled.

@author hursh jain
**/
public final class VFilledGroup extends FormValidator
{
Field[] fields;

boolean	allFilledOrEmpty;
boolean onlyOneEmpty;
boolean onlyOneFilled;
boolean oneOrMoreFilled;

static final boolean dbg = false;


/** 
@param	fields 	a non-empty list containting fields to be 
				checked. 
**/
public VFilledGroup(Form f, String name, String errorMessage, List fields)
	{
	//we must make a copy of the fields because the caller
	//can change the list behind our back (by re-using the
	//list variable during form creation)
	
	this(f, name, errorMessage, 
		  (Field[]) fields.toArray(new Field[0]));
	}

/** 
@param	fields 	a non-empty array containting fields to be 
				checked. 
**/
public VFilledGroup(Form f,
 String name, String errorMessage, Field[] fields)
	{
	super(f, name, errorMessage);
	Argcheck.istrue(fields.length > 0, "specified field list was empty");
	this.fields = fields;
	allFilledOrEmpty = true;
	onlyOneEmpty = false;
	onlyOneFilled = false;
	oneOrMoreFilled = false;
	}

/**
Calling this method will result in requiring that all fields
must be filled except for any one empty field. Calling 
this method overrides any other methods that were called 
previously.
*/
public VFilledGroup onlyOneEmpty()
	{
	allFilledOrEmpty = false;
	onlyOneEmpty = true;
	onlyOneFilled = false;
	oneOrMoreFilled = false;

	return this;
	}

/**
Calling this method will result in requiring that all fields be
empty except for exactly one filled field (any field). Calling
this method overrides any other methods that were called
previously.
*/
public VFilledGroup onlyOneFilled()
	{
	allFilledOrEmpty = false;
	onlyOneEmpty = false;
	onlyOneFilled = true;
	oneOrMoreFilled = false;

	return this;
	}

/**
Calling this method will result in requiring that one or more
fields by filled (all the fields cannot be empty). Calling 
this method overrides any other methods that were called 
previously.
*/
public VFilledGroup oneOrMoreFilled()
	{
	allFilledOrEmpty = false;
	onlyOneEmpty = false;
	onlyOneFilled = false;
	oneOrMoreFilled = true;

	return this;
	}

/**
This is the default mode. Requires that all fields must be
filled. Calling this method overrides any other methods that were
called previously.
*/
public VFilledGroup allFilledOrEmpty()
	{
	allFilledOrEmpty = true;
	onlyOneEmpty = false;
	onlyOneFilled = false;
	oneOrMoreFilled = false;
	
	return this;
	}

public boolean validate(FormData fd, HttpServletRequest req) 
	{
	if (oneOrMoreFilled)
		return v_oneOrMoreFilled(fd);
	else if (onlyOneEmpty)
		return v_onlyOneEmpty(fd);	
	else if (onlyOneFilled)
		return v_onlyOneFilled(fd);
	else 
		return v_allFilledOrEmpty(fd);	
	}
	
private boolean v_oneOrMoreFilled(FormData fd)
	{
	int count = fields.length;
	for (int n = 0; n < count; n++) 
		{
		Field f = fields[n];
		if (f.isFilled(fd))
			return true;
		}
		
	return false;
	}
		

private boolean v_onlyOneFilled(FormData fd)
	{
	int count = fields.length;
	boolean one_filled = false;
		
	for (int n = 0; n < count; n++) 
		{
		Field f = fields[n];
		if (dbg) System.out.println("field " + f.name + " isFilled()=" + f.isFilled(fd));
		if (f.isFilled(fd)) 
			{  
			if (one_filled)
				return false;
			one_filled = true; //latch
			}
		}	
		
	return true;
	}	


private boolean v_onlyOneEmpty(FormData fd)
	{
	int count = fields.length;
	boolean one_empty = false;
		
	for (int n = 0; n < count; n++) 
		{
		Field f = fields[n];
		if (dbg) System.out.println("field " + f.name + " isFilled()=" + f.isFilled(fd));
		if (! f.isFilled(fd)) 
			{  
			if (one_empty)
				return false;
			one_empty = true; //latch
			}
		}	
		
	return true;	
	}	


private boolean v_allFilledOrEmpty(FormData fd)
	{
	int count = fields.length;
	boolean allempty = true;
	boolean allfilled = true;
	
	boolean	result = true;
	
	for (int n = 0; n < count; n++) 
		{
		Field f = fields[n];
		if (dbg) System.out.println("field " + f.name + " isFilled()=" + f.isFilled(fd));
		if (f.isFilled(fd)) { //field is filled 
			allempty = false; //latch
			}
		else{ //field is not filled
			//all othe fields must also be empty (we have to 
			//go thru the entire list)
			allfilled = false;   //latch   
			}
		}
	
	return (allempty || allfilled);
	}	


}
