// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import java.io.*;
import java.util.*;
import java.sql.*;
import jakarta.servlet.*;
import jakarta.servlet.http.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
An refreshable HTML Select field. This is different than {@link Select}
in that is allows the values/options displayed by this select to be
changed dynamically/per-user via the {@link #setValue(FormData, List)}
method.
<p>
There is no easy way to persistently keep track of these values which may
be different for each user. Therefore, unlike other field classes, this
class does (by default) no check to see if the returned values by the
user match at least one of the values displayed to the user in the first
place. This can happen if values are hacked on the client side.
<p>
Database constraints (that only accept valid values) should be used to
possibly handle bad data returned by this field.

@author hursh jain
**/

public final class RefreshableSelect extends Select
{
static final class Data 
	{
	//These are null by default but can override the options for the
	//select instance when set per user.This is useful when values are refreshed
	//every submit by re-querying the database. (we don't want to set new
	//values in the instance fields because that would effectively cause
	//all threads to synchronize rendering this field)
	private	LinkedHashMap	options; 
	private Map				origSelectedMap; 
	
	//To store selected options submitted by the user
	//acts like a set, used via containsKey() to see if an options
	//is selected (so we can reselect it) during rendering. 
	private Map  selectedMap = null;  //null implies no submitted data	
	
	//to store all options submitted by user
	private List selectedList = null;
	}

/** 
Creates a new select element with no initial values. Multiple selections
are not initially allowed.

@param 	name		the field name
**/
public RefreshableSelect(String name)
	{
	super(name);
	}
	
/** 
Creates a new select element with the specified initial values
and no multiple selections allowed. There is no explicit default
selection which typically means the browser will show the first
item in the list as the selection item.

@param 	name		the field name
@param	values		the initial option values for this class. The
					objects contained in the list must be of type
					{@link Select.Option} otherwise a 
					<tt>ClassCastException</tt> may be thrown in 
					future operations on this object.

@throws	IllegalArgumentException  if the values parameter was null
								  (note: an empty but non-null List is ok)
**/
public RefreshableSelect(String name, List values)
	{
	super(name);
	Argcheck.notnull(values, "values param was null");
	initOptions(values);
	}

public Field.Type getType() {
	return Field.Type.SELECT;
	}

/**
Sets the options for this select in the specified form data. This is
useful for showing different <i>initial</i> values to each user (before
the form has been submitted by that user).
<p>
If the form has not been submitted, there is no form data object. A form
data object should be manually created if needed for storing the value.

@param	fd		the non-null form data used for rendering the form
@param	values	a list of {@link Select.Option} objects
*/
public void setValue(FormData fd, final List values)
	{
	Argcheck.notnull(fd, "specified fd param was null");
	Argcheck.notnull(values, "specified values param was null");

	Data data = new Data();
	data.options = new LinkedHashMap();
	data.origSelectedMap = new HashMap();
	fd.putData(name, data);
	
	for (int n = 0; n < values.size(); n++) {
		Select.Option item = (Select.Option) values.get(n);
		String itemval = item.getValue();
		data.options.put(itemval, item);
		if (item.isOrigSelected()) {
			data.origSelectedMap.put(itemval, item);	
			}
		}		
	//System.out.println("data.options ==> " + data.options);	
	}


/**
<i>Convenience</i> method that sets the options for this select in the
specified form data. All of the initial options are used and option
corresponding to the specified value is selected. There must be some
initial options set for this field and the specified value should match
one of the initial options.
<p>
If the form has not been submitted, there is no form data object. A form
data object should be manually created if needed for storing the 
selected value.

@param	fd		the non-null form data used for rendering the form
@param	value	the option with this value is selected
*/
public void setSelectedValue(FormData fd, String value)
	{
	Argcheck.notnull(fd, "specified fd param was null");
	Argcheck.notnull(value, "specified selected param was null");

	Data data = new Data();
	data.options = new LinkedHashMap(options);
	data.origSelectedMap = new HashMap();
	fd.putData(name, data);
	
	Iterator it = data.options.values().iterator();
	while (it.hasNext())
		{
		Select.Option opt = (Select.Option) it.next();
		String optval = opt.getValue();
		if (optval.equals(value)) {
			data.origSelectedMap.put(optval, opt);	
			}
		}		
	}

/**
Convenience method that invokes {@link #setSelectedValue(fd, String)}
after converting the specified integer value to a string.
*/
public void setSelectedValue(FormData fd, int value) {
	setSelectedValue(fd, String.valueOf(value));
	}

/** 
Returns a List containing the selected options. Each object in
the collection will be of type {@link Select.Option}. If there
are no selected options, the returned list will be an
empty list.

@param	fd 	the submited form data. This object should not be null
			otherwise a NPE will be thrown.
**/
public List getValue(FormData fd) 
	{
	RefreshableSelect.Data data = (RefreshableSelect.Data) fd.getData(name);
	if (data == null) {
		return Form.empty_list;
		}
	return data.selectedList;
	}

/** 
Convenience method that returns the selected option as a
String. <u>No guarantees are made as to which selection is
returned when more than one selection is selected (this
method is really meant for when the select only allows
single selections as a dropdown).</u>
<p>
The returned value is of type String obtained by called the
selected Option's {@link Select.Option#getValue} method.
<p>
If there is no selection, returns <tt>null</tt>
**/
public String getStringValue(FormData fd) 
	{
	String result = null;
	
	RefreshableSelect.Data data = (RefreshableSelect.Data) fd.getData(name);
	if (data == null) {
		return null;
		}

	List list = data.selectedList;
	if (list == null || list.size() == 0) {
		return result;
		}
	else {
		Select.Option opt = (Select.Option) list.get(0);
		return opt.getValue();
		}
	}


/**
Convenience method that returns the single value of this field
as a Integer.
<p>
All the caveats of {@link #getSingleValueAsString()} apply.

@throws NumberFormatException	if the value could not be
								returned as in integer.	
*/
public int getIntValue(FormData fd) {
	return Integer.parseInt(getStringValue(fd));
	}

/**
Convenience method that returns the single value of this field
as a boolean. 
<p>
All the caveats of {@link #getSingleValueAsString()} apply.
In particular, the formdata should contain non-null data
with at least one selection.
*/
public boolean getBooleanValue(FormData fd) {
	return Boolean.valueOf(getStringValue(fd)).booleanValue();
	}

/**
Returns <tt>true</tt> if some option has been selected
by the user, <tt>false</tt> otherwise. (also returns
<tt>false</tt> is the specified form data is <tt>null</tt>).
*/
public boolean isFilled(FormData fd) 
	{
	if (fd == null)
		return false;
		
	RefreshableSelect.Data data = (RefreshableSelect.Data) fd.getData(name);
	if (data == null) {
		return false;
		}
	List list = data.selectedList;
	return (list != null && list.size() != 0);
	}

/*
 Select values can be present as:
 selectname=value

 or for multiple selections (for say a select called "sel1")
 sel1=value1&sel1=val2
 
 The values sent are those in the value tag in the option
 and if missing those of the corresponding html for the option.

 If a select allows multiple and none are selected, then
 nothing at all is sent (note this is different than for
 single (and not multiple) selects, where some value will always
 be sent (since something is always selected).

 We can track submitted/selected options in 2 ways:
 1. go thru the option list and set a select/unselect on each option
 element and then the option element renders itself appropriately.
 2. keep a separate list of selected elements and at render time display
 the item as selected only if it's also in the select list.

 (w) implies that we have to specify select/no select at
 render time to each option as opposed to setting that flag
 in the option before telling the option to render itself.
 (2) is chosen here.

 the client can send hacked 2 or more options with the same
 value this method will simply add any such duplicate values
 at the end of the list. this is ok.
*/
public void setValueFromSubmit(FormData fd, HttpServletRequest req) 
throws SubmitHackedException
	{
	//value(s) associated with the selection field name
	String[] values = req.getParameterValues(name);
	
	if (values == null) {
		return;
		}

	//lazy instantiation
	RefreshableSelect.Data data = new RefreshableSelect.Data();
	data.selectedMap = new HashMap();
	data.selectedList = new ArrayList();
	
	fd.putData(name, data);
	
	if (multiple && values.length > 1)
		hacklert(req, "recieved multiple values for a single value select");
		
	if (multiple) 
		{
		for (int n = 0; n < values.length; n++) {
			addSelectedOpt(req, data, values[n]);
			}
		}
	else{
		addSelectedOpt(req, data, values[0]);
		}
	}

private void addSelectedOpt(
 HttpServletRequest req, RefreshableSelect.Data data, String submitValue)
	{
	Select.Option opt = new Select.Option(submitValue);
	data.selectedMap.put(opt.getValue(), opt);
	data.selectedList.add(opt);
	}	

public void renderImpl(FormData fd, Writer writer) throws IOException
	{
	RefreshableSelect.Data data = null;

	Map options =  this.options;
	Map	origSelectedMap = this.origSelectedMap;
	
	if (fd != null)  { //we have submit or initial data
		data = (RefreshableSelect.Data) fd.getData(name); //can be null
		if (data != null) {
			if (data.options != null) { 
				/*per user options & origMap were set */			
				options = data.options;
				origSelectedMap = data.origSelectedMap;
				}
			}
		}
	
	writer.write("<select");	
	writer.write(" name='");
	writer.write(name);
	writer.write("'");
	
	if (size > 0) {
		writer.write(" size='");
		writer.write(String.valueOf(size));
		writer.write("'"); 
		}
		
	if (! enabled || ! isEnabled(fd)) {
		writer.write(" disabled");
		}

	if (multiple) {
		writer.write(" multiple"); 
		}	
		
	if (renderStyleTag) {
		writer.write(" style='");
		writer.write(styleTag);
		writer.write("'");
		}
		
	final int arlen = arbitraryString.size();
	for (int n = 0; n < arlen; n++) {
		writer.write(" ");
		writer.write(arbitraryString.get(n).toString());
		}
		
	writer.write(">\n");
	
	Iterator it = options.values().iterator(); 
	while (it.hasNext()) 
		{
	 	Select.Option item = (Select.Option) it.next();
	 	String itemval = item.getValue();
	 	
		boolean selected = false;
		
		if (fd != null) 	/* maintain submit state */
			{					
			if (data.selectedMap != null) { /*there was a submit*/  
				selected = data.selectedMap.containsKey(itemval);
				}
			else {  					/*initial data*/
				selected = origSelectedMap.containsKey(itemval);
				}
			}
			else{  	 /* form not submitted, show original state */
		  		selected = origSelectedMap.containsKey(itemval);
		   	}
		   
		writer.write(item.render(selected));
		writer.write("\n");	//sufficient for easy view source in browser		
		}	

	writer.write("</select>");
	}
	
/** 
Adds a new option to the selection list. Replaces any previous
option that was added previously with the same value.

@param	opt	the option to be added
**/
public void add(Select.Option opt) 
	{
	Argcheck.notnull(opt, "opt param was null");
	String optval = opt.getValue();
	options.put(optval, opt);
	if (opt.isOrigSelected()) {
		origSelectedMap.put(optval, opt);	
		}
	}	

/** 
Adds a new option to the selection list. Replaces any previous
option that was added previously with the same value. This method
will have the same effect as if the {@link #add(Select.Option) 
add(new Select.Option(item))} method was invoked.

@param	item the option to be added
**/
public void add(String item) 
	{
	Argcheck.notnull(item, "item param was null");
	Select.Option opt = new Select.Option(item);
	add(opt);	
	}	
	
/**
Clears all data in this select.
*/
public void reset()
	{
	options.clear(); 
	origSelectedMap.clear();
	}
	
/** 
This value (if set) is rendered as the html <tt>SIZE</tt> tag. 
If the list contains more options than specified by size, the browser 
will display the selection list with scrollbars.

@return	this object for method chaining convenience
**/
public Select setSize(int size) {
	this.size = size;
	return this;
	}	

/** 
<tt>true</tt> is multiple selections are allowed, <tt>false</tt> otherwise. 
This value (if set) is rendered as the html <tt>MULTIPLE</tt> tag.

@return	this object for method chaining convenience
**/
public Select allowMultiple(boolean allow) {
	this.multiple = allow;
	return this;
	}	
}          //~class Select