// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import jakarta.servlet.*;
import jakarta.servlet.http.*;
import java.io.*;
import java.util.*;
import java.sql.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
An object that contains form data submitted by a user. This object
represents the data submitted by a browser upon each time the form is
submitted. This object can then be passed along to various other methods
in the form framework as needed. This object can also be created manually
before the form is shown if default/initial values must differ per user
(say, when user must initially see their last filled value which is
retreived from a database).

@see {@link Form#handleSubmit handleSubmit}. 
@author hursh jain
*/
public final class FormData 
{
//The following 2 store the same info, one all the
//error messages, the other validator-->error message
//
//list of of form errors
List			formValidateErrorsList;
//form-validator-->error message
Map				formValidateErrorsMap;
//
//fieldname->list of errors 
Map				fieldValidateErrors; 
//submitted value of each field, fieldname->value
Map				fieldValues; 
//
Set				disabledFields;
//dynamically added fields
Map				dynamicFields;
//
boolean			dependencyUpdated = false;
//
Log				log;
//
Connection 		connection; /*remains null typically until set*/
//
Map				messages; //usually null
//
Map				validatedData; //mostly null

public FormData(Log log) {
	this.log = log;
	fieldValues = new HashMap();
	formValidateErrorsList = new ArrayList();
	formValidateErrorsMap = new HashMap();
	fieldValidateErrors = new HashMap();
	disabledFields = new HashSet();
	}

public FormData() {
	this(Log.getDefault());
	}
	
/* 
Adds a new validation error for the specified field 
*/
List createFieldErrorList(Field field) 
	{
	String name = field.name;
	List list = (List) fieldValidateErrors.get(name);
	if (list == null) {
		list = new ArrayList();
		fieldValidateErrors.put(name, list);
		}
	return list;
	}

/**
Returns a map containing all validation errors for all
fields [map: field->List_of_errors]
*/
Map getFieldErrors() {
	return fieldValidateErrors;
	}

/**
Returns a list containing all validation errors for the
specified field. If  
*/
List getFieldErrors(Field field) {
	return (List) fieldValidateErrors.get(field.name);
	}

void addFormError(String validatorName, String errmsg)
	{
	formValidateErrorsList.add(errmsg);
	
	if (validatorName != null)
		{
		List list = (List) formValidateErrorsMap.get(validatorName);
		if (list == null) {
			list = new ArrayList();
			formValidateErrorsMap.put(validatorName, list);		
			}	
		list.add(errmsg);
		}
	}

/**
Returns a list containing all form level validation errors
*/
Collection getFormErrorsCollection() {
	return formValidateErrorsList;
	}

/**
Returns a map containing all form level validation errors keyed by the
validator name. The value in the map is a List containing all errors for
that validator. If the error was added using Form.addError(fd, null, "foo"),
then the MAP won't have any data (since the key was null) but the form error
LIST will have the error.
*/
Map getFormErrorsMap() {
	return formValidateErrorsMap;
	}

void putData(String fieldname, Object obj) {
	fieldValues.put(fieldname, obj);
	}

//set since i keep forgetting whether it's set or put
void setData(String fieldname, Object obj) {
	fieldValues.put(fieldname, obj);
	}
		
Object getData(String fieldname) {
 	return fieldValues.get(fieldname);
	}

void enableField(String fieldname, boolean enable)	
	{
	if (enable)	{
		if (disabledFields.remove(fieldname));
		}
	else {
		disabledFields.add(fieldname);
		}
	}

boolean isEnabled(String fieldname)
	{
	return ! disabledFields.contains(fieldname);
	}


void addMessage(String key, String value) 
	{
	//no need to synchronize this, form processing is sequential/per-thread
	//for each fd.
	if (messages == null)
		messages = new HashMap();
	
	messages.put(key, value);
	}

String getMessage(String key) 
	{
	if (messages == null || ! messages.containsKey(key))
		return "";
	
	return (String) messages.get(key);
	}

//================== Public methods =============================
//For convenience. These can go in the form itself if need be and made 
//private here.

/**
Useful if strings entered by the user in the browser form are converted
to dates/times etc as part of the validation. Validators can then put
those date/time/whatever objects here and then those same conversions
don't have to be repeated at a later stage while saving the form out to 
a database.
*/
public void putValidatedData(String fieldname, Object obj)
	{
	if (validatedData == null)
		validatedData = new HashMap();
	
	validatedData.put(fieldname, obj);
	}

public Object getValidatedData(String fieldname)
	{
	return validatedData.get(fieldname);
	}



/**
Sets a connection to be used during this request. Meant primarily for 
refresher/dependent fields. 
*/
public void setConnection(Connection con) {
	this.connection = con;
	}

public Connection getConnection() {
	return connection;
	}
	
/**
Adds a {@link Hidden} field to this form. This field exists only for the
duration of this form data object. This field can be rendered.
*/
public void addDynamicField(Hidden hidden) {
	//no need to synchronize this, form processing is sequential & mem-vis
	//not a concern... i guess 
	if (dynamicFields == null)
		dynamicFields = new HashMap();
	
	hidden.dynamicallyAdded = true;
	dynamicFields.put(hidden.getName(), hidden);	
	}

/**
Gets a {@link Hidden} field added to this formdata object. Returns
<tt>null</tt> if that field does not exist.
*/
public Hidden getDynamicField(String fieldName) 
	{
	if (dynamicFields == null) {
		log.warn("Cannot obtain field:", fieldName, " from this formdata. This field is only available after being added to the formdata and is lost when the formdata object is lost, typically after the form is rendered.");		
		return null;
		}
	
	return (Hidden) dynamicFields.get(fieldName);	
	}

/**
Dependencies should call this method to set the dependencyUpdated flag to
true anytime there is a dependency change.
*/
public void setDependencyUpdated()
	{
	dependencyUpdated = true;
	}

public String toString()
	{
	return new ToString(this, ToString.Style.VisibleLevel.PRIVATE)
		.reflect().render();	
    }
 
}