// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import jakarta.servlet.*;
import jakarta.servlet.http.*;
import java.io.*;
import java.util.*;
import java.sql.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
Represents an HTML form. It is possible to instantiate a different form
object per user and store each form in that users session scope. However,
more typically, instances of this object are instantiated once per unique
form and stored in web application scope.
<p>
Method in this class are thread safe since form processing is done
entirely within various methods and the form processing results in the
creation of seperate {@link FormData} objects for each request.
<p>
Various {@link Field Fields} can be added to the form at form creation
time. Some fields (like {@link Hidden} can also be added per user/request, 
dynamically, to the {@link FormData} object. This is useful while saving
some user or request specific information.
<p>
A form collects data from the front-end/web-client and often saves it to a
back end database entity. Form data validation can be done in 2 ways:
<ol>
	<li>at the <i>form</i> level where the form collects data and validates
	it via validators associated with the form object.
	<li>at the <i>field</i> level where the form itself does no higher
	level validation but populates a field and then asks that field to
	validate itself.
</ol>
In both cases, any errors associated with the form can be shown back to the
user. This framework allows both types of validation methods to be used.
Using validation and/or showing validation errors is optional. This framework
can simply be used to maintain form state and validation can be done via
other ad-hoc code (outside of this framework) as needed.
<p>
An HTML form presented to the user can contain fields from more than 1
database table. For example, there could to 2 database tables, say
"person" and "office" both with a "other" column. Since <u>each field
contained in the form object must have a unique name</u>, there will be a
name clash between the fields/attributes of the entities when 2 fields
with the name "other" are added to the form.
<p>
There are 2 general approaches to solving this problem. 
<ol>
<li>Seperate database tables should be represented by seperate Form objects
(all of which can be rendered in the same HTML document shown to the user).
Different instances of form objects (say 1 per table) act as seperate
name-spaces and fields having the same name can exist in these seperate
instances. However, the dynamic server page where the form object is
rendered/submitted has to obtain these seperate instances from the application
object (or wherever all forms are stored) -- which adds some complexity.
<li>Create 1 instance of a form object containing fields from more than one
table. In this case, some or all field names can be prepended (or suffixed)
with the name of the database table. This removes any name clashes if done
only for clashing fields; if done for every field may help in documenting
which table that field belongs to. There is more typing involved with this
approach.
</ol>
Either of the above approaches is valid. 
<p>
Sometimes, other methods need to obtain a field (for a particular table) by
name. For example, the managers generated by {@link fc.jdbc.dbo.Generate}
have a <tt>addValidators(form)</tt> method that extract the fields for a
particular table and add validators to that field. This method in
particular need to be passed either the appropriate form object in which
that field exists (if there are multiple form objects) or in the case of
only 1 form object, passed the prefix for that particular field, if that
field was added with a prefix to it's name.

@author hursh jain
**/
public class Form 
{
//useful for method that return empty lists/maps in lieu of nulls
static final public List empty_list = 
	Collections.unmodifiableList(new ArrayList());

static final public Map empty_map = 
	Collections.unmodifiableMap(new HashMap());

protected 	Log 			log;
//fields have field-level validators	
protected 	LinkedHashMap		fields;
//collections of fields -- t	hese fields are directly added
//to the fields map also	
protected	LinkedHashMap		fieldRef; 
protected 	LinkedHashMap 		formValidators; //form-level validators
protected	List				dependencies;
protected 	String				name;			//the form name
protected	String				method;			
protected	String				actionURL;
protected	String				displayURL;
//protected	String				fieldNamePrefix;
protected 	java.util.Timer		timer;
protected	SubmitHackedHandler	hackHandler;
/** 
Creates a new form with the specified name, method (get/post) and URL
**/
public Form(String name, String method, String actionURL, Log log, 
			SubmitHackedHandler handler) 
	{ 
	this.name = name; 
	this.method	= method;
	this.actionURL = actionURL;
	fields = new LinkedHashMap();
	fieldRef = new LinkedHashMap();
	formValidators = new LinkedHashMap();
	dependencies = new ArrayList();
	this.log = log;

	this.hackHandler = handler;
	if (handler == null) {
		hackHandler = new SubmitHackedHandler(this);
		}
	}

/*
Creates a new form with the specified name, method (get/post) and URL
and the default {@link SubmitHackedHandler} 
**/
public Form(String name, String method, String actionURL, Log log)		 
	{
	this(name, method, actionURL, log, null);
	}

/** 
Creates a new form with logging to the default log retreived by
{@link Log#getDefault} and the default {@link SubmitHackedHandler}
**/
public Form(String name, String method, String actionURL) 
	{ 
	this(name, method, actionURL, Log.getDefault(), null);
	}

/** 
Creates a new form with the specified name, method=POST and no action URL
**/
public Form(String name) 
	{ 
	this(name, "post", null);
	}

/**
This method should be called when the form is removed from session or
application scope. It is very important to do this if there are background
update tasks because this method stops such background refresh timer thread
(which would otherwise continue to run).
*/
public void destroy()
	{
	timer.cancel();
	fields.clear();
	}

/** 
Removes the specified field from the form. See {@link
java.util.Map#remove(Object)} for the general contract.

@param	field	the field to remove from the form.
**/
public Object remove(Field field) { 
	return fields.remove(field.getName());
	}

/** 
Adds the specified element to the form. The added field can
later be retrieved via it's name. The field iteration order
will be the same order in which fields are added via this
method.
<p>
Note, field names have to be unique. "Grouped" fields (such
as HTML form checkboxes or radio buttons) which require the
same field name are encapsulated by a single appropriate
grouped field object such as {@link RadioGroup} and {@link
CheckboxGroup}. This field object itself should have a
unique name.

@return	 A reference to this form as a convenience for method chaining
@throws  IllegalArgumentException	if an element with the same name
									already exists in this form.
**/
public Form add(Field field) 
	{ 
	String name = field.getName();
	
	//not needed,  prefix each field manually if needed.
	/*
	if (fieldNamePrefix != null) {
		name = fieldNamePrefix + name;
		}
	*/
	if (fields.containsKey(name)) {
		throw new IllegalArgumentException("A form element by that name [" + name + "] already exists");
		}
	fields.put(name, field); 
	field.setParentForm(this);

	return this;
	}

/**
Adds the specified dependency to the form.
*/
public Form add(Dependency d) 
	{ 
	dependencies.add(d); 

	return this;
	}


/**
Adds a {@link FieldRef} to this Form. This is useful when
to track a set of related fields in a form. (Note: those fields
<b>must</b> also directly be added to the form).
*/
public Form add(FieldRef ref) 
	{
	String name = ref.getName();
	if (fieldRef.containsKey(name)) {
		throw new IllegalArgumentException("A fieldRef by that name [" + name+ "] already exists");
		}
	fieldRef.put(name, ref);
	return this;
	}

/**
Adds a <b>form level</b> validation error message. This method is needed when
form level validation is done via classes that are <b>not</b> subclasses
of {@link FormValidator} (these classes are used outside of the
validation framework and their error message still needs to be added to
the form data).
<p> 
<b> This is a very useful method for adding the results of arbitrary code
to this form. </b>

@param	fd			form data
@param	errorKey	an arbitrary/optional validation key which allows 
					to later retrieve and display a particular 
					form error. Specify <tt>null</tt> for no
					key.
@param	errorMessage the form error message

@see Field#addError
**/
public void addError(FormData fd, String errorKey, String errorMessage) 
	{
	fd.addFormError(errorKey, errorMessage);
	}

/**
Convenience method that invokes {@link addError(FormData, String, String) 
addError} with an empty validation <tt>errorKey</tt>. 
*/
public void addError(FormData fd, String errorMessage) 
	{
	addError(fd, null, errorMessage);
	}

/**
Adds a {@link Hidden} field to this form. This field exists only for the
duration of this form data object and is useful for request specific 
data.
*/
public void addDynamicField(FormData fd, Hidden f) {
	fd.addDynamicField(f);
	}

/** 
Adds a form level validator to this form. Note, validator
names have to be unique and not clash with other form level
validator names.

@throws  IllegalArgumentException	if a validator with the same name
									already exists in this form.
**/
public void addValidator(FormValidator validator) {
	String name = validator.getName();
	if (formValidators.containsKey(name)) {
		throw new IllegalArgumentException("A validator by that name [" + name + "] already exists in this form");
		}
	formValidators.put(name, validator);
	}

/**
Adds an arbitrary message for this request. Messages can include arbitrary
warnings, usage help etc. 
*/
public void addMessage(FormData fd, String key, String value) 
	{
	fd.addMessage(key, value);
	}

/**
Returns true if the form contains the specified field.
*/
public boolean containsField(String fieldName) 
	{
	return fields.containsKey(fieldName);	
	}

/**
Retrieves a previously added message or <tt>null</tt> if the speccified
form data was <tt>null</tt> or if the message was not found.
*/
public String getMessage(FormData fd, String key) 
	{
	if (fd == null)
		return null;
		
	return fd.getMessage(key);
	}
				
public void addSubmitHackedHandler(SubmitHackedHandler handler)
	{
	Argcheck.notnull(handler, "handler param was null");
	this.hackHandler = handler;
	}

protected void handleSubmitHacked(HttpServletRequest req, String msg)
throws SubmitHackedException
	{
	hackHandler.handle(req, msg);
	}
				
/** 
Processes the submitted form. <b>This method must be called after each form
submission by the servlet or page code that handles form submits</b>. This
method does the following:
<ol>
<li>populates the fields of the form from the submitted request.
<li>if the form has any dependencies and if the dependencies need updating,
then those dependencies are updated. Validation is not carried out in this
case since the form has changed and the user needs to typically carry out
some other action before submitting the form. 
<li>if there are no updated dependencies, validates the form via any
validators attached to the form and it's fields.
</ol>
Note: To see if there were any validation errors, invoke the {@link
hasError(FormData)} method with the formdata returned by this method.
Seeing validation errors is optional of course, and so is using validators
in the first place. 

@param		req		the HttpServletRequest corresponding to the request
					being processed by this form.

@return 	a {@link FormData} object containing populated 
			field data and the results of the validation.
**/	
public FormData handleSubmit(HttpServletRequest req)  
throws ServletException, IOException
	{
	return handleSubmit(req, null);
	}

/**
This method is similar to {@link handleSubmit(HttpServletRequest)} except
it stores the specified connection in the form data. This connection can
be used by any attached dependency classes for database queries when 
updating form depedencies.  
<p>
Note, we can set arbitrary objects in the {@link HttpServletRequest} via
it's {@link HttpServletRequest#setattribute setAttribute} method.
However, specifying and storing the connection in the FormData is easier
to remember and use.
<p>
There are 2 cases:
<ol>
<li>The dependent fields (if any) will internally use {@link WebApp#getConnection}
to obtain a connection and no connection has to be set in the form data.</li>
The {@link handleSubmit(HttpServletRequest)} can then be used. 
<li>The dependent fields (if any) classes will expect a connection in the form data. In
that case, this method should be invoked.</li>
</ol>
If there are no dependent fields, then the {@link handleSubmit(HttpServletRequest)} 
is sufficient. 

@param		req		the HttpServletRequest corresponding to the request
					being processed by this form.
@param		con		a connection object. The connection is not closed and
					it is the callers responsiblity to do so.
*/ 
public FormData handleSubmit(HttpServletRequest req, Connection con)  
throws ServletException, IOException
	{
	FormData fd = populate(req);  //each field sets it's submitted value
	if (con != null)
		fd.setConnection(con);
	
	updateDependencies(fd, req);  //this can throw an exception.
	
	if (! fd.dependencyUpdated) 
		validate(fd, req);
	
	return fd;
	}


public FormData refresh(HttpServletRequest req) 
throws ServletException, IOException
	{
	return refresh(req, null);
	}

/**
Refreshes the form for this request by running all attached refreshers
and dependencies. Maintains form state. Does not run any validatators.
Useful for re-displaying a form when lookup fields have changed in the
database.
<p>
<u>To refresh the form for all users/requests from this point onwards,
create a new instance of the form and replace the existing instance
of the form with the newly created instance.</u>
*/
public FormData refresh(HttpServletRequest req, Connection con) 
throws ServletException, IOException
	{
	FormData fd = populate(req);  //each field sets it's submitted value
	if (con != null)
		fd.setConnection(con);
	
	updateDependencies(fd, req);  //this can throw an exception.
	
	return fd;
	}
	
/**
Populates field's based on the parameters submitted via the specified
request and stores it into the specified FormData object. 
**/
protected FormData populate(HttpServletRequest req) 
throws SubmitHackedException
	{
	FormData fd = new FormData();
	Iterator it = fields.entrySet().iterator();
	while (it.hasNext()) {
		Map.Entry entry = (Map.Entry) it.next();
		Field f = (Field) entry.getValue();
		f.setValueFromSubmit(fd, req);		
		}
	return fd;
	}

protected void updateDependencies(FormData fd, HttpServletRequest req)
throws DependencyException
	{
	for (int i = 0; i < dependencies.size(); i++)
		{
		Dependency d = (Dependency) dependencies.get(i);
		d.updateValues(fd, req);
		}
	}
	
/** 
Validates each field and also calls each form level validator.
<p>
Validation is a 2-step process. First, only enabled fields are validated.
Some validators may enable or disable fields as part of their validation
strategy (for example, if the user chooses a radio button marked 'detail',
then fields corresponding to the detail section need to be enabled). (other
fields might be disabled). Fields that are re-enabled in this fashion are
validated by this method.
<p>
After running through all the validators, returns <tt>false</tt> is there
is a validation error, <tt>true</tt> otherwise. The field and form
validation errors can then be retrieved via the {@link
FormData#getFieldErrors} and {@link FormData#getFormErrorsCollection},
{@link FormData#getFormErrorsMap} methods.
**/
protected boolean validate(FormData fd, HttpServletRequest req) 
	{
	boolean result = true;
	Map errors = fd.getFieldErrors();
	Field f = null;
	Iterator it = null;
	//List checkAgainList = new ArrayList();
	
	//for all form validators
	it = formValidators.values().iterator();
	while (it.hasNext()) {
		FormValidator v = (FormValidator) it.next();
		if (! v.validate(fd, req)) {
			result = false; //latch
			String tmp = v.getErrorMessage();
			String vname = v.getName();
			log.bug("Group validate error: ", vname, tmp);
			fd.addFormError(vname, tmp);			
			}
		}
	
	//for all field validators, (some could actually be
	//disabled by the form validators above).
	it  = fields.values().iterator(); 
	while (it.hasNext()) 
		{
		f = (Field) it.next();
		
		if (f.isEnabled(fd)) 
			{ 
			if (! f.validate(fd, req)) {
				result = false; //latch
				}
			}	
		} //!while
	
	return result;	
	}

/**
Returns <tt>true</tt> if validation did not succeed.
*/
public static boolean hasError(FormData fd) 
	{
	if (fd == null)
		return false; //no data so could not validate
	
	//FieldErrors = map: fieldname->list of errors
	//entries in the map (via field.validate) should
	//only be created if there is a validation error
	//an empty list of errors should not be created
	//otherwise the size() below will not work
	if (fd.getFormErrorsCollection().size() > 0 ||
		fd.getFieldErrors().size() > 0) 
		{
		return true;
		}
	
	return false;
	}

/**
Convenience method to render <i>form-level</i> validation errors. This method
writes validation error message for the specified form level validator.
The error messages are encapulated in a html &lt;div&gt; tag and the class
of this tag is set to <tt>form-errmsg</tt>. Each error message within this
div is assigned it's own div with class set to <tt>form-errmsg-item</tt>

@param	writer			the Writer to render errors to
**/
public void renderError(FormData fd, Writer writer) 
throws IOException 
	{
	if (fd == null)
		return;
	
	Collection errors = fd.getFormErrorsCollection();

	int size = errors.size();
	if (size == 0)
		return;
	
	writer.write("<div class='form-errmsg'>\n");
	Iterator it = errors.iterator();
	while (it.hasNext())
		{
		String msg = (String) it.next();
		if (msg == null)
			continue;
		writer.write("<div class='form-errmsg-item'>");
		writer.write(msg);
		writer.write("</div>\n");
		}
	writer.write("</div>\n");
	}


/**
Convenience method to render a validation error for a particluar form level
validator. The error messages are encapulated in a html &lt;div&gt; tag and
the class of this tag is set to <tt>form-errmsg</tt>.

@param	writer				the Writer to render errors to
@param	formValidatorName	the name of a form level validator
**/
public void renderErrorFor(
	FormData fd, Writer writer, String formValidatorName) 
throws IOException 
	{
	if (fd == null)
		return;
	
	List errors = (List) getFormErrorsFor(fd, formValidatorName);
	
	int size = errors.size();
	if (size == 0)
		return;
	
	writer.write("<div class='form-errmsg'>\n");
	Iterator it = errors.iterator();
	while (it.hasNext())
		{
		String msg = (String) it.next();
		if (msg == null)
			continue;
		writer.write("<div class='form-errmsg-item'>");
		writer.write(msg);
		writer.write("</div>\n");
		}
	writer.write("</div>\n");
	}

/** 
@param	 fd	a non-null form data object
@return 	a map containing field names as keys and the 
			field's validation error as the value. The data
			type of each value is that returned by the {@link
			Field#getValidateErrors} method
**/
public Map getFieldErrors(FormData fd) {
	return fd.getFieldErrors();
	}

/** 
@param	 fd	a non-null form data object
@return 	a list containing form level errors. The data type
			of each value is that returned by the {@link
			FormValidator#getErrorMessage()} method
**/
public Collection getFormErrorsCollection(FormData fd) {
	return fd.getFormErrorsCollection();
	}

/**
Returns a list containing all form-level errors for the specified
validator. If there are no errors for the specified validator, returns an
empty list.

@param	 fd					a non-null form data object
@param	 formValidatorName  a form validator name
*/
public List getFormErrorsFor(FormData fd, String formValidatorName) 
	{
	Map map = fd.getFormErrorsMap();
	List list = (List) map.get(formValidatorName);
	if (list == null)
		return empty_list;
		
	return list;
	}
	
/**
Renders <b>all</b> dynamically added hidden fields of the form. Dynamic
fields are created/ added per user per submit to the {@link FormData}
if/as needed.
<p>
<b>Note</b>: Dynamic fields are <b>not</b> part of a form's fields and
are not available via 
*/
public void renderDynamicFields(FormData fd, Writer out) 
throws SQLException, IOException
	{
	if (fd == null || fd.dynamicFields == null)
		return;

	Iterator it = fd.dynamicFields.values().iterator();
	while (it.hasNext()) {
		Hidden h = (Hidden) it.next();
		h.render(out);
		}
	}

/** Returns the name of this form **/
public String getName() { 
	return name; 
	}

/** 
Sets an arbitrary application specific string that can 
later be retrieved. Some applications may find 
this method useful.
**/
public void setActionURL(String url) {
	this.actionURL = url;
	}

/** 
Returns the action URL string. Returns <tt>null</tt>
if no url has been set.
**/	
public String getActionURL() {
	return actionURL;
	}


/** 
Sets an arbitrary application specific string. Intended
to store the URL to the jsp or front end page that will
render this form for the user.
**/
public void setDisplayURL(String url) {
	this.displayURL = url;
	}

/** 
Returns the display URL string. Returns <tt>null</tt>
if no display url has been set.
**/	
public String getDisplayURL() {
	return displayURL;
	}


/** 
Returns a collection view of all the fields contained in this
form.
**/
public Collection getAllFields() { 
	return fields.values(); 
	}


/*
*This is overkill, manually prefix fields as needed
*
*Automatically appends the specified prefix to the names of all fields
*added after this method is called. Suppose there are 2 tables contained in
*1 form and both tables contain a field called <tt>id</tt>. Then, since
*field names in the form must be unique, one can either add those fields
*under different names (say, "table1.id", and "table2.id") or can do the
*following:
*<blockquote>
*form.setFieldNamePrefix("table1");
*form.add(id);
*form.setFieldNamePrefix("table2");
*form.add(id);
*</blockquote>
*This latter way is useful when many field names clash.
*
*@param	prefix	the prefix string. Specify <tt>null</tt>
*				or the empty string to have no prefix.
*				(<tt>null</tt> is more efficient)
*
*public void setFieldNamePrefix(String prefix) {
*	if ( (prefix == null) || prefix.equals("") ) 
*		{
*		this.fieldNamePrefix = null;
*		return;
*		}	
*	this.fieldNamePrefix = prefix + "_";
*	}
*/

//get various Field types
/**
Returns the form field with the specified name.

@param		name	the name of the field

@return		the field or <tt>null</tt> if no element with that name 
			was found in this form. 
**/
public Field get(String name) 
	{
	Field result = (Field) fields.get(name);	
	
	if (result == null) 
		log.warn("Tried to obtain a non-existent field from the form. Did you add a field with this name to the form ? Fieldname: [", name, "]");
		
	return result;
	}

/**
Gets a {@link Hidden} field added to the form data object. Returns
<tt>null</tt> if the form data object is null or if the specified
field does not exist. <u>Dyanamic fields returned by this method are only 
available for the duration of the form data object and <u>can not
obtain a field from a submitted form.</u> (use {@link getDynamicField(FormData, HttpServletRequest)}
for that purpose.
*/
public Hidden getDynamicField(FormData fd, String fieldName) 
	{
	if (fd == null)
		return null;

	return fd.getDynamicField(fieldName);
	}


/**
Once the form is displayed to the user (with, say, hidden fields that
were added dynamically for that request), then when that form is
submitted, those dynamic fields can be retrieved via any of: 
<ul> 
<li> the {@link HttpServletRequest#getParameter} method (often the easiest
way)
<li> the {@link #getSubmittedField(HttpServletRequest)} method (which will
attempt to read the request parameter with the specified name and return
is as a Hidden field. (this is slightly more convenient if there is more
than 1 hidden field with the same name, since all hidden fields with the
same name are encapsulated by one object of type {@link Hidden}).
</ul>
*/
public Hidden getSubmittedField(HttpServletRequest req, String name) 
	{
	String str = req.getParameter(name);

	if (str == null)
		str = "";

	Hidden hidden = new Hidden(name, str);
	hidden.dynamicallyAdded = true;
	return hidden;
	}

/**
Returns all the fields that <b>contains</b> the specified name. This is
useful when a bunch of fields are created via database lookup tables (and
each such field is prefixed/suffixed by say <tt>lookup_name + some
number</tt>).

@param		name	the name of the field

@return		a List containing fields with matching names or or an empty list
			if no elements containing that name were found in this form.
**/
public List getContaining(String name) 
	{
	final List list = new ArrayList();
	final Iterator it = fields.values().iterator();
	while (it.hasNext())
		{
		final Field f = (Field) it.next();
		if ( f.getName().indexOf(name) >= 0 )
			list.add(f);
		}
	return list;
	}

/**
Returns the {@link FieldRef} with the specified key. This is
used to retrieve an arbitrary object added the via {@link
#addObject} method. Returns <tt>null</tt> if no object is
found for the specified key.
*/
public FieldRef getFieldRef(String key) 
	{
	return (FieldRef) fieldRef.get(key); 
	}

public java.util.Timer getTimer() 
	{
	synchronized (this) {
		if (timer == null) {
			timer = new java.util.Timer();
			}
		}
		
	return timer;
	}

/** 
Returns the field with the specific name as a {@link Select}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Select
**/
public Select getSelect(String name)
	{
	return (Select) get(name);
	}

/** 
Returns the field with the specific name as a {@link RefreshableSelect}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type RefreshableSelect
**/
public RefreshableSelect getUpdatableSelect(String name)
	{
	return (RefreshableSelect) get(name);
	}



/** 
Returns the field with the specific name as a {@link Checkbox}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Checkbox
**/
public Checkbox getCheckbox(String name)
	{
	return (Checkbox) get(name);
	}

/** 
Returns the field with the specific name as a {@link
Checkbox}. 

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Checkbox
**/
public CheckboxGroup getCheckboxGroup(String name)
	{
	return (CheckboxGroup) get(name);
	}

/** 
Returns the field with the specific name as a {@link
Hidden}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Hidden
**/
public Hidden getHidden(String name)
	{
	return (Hidden) get(name);
	}

/** 
Returns the field with the specific name as a {@link Text}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Text
**/
public Text getText(String name)
	{
	return (Text) get(name);
	}

/** 
Returns the field with the specific name as a {@link
TextArea}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type TextArea
**/
public TextArea getTextArea(String name)
	{
	return (TextArea) get(name);
	}

/** 
Returns the field with the specific name as a {@link
Password}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Password
**/
public Password getPassword(String name)
	{
	return (Password) get(name);
	}

/** 
Returns the field with the specific name as a {@link Radio}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Radio
**/
public Radio getRadio(String name)
	{
	return (Radio) get(name);
	}

/** 
Returns the field with the specific name as a {@link
Checkbox}.

@throws IllegalArgumentException if a field with the specified name does not exist or
								 if the field exists but is not of type Checkbox
**/
public RadioGroup getRadioGroup(String name)
	{
	return (RadioGroup) get(name);
	}
	
public String toString() 
	{	
	final int border_len = 70;
	final String nl = fc.io.IOUtil.LINE_SEP;

	final int formval_size = formValidators.size();
	final int fieldref_size = fieldRef.size();
	StringBuffer buf = new StringBuffer(256);
	StringBuffer tmp = null;
	
	buf.append(nl);
	buf.append(StringUtil.repeat('-', border_len));
	buf .append(nl)
		.append("Form Name: ")
		.append(name)
		.append(" [")
		.append(fields.size())
		.append(" Fields]")
		.append(" [")
		.append(formval_size)
		.append(" FormValidators]")
		.append(" [")
		.append(fieldref_size)
		.append(" FieldRefs]");

		 if (timer != null) {
			buf.append(" [Attached Timer: ")
			   .append(timer)
			   .append(" ]");
			}
			
		buf.append(nl);
		
		
	ByteArrayOutputStream out = null;
	TablePrinter.PrintConfig config = null;
	TablePrinter p = null;
	
	
	out = new ByteArrayOutputStream(2048);
	config = new TablePrinter.PrintConfig();
	config.setPrintBorders(false);
	config.setCellSpacing(1);
	config.setCellPadding(0);
	config.setAutoFit(true);
	p = new TablePrinter(3, new PrintStream(out), config);
	p.startTable();

	p.startRow();
	p.printCell("FIELD");
	p.printCell("TYPE");
	p.printCell("FIELD_VALIDATORS");
	p.endRow();

	Iterator it = fields.values().iterator();
	while(it.hasNext()) 
		{
		Field f = (Field) it.next();
		p.startRow();
		p.printCell(f.getName());
		p.printCell(f.getType().toString());
		List list = f.getValidators();
		int size = list.size();
		if (size == 0) {
			p.printCell("-");
			}
		else{
			tmp = new StringBuffer(256);
			for (int n = 0; n < size; n++) 
				{
				if (n != 0)
					tmp.append(", ");
				FieldValidator fv = (FieldValidator) list.get(n);
				if (fv instanceof VText) {
					tmp.append(fv); //prints a bit more info.
					}
				else{
					String s = fv.getClass().getName();
					tmp.append(s.substring((s.lastIndexOf(".") + 1), s.length()));
					}
				}
			p.printCell(tmp.toString());
			}	
		p.endRow();
		}
	
	p.endTable();
	buf.append(out.toString());

	if (formval_size > 0) 
		{
		buf.append(nl);	
		
		out.reset();
		config = new TablePrinter.PrintConfig();
		config.setPrintBorders(false);
		config.setCellSpacing(1);
		config.setCellPadding(0);
		config.setAutoFit(true);
		p = new TablePrinter(1, new PrintStream(out), config);
		p.startTable();

		p.startRow();
		p.printCell("Form-Validators....");
		p.endRow();
		
		it = formValidators.entrySet().iterator();
		while (it.hasNext())
			{
			p.startRow();
			Map.Entry me = (Map.Entry) it.next();
			String key = (String) me.getKey();
			tmp = new StringBuffer(tmp.length() + 2);
			tmp.append("\t");
			tmp.append(key);
			p.printCell(tmp.toString());
			p.endRow();
			}
		
		p.endTable();
		buf.append(out.toString());
		}

	if (fieldref_size > 0) 
		{
		buf.append(nl);	
		
		out.reset();
		config = new TablePrinter.PrintConfig();
		config.setPrintBorders(false);
		config.setCellSpacing(1);
		config.setCellPadding(0);
		config.setAutoFit(true);
		p = new TablePrinter(2, new PrintStream(out), config);
		p.startTable();

		p.startRow();
		p.printCell("Field References....");
		p.printCell("");
		p.endRow();
		
		it = fieldRef.values().iterator();
		while (it.hasNext())
			{
			p.startRow();
			FieldRef fr = (FieldRef) it.next();
			tmp = new StringBuffer(tmp.length() + 2);
			tmp.append("\t");
			tmp.append(fr.getName());
			p.printCell(tmp.toString());
			tmp = new StringBuffer(128);
			if (fr.getList() != null) {
				tmp.append(fr.getList().size());
				tmp.append(" fields");
				}
			else if (fr.getMap() != null) {
				tmp.append(fr.getMap().size());
				tmp.append(" fields");
				}
			else {
				tmp.append(fr.getObject());
				}
			p.printCell(tmp.toString());
			p.endRow();
			}
		
		p.endTable();
		buf.append(out.toString());
		}

	buf.append(StringUtil.repeat('-', border_len));
	return buf.toString();
	}

}          //~class Form