// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import jakarta.servlet.*;
import jakarta.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
Abstracts an HTML choice type such as choicebox or radio.
Does <b>not</b> abstract a HTML select element for which the
{@link Select} class should be used. Concrete subclasses
represent a particular choice element such as {@link
Checkbox} and {@link Radio}

@author hursh jain
**/
public abstract class Choice extends Field
{
//if this choice object's original state == selected
private boolean orig_selected;
//can be null if not specified
private String 	value;  

static class Data {
	//the value actually submitted by the browser (could
	//be 'on', 'ON' etc., or the value for that option (if set) 
	private String 	submit_value;
	}

/** 
Creates a new choice object.

@param 	name		the field name
@param 	value		the value of this choice item (can be null
					for unspecified)
@param	selected	<tt>true</tt> is this choice is 
					originally selected
**/
protected Choice(String name, String value, boolean selected)
	{
	super(name);
	this.value = value;
	this.orig_selected = selected;
	}

public abstract Field.Type getType();

/** 
Returns the current <b>value</b> of this field. This can be:
<blockquote>
	<li><tt>null</tt>: if no value is currently set which can happen if
	this field was not selected when the parent form was submitted.
	Browsers send nothing at all if choice type fields are not selected
	in an HTML form.
	<li>the value attribute of this field (if this choice field was
	created/displayed with a value attribute) or the string "on" (which
	is sent by browsers if there is no specific value attribute for this
	choice field). Note, the default value should be treated as case
	<u>in</u>sensitive, since browsers can send <tt>on</tt>, <tt>ON</tt>
	etc.
</blockquote>
**/
public String getValue(FormData fd) 
	{
	Choice.Data data = (Choice.Data) fd.getData(name);

	if (data == null)
		return null;

	return data.submit_value;
	}

/**
Sets the <u>selected or non-selected state</u> for this choice in the
specified form data. Selected choices are returned by the browser as
<tt>on, ON, oN</tt> etc., the value returned is not important as long as
something is returned. Therefore <i>any non-null value</i> set by this method
will have the effect of selecting this choice when it is rendered. A <tt>
null</tt> value wil unselect this choice.

@param	fd		a non-null form data object
@param	value	any non-null value
*/
public void setValue(FormData fd, String value)
	{
	Argcheck.notnull(fd, "specified fd param was null");
	
	if (value == null)
		return;
		
	Choice.Data data = new Choice.Data();
	fd.putData(name, data);
	data.submit_value = value;	
	}

/**
Convenience method that sets this choice to be selected/non-selected.

@param	fd			a non-null form data object
@param	selected 	true to select this choice, false otherwise
*/
public void setValue(FormData fd, boolean selected)
	{
	Argcheck.notnull(fd, "specified fd param was null");
		
	if (selected) {
		Choice.Data data = new Choice.Data();
		fd.putData(name, data);
 		//not submit_value = value since value can be null		
 		data.submit_value = "on";
		}
	}

/**
Sets the value for this choice.
*/
public void setValue(String value, boolean selected)
	{
	Argcheck.notnull(value, "specified value was null");
	
	this.value = value;	
	this.orig_selected = selected;
	}

/**
Convenience method that returns the value of this 
field as a Integer. 

@throws NumberFormatException	if the value could not be
								returned as in integer.	
*/
public int getIntValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Integer.parseInt(value);
	}

/**
Convenience method that returns the value of this 
field as a Short. 

@throws NumberFormatException	if the value could not be
								returned as a short.	
*/
public short getShortValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Short.parseShort(value);
	}


/**
Convenience method that returns the value of this field as a boolean.
<u>The returned value will be <tt>true</tt> if the submitted value is
"true" or "on" (both case insensitive), else <tt>false</tt></u>
*/
public boolean getBooleanValue(FormData fd) 
	{
	String value = getValue(fd);
	if (value == null)
		return false;
	
	if (value.equalsIgnoreCase("on"))
		value = "true";
		
	return Boolean.valueOf(value).booleanValue();
	}

//the value actually submitted by the browser (could
//be 'on', 'ON' etc., or the value for that option if set 
public void setValueFromSubmit(FormData fd, HttpServletRequest req) 
throws SubmitHackedException	
	{	
	String submittedValue = req.getParameter(name);
	
	//choice was not selected 
	if (submittedValue == null)
		return;
		
	Choice.Data data = new Choice.Data();
	fd.putData(name, data);
	
	if (value != null && ! value.equals(submittedValue))
		hacklert(req, "submitted value: [" + submittedValue + "] does not equal the value [" + value + "]of this field.");
		
	data.submit_value = submittedValue;	
	//log.bug(name, "setting data=", submittedValue);
	}

public void renderImpl(FormData fd, Writer writer) throws IOException
	{
	boolean selected = false;
	
	//no formdata, rendering first time
	if (fd == null) {
		selected = this.orig_selected;
		}
	else {  //submit or initial data
		String returned_value = getValue(fd); //can be null	
		if (returned_value != null) 
			selected = true;
		}

	writer.write("<input type='");
	writer.write(getType().toString());
	writer.write("' name='");
	writer.write(name);
	writer.write("'");
	
	if (value != null) {
		//we always write the orig. value, not the returned
		//one (since the returned value should be the same
		//as the original value unless hacked)
		writer.write(" value='");
		writer.write(value);
		writer.write("'"); 
		}

	if (! enabled || ! isEnabled(fd)) {
		writer.write(" disabled");
		}
	
	if (selected) {
		writer.write(" checked");
		}
		
	if (renderStyleTag) {
		writer.write(" style='");
		writer.write(styleTag);
		writer.write("'");
		}
		
	final int arlen = arbitraryString.size();
	for (int n = 0; n < arlen; n++) {
		writer.write(" ");
		writer.write(arbitraryString.get(n).toString());
		}
		
	writer.write(">");
	writer.write("</input>");
	}


/** 
@return 	<tt>true</tt> if this field is selected, 
			<tt>false</tt> otherwise
**/
public boolean isFilled(FormData fd) 
	{
	return getValue(fd) != null;
	}

/**
Sets the <b>initial</b> selection status for this field.

@param	select 	<tt>true</tt> if this field should be selected 
				<tt>false</tt> otherwise.
**/
public void setSelected(boolean select) {
	this.orig_selected = select;
	}	


public String toString() 
	{
	return super.toString() + 
		"; Orig. value: [" + value + "]" +
		"; Orig. selected: [" + orig_selected + "]";
	}	

}       