// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util.cache;	

import java.io.*;
import java.util.*;

//	-cache hits, total, per-item, expired amount, average age
//	-onexpire, onload


/** 
A data cache. This is essentially a wrapper over a collection that allows
for objects to be cached for a specified amount of time and expired
thereafter. Objects in the cache can expire automatically or 
can be removed manually.

@author 	hursh jain
@version 	1.0 12/29/2001
**/
public interface Cache 
{
/**
Empties the entire cache by expiring and removing all objects.
*/
void clear();

/* 
this is not needed, and implementations would have to convert all existing
cachedobject into a map of key,val pairs, because key,val pairs would
probably not be stored directly but wrapped around objects that also
maintain expire state 

Returns a {@link java.util.Map} consisting of all cached items. 
Implementations are required to return a <tt>Map</tt>,
regardless of their internal design.
*/
//public Map getCacheMap();


/**
Returns true if the cache has a valid cached item specified by the
key. If no item is found for that key or if the item has expired,
returns <tt>false</tt>.

@param	key		typically a <tt>String</tt> but can be any object
**/
public boolean containsKey(Object key);


/**
Returns the cached item specified by the key. If no item is found for that
key or if the item has expired, returns <tt>null</tt>.

@param	key		typically a <tt>String</tt> but can be any object
**/
public Object get(Object key);

/** 
Returns all objects in the cache. The returned map is implementation specific
but will typically contain <key,item> pairs of all objects in the cache. The
item might be encapsulated in another implementation-specific class.
<p>
Modification to this map will affect this cache.
**/
public Map getAll();

/**
Puts the specified object into the cache, mapped with the specified key.
The object stays in the cache for the specified amount of time, after which
it is discarded. If that key already contains a value, then that value is
returned by this method (and also replaced by the new value).

@param	key		key with which the specified value is to be associated.
@param	val 	value to be associated with the specified key.
@param  expiry 	time in <b>milliseconds</b> (from the time the
				object is put into the cache) before it is expired. Specify
				<tt>-1</tt> for the <tt>item</tt> to never expire.
**/
public Object put(Object key, Object val, long expiry);

/**
Convenience method that puts an object into the cache that expires in
the default TTL. The default TTL can be get/set by the {@link #setDefaultTTL} 
method and subclasses are free to implement a default TTL as they see fit.
If that key already contains a value, then that value is returned by this 
method (and also replaced by the new value).
**/
public Object put(Object key, Object item);

/**
Returns the amount of time left (in milliseconds) before the object will be
expired from the cache. 
<p>If the object specified by the key is <i>not</i> found in the cache, 
then <tt>0</tt> is returned.
<p>If the object has already expired, then <tt>0</tt> is returned.
<p>If the object will never expire, then <tt>-1</tt> is returned.

@param key	the key with which to find the object under consideration.
**/
public long getTimeLeft(Object key);

/**
Expires immediately the object specified by the key. Overrides any previous
setting for that object's expire time.

@param key	the key with which to find the object under consideration.
**/
void expire(Object key);

/**
Adds the specified amount of time to the object (to it's current time left)
before that object will be expired. Overrides any previous setting that the
object may have had, such as never expiring. Specify <tt>-1</tt> for that
object to never expire. Does not work for objects that have already
expired.

@param key 			the key with which to find the object 
					under consideration.
@param extendTime	exntension time in milliseconds, valid 
					values are >= 0 and -1
**/
void extend(Object key, long extendTime);

/**
Sets the default TTL in milliseconds.
*/
void setDefaultTTL(long millis);

/**
Gets the default TTL in milliseconds.
*/
long getDefaultTTL();

/**
Closes this cache, which makes all items in the cache unavailable. Any
items needed for later should be taken out of the cache before closing it.
**/
public void close();

/**
Returns true if this cache has been closed, false otherwise.
**/
public boolean isClosed();
}