// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util.cache;	

import java.text.*;
import java.util.*;
import fc.io.*;
import fc.util.*;

/**
In memory cache with the ability to specify an item bound/cache-limit 
functionality. By default,the upper bound is Integer.MAX_VALUE.
<p>
Existing items are not automatically expired. They are only removed when a new
item is added and the bound size is crossed. Which item is removed is according to BoundedCache.POLICY.
<p>
ThreadSafety: This class <b>is</b> thread safe and can be used by multiple 
threads concurrently.

@author 	hursh jain
@version 	1.0 7/16/2010
**/
public final class BoundedCache
{
/**
least_used, fifo
*/
public enum Policy
	{	
	LEAST_USED,
	FIFO;
	}

final		String 		myName;
			Map 		cache;
volatile 	boolean 	closed = false;
final		Log			log;
final		Policy 		cachePolicy;
			int			bound;

/**
Instantiates this class with the specified name, logger, policy and cache size.

@param  name		String denoting the name for this object
@param	logger		a reference to a {@link fc.io.Log}
@param  cachePolicy	item removal policy when cache is full
@param	bound		the upper bound of this cache.
**/
public BoundedCache(Log log, String name, Policy cachePolicy, int bound) 
	{
	if (log == null) {
		log = Log.getDefault();
		}
		
	this.log = log;
	this.myName = name;
	this.cachePolicy = cachePolicy;
	this.bound = bound;
	
	if (cachePolicy == Policy.LEAST_USED) {
		this.cache = Collections.synchronizedMap(new MyLinkedHashMap(
														bound, 0.75f, true));
		}
	else if (cachePolicy == Policy.FIFO) {
		this.cache = Collections.synchronizedMap(new MyLinkedHashMap(
														bound, 0.75f, false));
		}
	else {
		throw new IllegalArgumentException("Do not understand this cache policy: " + cachePolicy);
		}
	}

/**
Creates a memory cache with a system-assigned name, logger and the specified
policy and bound.
*/
public BoundedCache(Policy cachePolicy, int bound) 
	{
	this(null, 
		"BoundedCache/created@" 
		+ DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT)
			.format(new Date()), 
		cachePolicy, 
		bound
		);
	}

/**
Sets the upper bound of this cache.
*/
public void setBound(int items) {
	this.bound = items;
	}

/**
Gets the current bound of this cache.
*/
public long getBound() {
	return bound;
	}

/**
Removes the object specified by the key.
*/
public void expire(java.lang.Object key)
	{
	synchronized (cache) {
		cache.remove(key);
		}
	}

/**
Returns the item for the specified key or <tt>null</tt> if the item does 
not exist.
*/
public Object containsKey(Object key) 
	{
	Argcheck.isfalse(isClosed(), "Cache has been closed");

	Object item = null;
	
	synchronized (cache) {
		item = cache.get(key);	
		}
		
	return item != null;
	}
	
	
/**
Returns the item for the specified key or <tt>null</tt> if the item does 
not exist.
*/
public Object get(Object key) 
	{
	Argcheck.isfalse(isClosed(), "Cache has been closed");

	Object item = null;
	
	synchronized (cache) {
		item = cache.get(key);	
		}
		
	return item;
	}

/**
Returns the underlying storage read-only map for this cache.
*/
public Map getAll() {
	return Collections.unmodifiableMap(cache);
	}

/**
Puts the item for the specified key. Returns the previous item for this key 
or <tt>null</tt> if no previous item existed.
*/
public Object put(Object key, Object val)
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");
		
	Object item = null;
	
	synchronized (cache) {
	 	item = cache.put(key, val);			
		}
		
	return item;
	}

/**
Closes this cache, which makes all items in the cache unavailable. Any items
needed for later should be taken out of the cache before closing it.
<p>
<b>Note:</b>, it is a good idea to close the cache once it's not needed,
because it releases resources, including any internal threads spawned and used
by this implementation.
**/
public void close()
	{
	this.closed = true;
	cache.clear();
	log.info("*** cache:[", myName, "] closed. ***"); 
	}

/**
Returns true if this cache has been closed, false otherwise.
**/
public boolean isClosed()
	{
	return this.closed;
	}

public void clear() {
	synchronized (cache) { 
		cache.clear();
		}
	}

public String toString()
	{
	final int size = cache.size();
	String temp = this.myName + " [bound=" + bound + ", used=";
	temp += size;
	temp += "] items;";
	temp += " [Policy = " + cachePolicy + "] ";
	temp += isClosed() ? "cache is closed. " : "cache is open.";
	return temp;
	}



private class MyLinkedHashMap extends LinkedHashMap
{
MyLinkedHashMap(int initialCapacity, float loadFactor, boolean accessOrder)	
	{
	super(initialCapacity, loadFactor, accessOrder);
	}
	
protected boolean removeEldestEntry(Map.Entry eldest) {
	return size() > bound;	
	}
}


public static void main(String[] args)
	{
	new Test();
	}

static private class Test
	{
	Test() 
		{
		try {
			System.out.println("Testing FIFO...");
			BoundedCache mycache = new BoundedCache(Policy.FIFO, 3);
			
			System.out.println("putting key1, " + mycache.put("key1", "val1"));
			System.out.println("putting key2, " + mycache.put("key2", "key2"));
			System.out.println("putting key3, " + mycache.put("key3", "key3"));
			System.out.println("putting key4, " + mycache.put("key4", "key4"));

			System.out.println("get key1 = " + mycache.get("key1"));
			System.out.println("get key2 = " + mycache.get("key2"));
			System.out.println("getkey3 = " + mycache.get("key3"));
			System.out.println("get key4 = " + mycache.get("key4"));
			System.out.println("mycache.toString() = " + mycache);
			
			System.out.println("=========================");
			System.out.println("Testing LRU...");
		
			mycache = new BoundedCache(Policy.LEAST_USED, 3);
			System.out.println("putting key1, " + mycache.put("key1", "val1"));
			System.out.println("putting key2, " + mycache.put("key2", "key2"));
			System.out.println("putting key3, " + mycache.put("key3", "key3"));
			System.out.println("putting key4, " + mycache.put("key4", "key4"));
			
			System.out.println("getting key2 [4times] and key3[1 time]");
			System.out.println(mycache.get("key2"));
			System.out.println(mycache.get("key2"));
			System.out.println(mycache.get("key2"));
			System.out.println(mycache.get("key2"));
			System.out.println(mycache.get("key3"));
			
			System.out.println("putting key5, " + mycache.put("key5", "key5"));

			System.out.println("get key1=" + mycache.get("key1"));
			System.out.println("get key2=" + mycache.get("key2"));
			System.out.println("get key3=" + mycache.get("key3"));
			System.out.println("get key4=" + mycache.get("key4"));
			System.out.println("get key5=" + mycache.get("key5"));

			System.out.println("mycache.toString() = " + mycache);
			}
		catch (Exception e) {
			e.printStackTrace();
			}
		} //~init
	} //~class test

}			//~BoundedCache
