// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;
import java.text.*;

/** 
Useful to store thread-local Random instances.  This class is intended
for servlets/molly pages. Instead of static get/set methods, this class
must be instantiated and the instance methods used to get/set the random
object. This allows multiple instances of this class in the webapp, with
each instance being able to get/set a separate random generator. [If the methods in
this class were static, then only 1 random could be get/set per thread].
<p>
<font color=red><b>Each thread must remember to individually create a separate
random instance and store it via the set method</b></font>. The usage idiom
is:
<blockquote>
<pre>
<font color=blue>
//WebApp has a map of ThreadLocalRandoms and also a instance variable
//pointing to a default ThreadLocalRandom
ThreadLocalRandom myrand = WebApp.getThreadLocalRandom("foo");
</font><font color=red>
if (myrand.isNull())  {
	myrand.set(new java.util.Random());
	}
</font><font color=blue>
Random rand = myrand.get();
</font>
</pre>
</blockquote>
Note, the lines in red are always needed anywhere/anytime this class is used.
<p>
The methods in java.util.Random are synchronized (or at least they
internally use the synchronized <code>next(int)</code> method), so
these ThreadLocalRandoms are really only useful in heavily Multi threaded
apps where thread contention over a single random number generator
can slow things down.
*/
public final class ThreadLocalRandom
{
public ThreadLocalRandom()
	{ }
/*
Each get/set into the threadlocal must be seperately by each thread (the
initialValue() method is good for auto-assigning a new value but we
may need to assign a custom calendar value per thread, so we can't use
initialValue()
*/
private final ThreadLocal tlrand = new ThreadLocal();

public Random get()
	{
	return (Random) tlrand.get();
	}

public void set(Random cal)
	{
	tlrand.set(cal);
	}

public boolean isNull()
	{
	return tlrand.get() == null;
	}

public static void main (String args[]) throws Exception
	{ 
	final ThreadLocalRandom cal1 = new ThreadLocalRandom();
	
	Thread t1 = new TestThread(false, cal1);
	Thread t2 = new TestThread(false, cal1);
	t1.start();
	t2.start();
	
	for (int n = 0; n < 100; n++) {
		new TestThread(true, cal1).start();
		}
	}

static class TestThread extends Thread
	{
	boolean timing_only = false;
	ThreadLocalRandom rand;
	TestThread(boolean timing_only, ThreadLocalRandom rand) {
		this.timing_only = timing_only;
		this.rand = rand;
		}
		
	public void run()
		{
		//warm up
		if (rand.isNull()) {
			rand.set(new java.util.Random());
			}

		Watch w = new NanoWatch();
		w.start();
		if (rand.isNull()) {
			rand.set(new java.util.Random());
			}
		Random c = rand.get();
		w.stop();
		
		NumberFormat nf = NumberFormat.getNumberInstance();
		nf.setMinimumFractionDigits(2);
		nf.setMaximumFractionDigits(2);
		nf.setMinimumIntegerDigits(2);
	
		if (timing_only)
			System.out.println("[" + nf.format(w.getTime() / 1000000.00D) + " ms]");
		else
			System.out.println("[" + nf.format(w.getTime() / 1000000.00D) + " ms]" + Thread.currentThread() + "/Calendar-ID:[" + System.identityHashCode(c) + "] "+ c);
		}
	}
}