// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.io.*;
import java.util.*;
import java.util.regex.*;

import fc.io.*;

/**
Provides an ultra-simple template/merge type capability. Can be used
standalone or as part of a servlet/cgi environment.
<p> Template description:
<blockquote>
<ul>
<li>The template for the merge file can contain any data with embedded template variables.
	<blockquote>
	A template variable is a alphanumeric name and starts with a <tt>$</tt>. The name can also
	contain the <tt>_</tt> and <tt>-</tt> characters. Examples of variable names are:
	<i><tt>$foo</tt></i>, <i><tt>$bar9-abc</tt></i> and <i><tt>$ab_c</tt></i>. Names must begin
	with a alphabet letter and not with a number or any other character. (this makes is easy to
	have currency numbers like $123 in the template text, without it getting affected by the
	template engine treating it as a variable). 
	<p>
	The <tt>$</tt> char itself can be output via a special template variable <b><font
	color=blue><tt>${dolsign}</tt></font></b> which gets replaced by a literal <b><tt>$</tt></b> in
	the output. Note, the braces are necessary in this case. So, to produce a literal <tt>$foo</tt>
	in the output, there are two choices:
		<blockquote>
		<ul>
			<li>set the value of $foo to the string "$foo", which will then be substituted in the
			resulting output.
			<li>specify ${dolsign}foo in the template text. The braces are necessary, because 
			"$foo", if found in the template text is simply ignored (not a valid variable) and hence
			<tt>dolsign</tt> is not needed at all. For the relevant case <tt>$foo</tt>, the braces
			serve to separate the word <tt>dolsign</tt> from <tt>foo</tt>.
		</ul>
		</blockquote>
	<p>The template engine starts from the beginning of the template text and replaces each
	template variable by it's corresponding value. This value is specified via {@link #set(String,
	String)} method. Template variables are not recursively resolved, so if <tt>$foo</tt> has the value
	"bar" and <tt>$bar</tt> has the value <tt>baz</tt>, then <tt>$$foo</tt> will be resolved to
	<tt>$bar</tt> but the resulting <tt>$bar</tt> will <b>not</b> be resolved further (to
	<tt>baz</tt>).
	</blockquote>
</li>
<li>In addition to template variables, Templates can also contain custom java code. This can be
done by a special template variable, which is always called <tt>$code</tt> and is used with an
argument that denotes the java object to call. For example: <tt>$code(mypackage.myclass)</tt> will
call the <tt>code</tt> method in class <tt>mypackage.myclass</tt>. The specified custom class must
implement the {@link fc.util.CustomCode} interface. The specified custom class must contain a
no-arg constructor and is instantiated (once and only once) by the template engine. Global state
can be stored in a custom singleton object that can be created/used by the custom java code.
</li>
<li>The template engine executes in textual order, replacing/running code as it appears starting
from the beginning of the template. If a corresponding value for a template variable is not found
or a specified custom-code class cannot be loaded, that template variable is ignored (and removed)
from the resulting output. So for example, if <tt>$foo</tt> does not have any value associated with
it (i.e., is <tt>null</tt> by default), then it is simply removed in the resulting output.
Similarly, if class <tt>mypkg.foo</tt> specified in the template text as <tt>$code(mypkg.foo)</tt>
cannot be loaded/run, then it'll simply be ignored in the resulting output.
</li>
<li>Templates don't provide commands for looping (such as <tt>for</tt>, <tt>while</tt> etc).
Templates are limited to simple variable substitution. However, the java code that
creates/sets values for these variables can do any arbitrary looping, and using such
loops, set the value of a template variable to any, arbitrarily large, string.
</li>
</ul>
</blockquote>
<p>
Thread safety: This class is not thread safe and higher level synchronization should be used if
shared by multiple threads.

@author	 hursh jain
@date 	 3/28/2002
**/
public final class Template 
{
//class specific debugging messages: interest to implementors only
private boolean dbg = false;
//for the value() method - useful heuristic when writing to a charwriter
private int approx_len;

static final String dolsign = "{dolsign}";
Map 		datamap;
List 		template_actions;

/* 
pattern for checking the syntax of a template variable name. stops at next whitespace or at the end
of the file, no need to specify any minimal searching. don't need dotall, or multiline for our
purposes.

group 1 gives the template variable or null
group 2 gives the code class or null
*/		
Pattern namepat;

/** 
Constructs a new template object 
@param	templatefile	the absolute path to the template file
**/
public Template(File templatefile) throws IOException
	{
	Argcheck.notnull(templatefile, getClass().getName() + ":<init> specified templatefile parameter was null");
	String templatepath = templatefile.getAbsolutePath();
	String template = IOUtil.fileToString(templatepath);
	if (template == null)
		throw new IOException("The template file: " + templatepath + " could not be read");
	doInit(template);
	}
	
/** 
Constructs a new template object from the given String. Note, this is the the String to use as the
template, <b>not</b> the name of a file. Various input streams can be converted into a template
using methods from the {@link fc.io.IOUtil} class.
**/
public Template(String template) throws IOException
	{	
	Argcheck.notnull(template, getClass().getName() + ":<init> specified template parameter was null");
	doInit(template);
	}
	
private void doInit(String template)  throws IOException
	{
	/* the regex:
		(							#g1 
		\$(?!code) 					#$ not followed by code
		([a-zA-Z](?:\w|-)*) 		#g2: foo_name
		)							#~g1
	|
		(							#g3
		\$code\s*\					#$ followed by code and optional whitespace
		(\s*([^\s]+)\s*\)			#g4: ( whitespace pkg.foo whitespace )
		)
	*/
 	namepat = Pattern.compile(	
 		"(\\$(?!code)([a-zA-Z](?:\\w|-)*))|(\\$code\\s*\\(\\s*([^\\s]+)\\s*\\))" ,
 		Pattern.CASE_INSENSITIVE);  //for $code, $coDE etc.				
	
	datamap = new HashMap(); 
	template_actions = new ArrayList(); 
	
	Matcher matcher = namepat.matcher(template); 
	//find and save position of all template variables in textual order	 
	int pos = 0;
	int len = template.length();

	while (matcher.find()) 
		{
		String g1 = matcher.group(1); String g2 = matcher.group(2);
		String g3 = matcher.group(3); String g4 = matcher.group(4); 
		if (dbg) System.out.println("found, begin:" + matcher.group() + ",g1=" + g1 + ", g2=" + g2 + ", g3=" + g3 + ", g4=" + g4);
		if ( (g1 != null && g3 != null) || (g1 == null && g3 == null) )
			throw new IOException("Error parsing template file, found input I don't understand:" + matcher.group());  
		if (g1 != null) {  //$foo
			int start = matcher.start(1);
			if (dbg) System.out.println("g1:" + pos + "," + start);
			template_actions.add(new Text(template.substring(pos,start)));
			template_actions.add(new Var(g2));
			pos = matcher.end(1);
			if (dbg) System.out.println("finished g1");
			}
		else if (g3 != null) {	//$code(foo)
			int start = matcher.start(3);
			if (dbg) System.out.println("g3:" + pos + "," + start);
			template_actions.add(new Text(template.substring(pos,start)));					
			template_actions.add(new Code(g4));
			pos = matcher.end(3);
			if (dbg) System.out.println("finished g3");
			}
		}	//~while
	
	if (pos != len) {
		template_actions.add(new Text(template.substring(pos,len)));
		}
	
	approx_len = template.length() * 2;

	if (dbg) System.out.println("template_actions = " + template_actions);
	}		//~end constructor

/** 
Returns the template data, which is a <tt>Map</tt> of template variables to values (which were all
set, using the {@link #set(String, String)} method. This map can be modified, as deemed necessary.
**/
public Map getTemplateData() {
	return datamap;
	}


/** 
Resets the template data. If reusing the template over and over again, this is faster, invoke this between
each reuse. (rather than recreate it from scratch every time).
**/
public void reset() {
	datamap.clear();
	}


/**
A template variable can be assigned data using this method. This method should
be called at least once for every unique template variable in the template.

@param 	name	the name of the template variable including the preceding<tt>$</tt>
 				sign. For example: <tt>$foo</tt> 
@param	value 	the value to assign to the template variable.

@throws	IllegalArgumentException 
		if the specified name of the template variable is not syntactically valid.						 	
**/
public void set(String name, String value) {
	if (! checkNameSyntax(name)) {
		throw new IllegalArgumentException("Template variable name " + name + " is not syntactically valid (when specifying variable names, *include* the \"$\" sign!)"); 
		}
	datamap.put(name, value);
	}


/**
An alias for the {@link set} method.
*/
public void fill(String name, String value) {
	set(name, value);
	}

/**
An alias for the {@link set} method.
*/
public void fill(String name, int value) {
	set(name, String.valueOf(value));
	}

/**
An alias for the {@link set} method.
*/
public void fill(String name, long value) {
	set(name, String.valueOf(value));
	}

/**
An alias for the {@link set} method.
*/
public void fill(String name, boolean value) {
	set(name, String.valueOf(value));
	}


/**
An alias for the {@link set} method.
*/
public void fill(String name, Object value) {
	set(name, String.valueOf(value));
	}

/** 
Merges and writes the template and data. Always overwrites the specified
destination (even if the destionation file already exists); 
@param 	destfile	the destination file (to write to).
*/
public void write(File destfile) throws IOException{
	write(destfile, true);
	}

/** 
Merges and writes the template and data. Overwrites the specified destination, only if the
<tt>overwrite</tt> flag is specified as <tt>true</tt>, otherwise no output is written if the
specified file already exists. 
<p>
The check to see whether an existing file is the same as the output file for this template is
inherently system dependent. For example, on Windows, say an an existing file "foo.html" exist in
the file system. Also suppose that the output file for this template is set to "FOO.html". This
template then will then overwrite the data in the existing "foo.html" file but the output filename
will not change to "FOO.html". This is because the windows filesystem treats both files as the same
and a new File with a different case ("FOO.html") is <u>not</u> created.

@param 	destfile	the destination file (to write to).
@param	overwrite	<tt>true</tt> to overwrite the destination
@throws IOException	if an I/O error occurs or if the destination file cannot
		be written to.
*/
public void write(File destfile, boolean overwrite) throws IOException
	{
	Argcheck.notnull(destfile);
	
	if ( destfile.exists() ) 
		{
		if (! overwrite)  {
			return;
			}
		if (! destfile.isFile()) {
			throw new IOException("Specified file: " + destfile + " is not a regular file");
			}
		}
		
	BufferedWriter out = new BufferedWriter(new FileWriter(destfile));
	mergeWrite(out);
	}


/** 
Merges and writes the template and data to the specified Writer
*/
public void write(Writer out) throws IOException
	{
	mergeWrite(out);
	}

/** 
Merges and writes the template and data to the specified Writer
*/
public void write(PrintStream out) throws IOException
	{
	mergeWrite(new PrintWriter(out));
	}


/** 
Merges and writes the template and data and returns it as a String
*/
public String value() throws IOException
	{
	CharArrayWriter cw = new CharArrayWriter(approx_len);
	mergeWrite(new PrintWriter(cw));
	return cw.toString();
	}

/*
don't call mergeWrite(out), if toString() invoked from a custom class or template variable, then it
becomes recursive.
*/
public String toString()
	{
	StringBuilder buf = new StringBuilder(approx_len);
	for (int i = 0; i < template_actions.size(); i++)
		{
		TemplateAction act = (TemplateAction) template_actions.get(i);
		if (act instanceof Text) {
			buf.append(((Text)act).value);
			}
		else if (act instanceof Var) {
			Object val = datamap.get("$" + ((Var)act).varname);
			buf.append("[$" + ((Var)act).varname);
			buf.append("]-->[");
			buf.append(val);			
			buf.append("]");			
			}
		else if (act instanceof Code) {
			buf.append("Classname:[");
			buf.append(((Code)act).classname);
			buf.append("]");
			}
		}
	return buf.toString();
	}


//#mark -
protected boolean checkNameSyntax(String name) {
	return namepat.matcher(name).matches();
	}


protected void mergeWrite(Writer out) throws IOException
	{
	if (dbg) System.out.println("using datamap = " +  datamap);
	Iterator it = template_actions.iterator();		
	while (it.hasNext()) {
		((TemplateAction) it.next()).write(out);
		}
	out.close();
	}

abstract class TemplateAction 
	{
	public abstract void write(Writer writer) throws IOException;
	}

class Text extends TemplateAction
	{
	String value;
	public Text(String val) { value = val; } 
	public void write(Writer writer) throws IOException {
		writer.write(value);
		}
	public String toString() { return "Text:" + value; }
	}

class Var extends TemplateAction {
	String varname;
	public Var(String name) { varname = name; }
	public void write(Writer writer) throws IOException {
		Object val = datamap.get("$" + varname);
		writer.write( (val!=null) ? (String) val : "" );
		}
	public String toString() { return "Var:" + varname; }
	}

static HashMap loadedclasses = new HashMap(); 

class Code extends TemplateAction  
	{
	String classname;
	public Code(String classname) 
		{
		try {
			if ( ! loadedclasses.containsKey(classname)) {
				Class c = Class.forName(classname);
				if (c != null) {
					if (! CustomCode.class.isAssignableFrom(c))
						return;
					loadedclasses.put(classname, c.newInstance());
					}
				}
			this.classname = classname;
			}
		catch (Exception e) {
			e.printStackTrace();
			}
		}
		
	public void write(Writer writer) throws IOException {
		Object obj = loadedclasses.get(classname);
		if ( obj != null ) {
			((CustomCode)obj).code(writer, Template.this);
			}
		}
	public String toString() { return "CustomClass: " + classname; }	
	}

//Template foo = this;	
/**	
Unit Test History 	
<pre>
Class Version	Tester	Status		Notes
1.0				hj		passed		
</pre>
**/
public static void main(String[] args) throws Exception
	{
	new Test(args);		
	}

private static class Test
{
Test(String[] args) throws Exception
	{
	String templateFileName = "test-template.txt";
	String resultFileName = "template-merged.txt";

	System.out.println("Running test using template file: " + templateFileName);
	File f = new File(templateFileName);
	Template t = new Template(f);
	t.set("$a", "a-value");
	t.set("$b", "b-value");
	t.set("$c", "c-value");
	t.write(new File(resultFileName));
	System.out.println("Completed. Results in: " + resultFileName);
	}
} //~inner class test

}           //~class Template