// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;

/** 
Implements a simple table data structure. Like a 2D array
but uses nested hash tables to get (row, col) based on a 
(name, name) pair. The table grows automatically as needed.
<p>
Differs from a 2D Object[][] in that names can be used 
instead of indices.
<p>
All indexes are internally strings. When adding or retrieving
values, either numerical and string indices can be specified
for either row or col when retrieving a (row,col) value. <i>A
numerical index is converted into a string value</i> so if one puts
a value at (3, 5), that can later be retrieved as ("3", "5")
if desired.
<p>
Thread safety: This data structure is fully thread safe.

@author hursh jain
**/
public final class Table
{
private static final boolean dbg = false;

int row_default_size;
int col_default_size;

/*    
    		 A     B    C
rows	1:   x     y    z
    	2:   a     e    n
    	3:   c     d    m

rows(1) => hashtable (A->x, B->y, C->z)
rows(2) => hashtable (A->a, B->e, C->n)
..
*/
Map table;

/** 
Constructs a new table.
**/
public Table() {
	this(32, 32);
	}

/** 
Constructs a new table of the specified initial size.
**/
public Table(int rows, int cols) {
	this.row_default_size = rows;
	this.col_default_size = cols;
	this.table = new Hashtable(rows);
	}

/** 
Returns the object at row, col or <tt>null</tt> if the
specified row or col do not exist.
*/
public Object get(String row, String col) 
	{
	Object val = null;
	
	Map cols = (Map) table.get(row);
	
	if (cols != null)	
		val = cols.get(col);
		
	return val;
	}


/**
Returns the object at row, col or <tt>null</tt> if the
specified row or col do not exist.
*/
public Object get(int row, String col) 
	{
	return get(String.valueOf(row), col);
	}

/**
Returns the object at row, col or <tt>null</tt> if the
specified row or col do not exist.
*/
public Object get(String row, int col) 
	{
	return get(row, String.valueOf(col));
	}

/**
Returns the object at row, col or <tt>null</tt> if the
specified row or col do not exist.
*/
public Object get(int row, int col) 
	{
	return get(String.valueOf(row), String.valueOf(col));
	}

/**
Returns the object at row, col or <tt>null</tt> if the
specified row or col do not exist.

@return the old object at that (row, col) if it exists
*/
public Object put(String row, String col, Object obj) 
	{
	Object val = null;
	
	Map c = (Map) table.get(row);
	if (c == null) {
		c = new Hashtable(col_default_size);
		table.put(row, c);
		}
	
	val = c.put(col, obj);	
	return val;
	}


public Object put(int row, String col, Object obj) 
	{
	return put(String.valueOf(row), col, obj);
	}

public Object put(String row, int col, Object obj) 
	{
	return put(row, String.valueOf(col), obj);
	}

public Object put(int row, int col, Object obj) 
	{
	return put(String.valueOf(row), String.valueOf(col), obj);
	}


public String toString() {
	return "Table: [" + table.size() + " rows]";
	}
	
public static void main (String args[])
	{
	Table t = new Table();
	System.out.println("t.get(1,1)="+t.get(1,1));
	System.out.println("t.put(1,1,\"hello\")="+t.put(1,1, "hello"));
	System.out.println("t.get(\"1\",\"1\")="+t.get("1","1"));
	System.out.println("t.get(1,\"1\")="+t.get(1,"1"));
	System.out.println("t.get(1,0)="+t.get(1,0));
	System.out.println("t.put(1,1,\"hello\")="+t.put(1,1, "world"));
	System.out.println("t.get(\"1\",1)="+t.get("1",1));

	for (int n = 0; n < 10; n++) {
		t.put(n, n, n);
		}

	for (int n = 0; n < 10; n++) {
		System.out.println("table["+n+","+n+"]:"+t.get(n, n));
		}
	
	Watch w = new Watch().start();
	t.get(5,5);
	w.stop();
	System.out.println("time for one t.get() op: " + w.time() + " ms");
	}
}		//~class Table