// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;


/** A set of date ranges. **/
public class DateRangeSet
{
DateRange seedRange;	
List union;
List intersect;

/**
Constructs a new DateRangeSet with the initial set containing
only the specified DateRange

@param	cs	the initial DateRange
**/
public DateRangeSet(DateRange cs) 
	{
	Argcheck.notnull(cs);
	seedRange = cs;	
	//do _not_ lazy instantiate, otherwise have to put in
	//null checks in other places
	union = new ArrayList();
	intersect = new ArrayList();
	}

/**
Adds this specified range as a union to the existing set of 
ranges (for purposes of {@link inRange(char)} method). 
Overlapping ranges are ok.

@param	cs	a DateRange to unite with
@throws IllegalArgumentException 
			if the specified range was null
**/
public void union(DateRange r) 
	{
	Argcheck.notnull(r);
	union.add(r);
	}

/**
Adds the specified range as an intersection to the existing
ranges (for purposes of {@link inRange(char)} method). 
Overlapping ranges are ok. 

@param	r	the range to add
**/
public void intersection(DateRange r) 
	{
	Argcheck.notnull(r);	
	intersect.add(r);
	}

/** 
Consider a set of ranges <tt>A</tt>, <tt>B</tt> added as a union
(logical <tt>or</tt>) and ranges <tt>C</tt> and <tt>D</tt> added as an
intersection (logical <tt>and</tt>). A character <tt>c</tt> is in range if
it exists in:

<blockquote> <tt>

(A.inRange(c) || B.inRange(c) || ...) && C.inRange(c) && D.inRange(c)
&& ...

</tt></blockquote> 

This can be generalized to an arbitrary number of sub ranges. If
intersection or union ranges don't exist, then they are not
considered in the above expression. Note, the interaction may be
subtle if any of the ranges (<tt>A, B, C...</tt>etc) are
individually negated because in that case the inRange method for
that negated range would return true if the specified character
was <b>not</b> in that range.

@return	<tt>true</tt> if the passed in character is allowed by this 
		set of ranges.

**/
public boolean inRange(Date val) 
	{
	boolean result = seedRange.inRange(val);
	for (int i = 0, n = union.size(); i < n; i++) {
		DateRange item = (DateRange) union.get(i);
		result = result || item.inRange(val);
		}		
	for (int i = 0, n = intersect.size(); i < n; i++) {
		DateRange item = (DateRange) intersect.get(i);
		result = result && item.inRange(val);
		}	
	return result;
	}

public String toString() {
	StringBuffer buf = new StringBuffer(128); 
	buf.append("DateRangeSet:[");	
	
	if (intersect.size() > 0)
		buf.append("(");

	buf.append(seedRange);

	int len = union.size();
	if (len > 0) 
   		{
   		buf.append(" || " );
   		for (int i = 0; i < len; i++) {
   			buf.append(union.get(i).toString());
   			if (i < (len-1))
   				buf.append(" || ");
   			}			
		}
		
	len = intersect.size();
	if (len > 0) 
		{
   		buf.append(") && " );			
		for (int i = 0; i < len; i++) {
			buf.append(intersect.get(i).toString());
			if (i < (len-1))
				buf.append(" && ");
			}			
		}
		
	buf.append("]");
	return buf.toString();
	}


public static void main(String[] args)
	{
	Date now = new Date();

	Calendar cal = Calendar.getInstance();
	Date d1 = cal.getTime();
	//back 1 month
	cal.add(Calendar.MONTH, -1);
	Date d2 = cal.getTime();
	   
	DateRange r = new DateRange(d2, d1);

	cal.setTime(now);
	cal.add(Calendar.MONTH, -3);
	Date d3 = cal.getTime(); 
	cal.setTime(now);
	cal.add(Calendar.MONTH, -6);
	Date d4 = cal.getTime(); 

	DateRange r2 = new DateRange(d4, d3);

	DateRangeSet crs = new DateRangeSet(r);
	crs.union(r2);
	System.out.println("constructed: " + crs);
	test(crs);
	
	cal.setTime(now);
	cal.add(Calendar.MONTH, -4);
	d3 = cal.getTime();
	cal.setTime(now);
	cal.add(Calendar.MONTH, -5);
	d4 = cal.getTime();
	r2 = new DateRange(d4, d3);
	crs.intersection(r2);
	System.out.println("constructed: " + crs);
	test(crs);
	} 
	
private static void test(DateRangeSet crs) 
	{	
	Calendar cal = Calendar.getInstance();
	cal.add(Calendar.MONTH, -5);
	Date d = cal.getTime(); 
	System.out.println("in range: " + d + "=" + crs.inRange(d));
	cal.setTime(new Date());
	cal.add(Calendar.MONTH, 3);
	d = cal.getTime(); 
	System.out.println("in range: " + d + "=" + crs.inRange(d));
	}	
}