// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;

/** 
A range of characters.
<p>
Note, many methods in this particular class have the same
signatures as in the apache <tt>commons.lang</tt> package but the
<tt>implementation</tt> was strictly clean-room.
<p>
Thread Safety: This class is not thread safe in the face of any
concurrent modification (via changing the start/end chars or
union/intersection operations).

@author hursh jain
**/
public class CharRange extends Range
{
char startChar;
char endChar;

/** 
Constructs a character range between the specified
start and end chars (both inclusive). 

@throws IllegalArgumentException 
			if the end char is lesser (via a <tt>&lt;</tt> comparison) 
			than the start char (equal to the start char is ok)
**/
public CharRange(char start, char end) 
	{
	Argcheck.istrue(start <= end, "end char lesser than start char in the specified range, start=["+start+"], end=["+end+"]");
	startChar = start;
	endChar = end;
	}
	
/** Get the start character for this character range **/
public char getStart() 
	{
	return startChar;
	}

/** 
Get the end character for this character range. 
**/
public char getEnd() 
	{
	return endChar;
	}

/** Is the passed in character inside this range **/
public boolean inRange(char c) 
	{
	boolean result = (c >= startChar) && (c	<= endChar);
	
	if (negated)
	 	return !result;
	else
		return result;
	}
 
/** 
Sets the end character for this character range 

@param	c	the end char
@throws IllegalArgumentException 
			if the end char is lesser (via a <tt>&lt;</tt> comparison) 
			than the current start char (equal to the start char is 
			ok)
**/
public void setEnd(char c) 
	{
	Argcheck.istrue(startChar <= c, "end char lesser than start char in the specified range, start=["+startChar+"], specified end=["+c+"]");
	endChar = c;
	}
 
/** Set the start character for this character range 

@throws IllegalArgumentException 
			if the start char is greater (via a <tt>&gt;</tt> comparison) 
			than the current end char (equal to the end char is 
			ok)
**/
public void setStart(char c) 
	{
	Argcheck.istrue(c <= endChar, "start char greater than end char in the specified range, end=["+endChar+"], specified start=["+c+"]");
	startChar = c;
	}

/** Output a string representation of the character range **/
public java.lang.String toString() 
	{
	String str = "CharRange:[";
	if (isNegated())
		str +=	"^";
	str +=	"'" + startChar + "'-'" + endChar + "']";
	return str;
	}

public static void main(String[] args)
	{
	CharRange r = new CharRange('b', 'z');
	System.out.println("constructed range:" + r);
	//normal
	test(r);
	
	//negate
	r.setNegated(true);
	System.out.println("setting the range to be negated:" + r);
	test(r);

	System.out.println("the following should throw an exception");
	r = new CharRange('b', 'a');
	}

private static void test(CharRange r) 
	{
	System.out.println("'b' in range:" + r.inRange('b'));
	System.out.println("'z' in range:" + r.inRange('z'));
	System.out.println("'d' in range:" + r.inRange('d'));
	System.out.println("'a' in range:" + r.inRange('a'));
	System.out.println("'A' in range:" + r.inRange('A'));
	}

}          //~class CharRange