// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;
import java.text.*;
import java.util.concurrent.*;

public final class CalendarUtil
{
private static final boolean debug = false;

public static enum Duration 
	{ 
	HOURLY,
	DAILY, 
	WEEKLY, 
	MONTHLY;
	
	/** 
	Returns the date representing start time + duration. Uses the default
	calendar instance internally.
	*/
	public static Date getDateAfterDuration(Date start, Duration duration) 
		{
		Calendar cal = Calendar.getInstance();
		
		Date endDate = null;
		
		switch (duration) {
			case HOURLY:
				endDate = CalendarUtil.getEndOfHour(cal, start);
				break;
			case DAILY:
				endDate = CalendarUtil.getEndOfDay(cal, start);
				break;
			case WEEKLY:
				endDate = CalendarUtil.getLastDayOfWeek(cal, start);
				break;
			case MONTHLY:
				endDate = CalendarUtil.getLastDayOfMonth(cal, start);
				break;
			default: 
				throw new IllegalArgumentException("Don't know how to handle this duration");
			}
			
		return endDate;
		}
	}	

/** Useful constant of 1 second (in milliseconds) */
public static final long ONE_SEC 	= 1000;
	
/** Useful constant of 1 minute (in milliseconds) */
public static final long ONE_MIN 	= 1  * 60  * 1000;

/** Useful constant of 2 minutes (in milliseconds) */
public static final long TWO_MIN 	= 2  * 60  * 1000;

/** Useful constant of five minutes (in milliseconds) */
public static final long FIVE_MIN 	= 5  * 60  * 1000;

/** Useful constant of ten minutes (in milliseconds) */
public static final long TEN_MIN 	= 10 * 60 * 1000;

/** Useful constant of thirty minutes (in milliseconds) */
public static final long THIRTY_MIN = 30 * 60 * 1000;

/** Useful constant of one hour (in milliseconds) */
public static final long ONE_HOUR 	= 60 * ONE_MIN;

/** Useful constant of two hours (in milliseconds) */
public static final long TWO_HOUR 	= 2  * ONE_HOUR;

/** Useful constant of four hours (in milliseconds) */
public static final long FOUR_HOUR 	= 4  * ONE_HOUR;

/** Useful constant of eight hours (in milliseconds) */
public static final long EIGHT_HOUR = 8  * ONE_HOUR;

/** Useful constant of twelve hours (in milliseconds) */
public static final long TWELVE_HOUR = 12  * ONE_HOUR;

/** Useful constant of twenty four hours (in milliseconds) */
public static final long TWENTY_FOUR_HOUR = 24  * ONE_HOUR;

/** Useful constant of 1 day (in milliseconds) */
public static final long ONE_DAY = TWENTY_FOUR_HOUR;

/** Useful constant of 3 days (in milliseconds) */
public static final long THREE_DAY = 3 * ONE_DAY;

/** Useful constant of 1 week (in milliseconds) */
public static final long ONE_WEEK = 7  * ONE_DAY;

/** Useful constant of 2 weeks (in milliseconds) */
public static final long TWO_WEEK = 14  * ONE_DAY;

/** Useful constant of 1 month (in milliseconds) [30 day month] */
public static final long ONE_MONTH = 30  * ONE_DAY;

/** Useful constant of 2 months (in milliseconds) [30 day month] */
public static final long TWO_MONTH = 60  * ONE_DAY;

/** Useful constant of 3 months (in milliseconds)[30 day month]  */
public static final long THREE_MONTH = 90  * ONE_DAY;

/** Useful constant of 6 months (in milliseconds) [30 day month] */
public static final long SIX_MONTH = 180  * ONE_DAY;

/** Useful constant of one year (in milliseconds) [365 days]*/
public static final long ONE_YEAR = 365  * ONE_DAY;

/**
Sets the month in the specified calendar based on the specified 0-based
month number. There is no direct setMonth method in Calendar. Therefore,
this method is essentially a giant switch statement, like: 
<blockquote><pre>
  case 0: 
	  cal.set(Calendar.MONTH, Calendar.JANUARY); 
	  break;
  case 1: 
	  cal.set(Calendar.MONTH, Calendar.FEBRUARY); 
	  break;
  ... etc...
</pre></blockquote>
*/
public static final void setMonth(Calendar cal, int monthNum)  
	{ // 0-based
	switch (monthNum)
		{
		case 0: cal.set(Calendar.MONTH, Calendar.JANUARY); break;
		case 1: cal.set(Calendar.MONTH, Calendar.FEBRUARY); break;
		case 2: cal.set(Calendar.MONTH, Calendar.MARCH); break;
		case 3: cal.set(Calendar.MONTH, Calendar.APRIL); break;
		case 4: cal.set(Calendar.MONTH, Calendar.MAY); break;
		case 5: cal.set(Calendar.MONTH, Calendar.JUNE); break;
		case 6: cal.set(Calendar.MONTH, Calendar.JULY); break;
		case 7: cal.set(Calendar.MONTH, Calendar.AUGUST); break;
		case 8: cal.set(Calendar.MONTH, Calendar.SEPTEMBER); break;
		case 9: cal.set(Calendar.MONTH, Calendar.OCTOBER); break;
		case 10: cal.set(Calendar.MONTH, Calendar.NOVEMBER); break;
		case 11: cal.set(Calendar.MONTH, Calendar.DECEMBER); break;
		default:
			throw new RuntimeException("Month value out of range [" + monthNum + "]");
		}
	}

		
/**
Returns the date representing the addition/subtraction of a number of hours from the specified
starting timestamp. Specify a positive number for future/adding hours or negative for 
past/subtracing hours.
<p>
The state of the calendar is not affected by the calculations performed by this method.
*/
public static Date addHours(Calendar cal, Date d, int hours) 
	{
	cal = (Calendar) cal.clone();  //don't affect underlying calendar
	
	cal.setTime(d);
	
	cal.add(Calendar.HOUR_OF_DAY, hours);

	return cal.getTime();
	}


/**
Returns the closest hour, starting from the specified time and the specified calendar. The state of the calendar is not affected by the calculations performed by this method.
<p>
For example, any date with time component <tt>12.30pm</tt> returns the same date with a time component <tt>12.00pm</tt>.
*/
public static Date getBeginOfHour(Calendar cal, Date d) 
	{
	cal = (Calendar) cal.clone();  //don't affect underlying calendar
	
	cal.setTime(d);

	//don't have to do anything with hour of day, we keep that hour	
	//cal.roll(Calendar.HOUR_OF_DAY, -1);

	cal.clear(Calendar.MINUTE);
	cal.clear(Calendar.SECOND);
	cal.clear(Calendar.MILLISECOND);

	return cal.getTime();
	}


/**
Returns the closest hour, starting from the current time and the specified calendar.
The state of the calendar is not affected by the calculations performed by this
method.
<p>
For example, any date with time component <tt>12.30pm</tt> returns the same date with a time component <tt>12.00pm</tt>.
*/
public static Date getBeginOfCurrentHour(Calendar cal) 
	{
	return getBeginOfHour(cal, new Date());
	}


/**
Returns the end of the hour, starting from the specified time and the specified calendar. The state of the calendar is not affected by the calculations performed by this method.
<p>
For example, any date with time component <tt>12.30pm</tt> returns the same date with a time component <tt>1.00pm</tt>.
*/
public static Date getEndOfHour(Calendar cal, Date d) 
	{
	cal = (Calendar) cal.clone();  //don't affect underlying calendar
	
	cal.setTime(d);

	cal.roll(Calendar.HOUR_OF_DAY, 1);

	cal.clear(Calendar.MINUTE);
	cal.clear(Calendar.SECOND);
	cal.clear(Calendar.MILLISECOND);

	return cal.getTime();
	}


/**
Returns the end of the hour, starting from the current time and the specified calendar. The state of the calendar is not affected by the calculations performed by this method.
<p>
For example, any date with time component <tt>12.30pm</tt> returns the same date with a time component <tt>1.00pm</tt>.
*/
public static Date getEndOfCurrentHour(Calendar cal) 
	{
	return getEndOfHour(cal, new Date());
	}
		
/**
Returns the date representing the beginning of the specified day (at
12:00:00.000 AM), starting from the current time and the specified calendar.
The state of the calendar is not affected by the calculations performed by this
method.
*/
public static Date getBeginOfDay(Calendar cal, Date d) 
	{
	cal = (Calendar) cal.clone();  //don't affect underlying calendar
	
	cal.setTime(d);

	cal.set(Calendar.HOUR_OF_DAY, 0); 
	cal.clear(Calendar.MINUTE);
	cal.clear(Calendar.SECOND);
	cal.clear(Calendar.MILLISECOND);

	return cal.getTime();
	}


/**
Returns the date representing the beginning of the current day (at 12:00:00.000
AM), starting from the current time and the specified calendar. The state of
the calendar is not affected by the calculations performed by this method.
*/
public static Date getBeginOfCurrentDay(Calendar cal) 
	{
	return getBeginOfDay(cal, new Date());
	}
	
/**
Returns the date representing the end of the specified day (11:59:59.999 PM),
starting from the current time and the specified calendar. The state of the
calendar is not affected by the calculations performed by this method.
*/
public static Date getEndOfDay(Calendar cal, Date d) 
	{
	cal = (Calendar) cal.clone();  //don't affect underlying calendar
	
	cal.setTime(d);

	cal.set(Calendar.HOUR_OF_DAY, 23); 
	cal.set(Calendar.MINUTE, 59);
	cal.set(Calendar.SECOND, 59);
	cal.set(Calendar.MILLISECOND, 999);

	return cal.getTime();
	}

/**
Returns the date representing the end of the current day (11:59:59.999 PM),
starting from the current time and the specified calendar. The state of the
calendar is not affected by the calculations performed by this method.
*/
public static Date getEndOfCurrentDay(Calendar cal) 
	{
	return getEndOfDay(cal, new Date());
	}
	
/**
Returns the date representing the first day of the week (at 12:00:00.000 AM),
using the specified date as the starting point of the week under consideration
and the specified calendar. The state of the calendar is not affected by the
calculations performed by this method. 
<p>
The first day of week can vary based on the locale associated with the specified
calendar (sunday in us, monday in fr, etc).
*/
public static Date getFirstDayOfWeek(Calendar cal, Date d) 
	{
	cal = (Calendar) cal.clone();  //don't affect underlying calendar
	
	cal.setTime(d);

	cal.set(Calendar.HOUR_OF_DAY, 0); 
	cal.clear(Calendar.MINUTE);
	cal.clear(Calendar.SECOND);
	cal.clear(Calendar.MILLISECOND);

	cal.set(Calendar.DAY_OF_WEEK, cal.getFirstDayOfWeek());

	return cal.getTime();
	}

/**
Returns the date representing first day of the week (at 12:00:00.000 AM), starting
from the current time and the specified calendar. The state of the calendar is
not affected by the calculations performed by this method.
*/
public static Date getFirstDayOfCurrentWeek(Calendar cal) 
	{
	return getFirstDayOfWeek(cal, new Date());
	}

/**
Returns the date representing the last day of the week (at 11:59:59.999 PM)
using the specified date as the starting point of the week under consideration
and the specified calendar. The state of the calendar is not affected by the
calculations performed by this method.
<p>
The last day of week can vary based on the locale associated with the specified
calendar (saturday in us, sunday in fr, etc).
*/
public static Date getLastDayOfWeek(Calendar cal, Date d) 
	{
	cal = (Calendar) cal.clone();  //don't affect underlying calendar
	
	cal.setTime(d);

	cal.set(Calendar.HOUR_OF_DAY, 23); 
	cal.set(Calendar.MINUTE, 59);
	cal.set(Calendar.SECOND, 59);
	cal.set(Calendar.MILLISECOND, 999);

	cal.set(Calendar.DAY_OF_WEEK, cal.getFirstDayOfWeek() + 6);

	return cal.getTime();
	}

/**
Returns the date representing the last day of the week (11:59:59.999 PM), starting from the current
time and the specified calendar. The state of the calendar is not affected by
the calculations performed by this method.
*/
public static Date getLastDayOfCurrentWeek(Calendar cal) 
	{
	return getLastDayOfWeek(cal, new Date());
	}

/**
Returns the first day of the specified month with the default locale;
*/
public static Date getFirstDayOfMonth(Calendar cal, Date d) 
	{
	List list = getLastNMonths(cal, d, 1, Locale.getDefault());
	return ((Date[])list.get(0))[0];
	}

/**
Returns the first day of the current month with the default locale;
*/
public static Date getFirstDayOfCurrentMonth(Calendar cal) 
	{
	return getFirstDayOfMonth(cal, new Date());
	}

/**
Returns the last day of the specified month with the default locale;
*/
public static Date getLastDayOfMonth(Calendar cal, Date d) 
	{
	List list = getLastNMonths(cal, d, 1, Locale.getDefault());
	return ((Date[])list.get(0))[1];
	}

/**
Returns the last day of the current month with the default locale;
*/
public static Date getLastDayOfCurrentMonth(Calendar cal) 
	{
	return getLastDayOfMonth(cal, new Date());
	}	
	
/**
Returns {#getLastNMonths} with the default locale;
*/
public static List getLastNMonths(Calendar cal, int nummonths) 
	{
	return getLastNMonths(cal, new Date(), nummonths, Locale.getDefault());
	}
	
/**
Returns a List of Date[][], with each item being a month. The nummonths is the number
of months to return and the startingDate is the starting month, to count backwards
from. 
<p>
In the returned list, date[0] represents the first day, 12:00:00.000 AM and date[1] representing the last day, 11:59:59.999 PM. The starting item in the list is the nearest
month and each successive item is the prior month.
<p>
For example: To get last 6 months, say: <tt>getLastNMonths(cal, today, 6)</tt> where
cal is a calendar to use and today is today's date.
*/
public static List getLastNMonths(Calendar cal, Date startingDate, int nummonths, Locale locale) 
	{
	DateFormatSymbols dfs = new DateFormatSymbols(locale);
	String[] monthnames_short = dfs.getShortMonths();

	cal = (Calendar) cal.clone();  //don't affect underlying calendar

	cal.setTime(startingDate);
	
	if (debug) System.out.println("Current Month (0 based): " + cal.get(Calendar.MONTH));

	List list = new ArrayList();
	for (int n = 0; n < nummonths; n++) 
		{
		int month = cal.get(Calendar.MONTH);

		int days_in_month = cal.getActualMaximum(Calendar.DAY_OF_MONTH);
		if (debug) System.out.println("number of days in month: " + days_in_month);

		Date[] dates = new Date[2];

		//date[0], start of month
		cal.set(Calendar.DAY_OF_MONTH, 1);
		
		cal.set(Calendar.HOUR_OF_DAY, 0); 
		cal.clear(Calendar.MINUTE);
		cal.clear(Calendar.SECOND);
		cal.clear(Calendar.MILLISECOND);
		
		dates[0] = cal.getTime();
		
		//date[1], end of month
		cal.set(Calendar.DAY_OF_MONTH, days_in_month);

		cal.set(Calendar.HOUR_OF_DAY, 23); 
		cal.set(Calendar.MINUTE, 59);
		cal.set(Calendar.SECOND, 59);
		cal.set(Calendar.MILLISECOND, 999);

		dates[1] = cal.getTime();
		list.add(dates);
		
		cal.roll(Calendar.MONTH, -1);
		}
	
	return list;
	}

//#mark -- other misc --

/**
returns true if the specified time is within the specified days, starting from 
the current time.
*/
public static boolean withinLastNDays(long time, int n_days)
	{
	return System.currentTimeMillis() - time <= n_days * CalendarUtil.ONE_DAY;
	}
 
/**
returns true if the specified time is within the specified days, starting from 
the current time.
*/
public static boolean withinLastNDays(Date time, int n_days)
	{
	return System.currentTimeMillis() - time.getTime() <= n_days * CalendarUtil.ONE_DAY;
	} 

/**
returns true if the specified time is within the specified hours, starting from 
the current time.
*/
public static boolean withinLastNHours(long time, int n_hours)
	{
	return System.currentTimeMillis() - time <= n_hours * CalendarUtil.ONE_HOUR;
	}

/**
returns true if the specified time is within the specified hours, starting from 
the current time.
*/
public static boolean withinLastNHours(Date time, int n_hours)
	{
	return System.currentTimeMillis() - time.getTime() <= n_hours * CalendarUtil.ONE_HOUR;
	}


/**
returns true if the specified time is within the specified min, starting from 
the current time.
*/
public static boolean withinLastNMin(Date time, int n_min)
	{
	return System.currentTimeMillis() - time.getTime() <= n_min * CalendarUtil.ONE_MIN;
	}


/**
returns true if the specified time is within the specified seconds, starting from 
the current time.
*/
public static boolean withinLastNSeconds(Date time, int n_seconds)
	{
	return System.currentTimeMillis() - time.getTime() <= n_seconds * 1000;
	}
	
	
/**
returns the number of millis in the specified number of days
*/
public static long daysToMillis(int days)
	{
	return days * CalendarUtil.ONE_DAY;
	} 

/**
returns the number of millis in the specified number of hours
*/
public static long hoursToMillis(int hours)
	{
	return hours * CalendarUtil.ONE_HOUR;
	} 

/**
returns the number of millis in the specified hours
*/
public static long minutesToMillis(int minutes)
	{
	return minutes * CalendarUtil.ONE_MIN;
	} 


/**
returns number of hours between begin and end timestamps. the
order of begin/end is irrelevant, uses absolute value of difference
between the two for the calculation.
*/
public static int hoursBetween(Date begin, Date end)
	{
	final long diff = Math.abs(begin.getTime() - end.getTime());
	return (int) TimeUnit.HOURS.convert(diff, TimeUnit.MILLISECONDS);
	}

/**
returns now as java.sql.Timestamp
*/
public static java.sql.Timestamp getNowTimestamp()
	{
	return new java.sql.Timestamp(System.currentTimeMillis());
	} 

/**
returns now as java.sql.Timestamp
*/
public static java.sql.Timestamp now()
	{
	return getNowTimestamp();
	} 

public static void main (String args[])	 throws Exception
	{
	Args myargs = new Args(args);
	ToString.Style style = new ToString.Style();
	style.reflectStatics = true;
	System.out.println(
		new ToString(new CalendarUtil(), style).reflect().render());
				
	Calendar cal = Calendar.getInstance();
	Date d = new Date();
	if (myargs.flagExists("date")) {
		d = DateFormat.getDateInstance(DateFormat.SHORT)
				.parse(myargs.get("date"));
		}

	System.out.println("========== hour =========");
	System.out.println("beginOfHour: " + getBeginOfHour(cal, d));
	System.out.println("endOfHour: " + getEndOfHour(cal, d));
	System.out.println("beginOfCurrentHour: " + getBeginOfCurrentHour(cal));
	System.out.println("endOfCurrentHour: " + getEndOfCurrentHour(cal));

	System.out.println("========== day =========");
	System.out.println("beginOfDay: " + getBeginOfDay(cal, d));
	System.out.println("endOfDay: " + getEndOfDay(cal, d));
	System.out.println("beginOfCurrentDay: " + getBeginOfCurrentDay(cal));
	System.out.println("endOfCurrentDay: " + getEndOfCurrentDay(cal));

	System.out.println("========== week =========");	
	System.out.println("firstDayOfWeek: " + getFirstDayOfWeek(cal, d));
	System.out.println("lastDayOfWeek: " + getLastDayOfWeek(cal, d));
	System.out.println("firstDayOfCurrentWeek: " + getFirstDayOfCurrentWeek(cal));
	System.out.println("lastDayOfCurrentWeek: " + getLastDayOfCurrentWeek(cal));

	System.out.println("========== month =========");
	System.out.println("firstDayOfMonth: " + getFirstDayOfMonth(cal,d));
	System.out.println("lastDayOfMonth: " + getLastDayOfMonth(cal,d));
	System.out.println("firstDayOfCurrentMonth: " + getFirstDayOfCurrentMonth(cal));
	System.out.println("lastDayOfCurrentMonth: " + getLastDayOfCurrentMonth(cal));

	System.out.println("==== lastNMonths(6) ======");
	System.out.println(Arrays.deepToString(getLastNMonths(cal, d, 6, Locale.getDefault()).toArray()));
	}
}

