// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;

/**
Misc Binary/bit operation related utility methods.

All bit positions are 0-based. For example, to set a 8th logical bit (highest bit in a byte), say 
<tt>setBit(byte, 7)</tt>.
*/

public class BitUtil
{
/* 
we could only use the long version (all types get widened to long) but we won't be able
to error check for the bit position number relevant to the type - so different methods.
The spec says jvm will shift more than 31 or 63 bits anyway (even if larger value specified)
but this allows us to catch bugs
*/

/** 
returns <tt>true</tt> is specified bit in the specified byte is set. The num parameter is the bit
position (0-based index). Set means the bit has value 1.

@throws Runtime
*/
public static boolean isBitSet(byte i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 7, "bit number must be between [0-7] for char, I got num=[" + num + "]");

	return ((i >> num) & 0x1) == 1;
	}

public static boolean isBitSet(short i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 15, "bit number must be between [0-15] for short, I got num=[" + num + "]");
			
	return ((i >> num) & 0x1) == 1;
	}

public static boolean isBitSet(char i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 15, "bit number must be between [0-15] for char, I got num=[" + num + "]");
			
	return ((i >> num) & 0x1) == 1;
	}

public static boolean isBitSet(int i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 31, "bit number must be between [0-31] for int, I got num=[" + num + "]");
			
	return ((i >> num) & 0x1) == 1;
	}

public static boolean isBitSet(long i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 63, "bit number must be between [0-63] for char, I got num=[" + num + "]");
	
	return ((i >> num) & 0x1L) == 1;
	}
	
public static byte setBit(byte i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 7, "bit number must be between [0-7] for byte, I got num=[" + num + "]");
			
	return (byte) (i | (1 << num));
	}

public static short setBit(short i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 15, "bit number must be between [0-15] for short, I got num=[" + num + "]");
			
	return (short) (i | (1 << num));
	}

public static char setBit(char i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 15, "bit number must be between [0-15] for char, I got num=[" + num + "]");
			
	return (char) (i | (1 << num));
	}

public static int setBit(int i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 31, "bit number must be between [0-31] for int, I got num=[" + num + "]");
		
	return i | (1 << num);	
	}

public static long setBit(long i, long num) 
	{
	Argcheck.istrue(num >= 0 && num <= 63, "bit number must be between [0-63] for long, I got num=[" + num + "]");

	return i | (0x1L << num);  //HAVE TO SAY 1L !!! 1 << num will only shift a max of 32 bits
							   //since num is constrained by spec to 0-31 max if lhs is int
							   //by default, all literals are int unless L specified.
							   //spent all day chasing this bug down.
	}


public static byte clearBit(byte i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 7, "bit number must be between [0-7] for byte, I got num=[" + num + "]");
	
	return (byte) (i & ~(1 << num)); 
	}

public static short clearBit(short i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 15, "bit number must be between [0-15] for short, I got num=[" + num + "]");
			
	return (short) (i & ~(1 << num)); 
	}

public static char clearBit(char i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 15, "bit number must be between [0-15] for char, I got num=[" + num + "]");
		
	return (char) (i & ~(1 << num));
	}

public static int clearBit(int i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 31, "bit number must be between [0-31] for int, I got num=[" + num + "]");
			
	return i & ~(1 << num);
	}

public static long clearBit(long i, int num) 
	{
	Argcheck.istrue(num >= 0 && num <= 63, "bit number must be between [0-63] for long, I got num=[" + num + "]");
	
	return i & ~(1L << num);
	}


//Integer.toBinaryString is doesn't left pad and the logic to do it is stupid so 
//just roll our own (faster this way anyway)

public static String toBinaryString(byte i) 
	{	
	char[] buf = new char[8];
	for (int n = 0; n < 8; n++) {
		buf[n] = ((i >> (7-n)) & 0x1) == 1 ? '1' : '0';
   		}
   	return new String (buf);
	}

/** 
returns a 16 char binary digit string
*/
public static String toBinaryString(short i) 
	{
	char[] buf = new char[16];
	for (int n = 0; n < 16; n++) {
		buf[n] = ((i >> (15-n)) & 0x1) == 1 ? '1' : '0';
   		}
   	return new String (buf);
	}
	
/** 
returns a 16 char binary digit string
*/
public static String toBinaryString(char i) 
	{
	char[] buf = new char[16];
	for (int n = 0; n < 16; n++) {
		buf[n] = ((i >> (15-n)) & 0x1) == 1 ? '1' : '0';
   		}
   	return new String (buf);
	}

/** 
returns a 32 char binary digit string (Integer.toBinaryString is not padded, can be less than 32
chars) 
*/
public static String toBinaryString(int i) 
	{
	char[] buf = new char[32];
	for (int n = 0; n < 32; n++) {
		buf[n] = ((i >> (31-n)) & 0x1) == 1 ? '1' : '0';
   		}
   	return new String (buf);
	}

/** 
returns a 64 char binary digit string (Long.toBinaryString is not padded, can be less than 64
chars)) 
*/
public static String toBinaryString(long i) 
	{
	char[] buf = new char[64];
	for (int n = 0; n < 64; n++) {
		buf[n] = ((i >> (63-n)) & 0x1L) == 1 ? '1' : '0';
   		}
   	return new String (buf);
	}


static final int DEF_SEP_NUM = 4;
static final char DEF_SEP_CHAR = ' ';

private static Map getFormatMap(String formatstr)
	{
	int sep_num = DEF_SEP_NUM;
	char sep_char = DEF_SEP_CHAR;
	
	char[] arr = null;
	
	if (formatstr != null) {
		arr = formatstr.toCharArray();
		if (arr.length > 0) {
			try {
				sep_num = Integer.parseInt(String.valueOf(arr[0]));
				}
			catch (NumberFormatException e) {
				sep_char = arr[0];
				}
			}
		if (arr.length > 1) {
			sep_char = arr[1];
			}
		}
	Map m = new HashMap();
	m.put("sep_num", sep_num);
	m.put("sep_char", sep_char);	
	return m;
	}

/** 
returns a 8 char binary digit string. 
<p>The format string has a number and separator, like "4 " which will print chunks of size <num>
with the specified <separator>. The number must be between [1-9] and separator must be a single
character, like: | or space. Errors in the format string are ignored and default values are used
instead (4 and space).
*/
public static String toBinaryString(byte i, String format) 
	{
	String result = toBinaryString(i);
	
	Map m = getFormatMap(format);
	int sep_num = (Integer) m.get("sep_num");
	char sep_char = (Character) m.get("sep_char");
	
	StringBuilder builder = new StringBuilder();
	for (int n = 8, pos = 1; n > 0; n--, pos++) 
		{
		builder.append(result.charAt(n-1));
		if ((pos % sep_num == 0) && n > 1) {
			builder.append(sep_char);
			}
		}
	return builder.reverse().toString();
	}

/** 
returns a 16 char binary digit string. 
<p>The format string has a number and separator, like "4 " which will print chunks of size <num>
with the specified <separator>. The number must be between [1-9] and separator must be a single
character, like: | or space. Errors in the format string are ignored and default values are used
instead (4 and space).
*/
public static String toBinaryString(short i, String format) 
	{
	String result = toBinaryString(i);
	
	Map m = getFormatMap(format);
	int sep_num = (Integer) m.get("sep_num");
	char sep_char = (Character) m.get("sep_char");
	
	StringBuilder builder = new StringBuilder();
	for (int n = 16, pos = 1; n > 0; n--, pos++) 
		{
		builder.append(result.charAt(n-1));
		if ((pos % sep_num == 0) && n > 1) {
			builder.append(sep_char);
			}
		}
	return builder.reverse().toString();
	}
	
/** 
returns a 16 char binary digit string. 
<p>The format string has a number and separator, like "4 " which will print chunks of size num with
the specified separator.
*/
public static String toBinaryString(char i, String format) 
	{
	String result = toBinaryString(i);
	
	Map m = getFormatMap(format);
	int sep_num = (Integer) m.get("sep_num");
	char sep_char = (Character) m.get("sep_char");
	
	StringBuilder builder = new StringBuilder();
	for (int n = 16, pos = 1; n > 0; n--, pos++) 
		{
		builder.append(result.charAt(n-1));
		if ((pos % sep_num == 0) && n > 1) {
			builder.append(sep_char);
			}
		}
	return builder.reverse().toString();
	}


/** returns a 32char binary digit string (Long.toBinaryString is not padded, can be less than 
32 chars). 
<p>The format string has a number and separator, like "4 " which will print chunks of size <num>
with the specified <separator>. The number must be between [1-9] and separator must be a single
character, like: | or space. Errors in the format string are ignored and default values are used
instead (4 and space).
*/
public static String toBinaryString(int i, String format) 
	{
	String result = toBinaryString(i);
	
	Map m = getFormatMap(format);
	int sep_num = (Integer) m.get("sep_num");
	char sep_char = (Character) m.get("sep_char");
	
	StringBuilder builder = new StringBuilder();
	for (int n = 32, pos = 1; n > 0; n--, pos++) 
		{
		builder.append(result.charAt(n-1));
		if ((pos % sep_num == 0) && n > 1) {
			builder.append(sep_char);
			}
		}
	return builder.reverse().toString();
	}

/** returns a 64 char binary digit string (Long.toBinaryString is not padded, can be less than 
64 chars). 
<p>The format string has a number and separator, like "4 " which will print chunks of size <num>
with the specified <separator>. The number must be between [1-9] and separator must be a single
character, like: | or space. Errors in the format string are ignored and default values are used
instead (4 and space).
*/
public static String toBinaryString(long i, String format) 
	{
	String result = toBinaryString(i);
	
	Map m = getFormatMap(format);
	int sep_num = (Integer) m.get("sep_num");
	char sep_char = (Character) m.get("sep_char");
	
	StringBuilder builder = new StringBuilder();
	for (int n = 64, pos = 1; n > 0; n--, pos++) 
		{
		builder.append(result.charAt(n-1));
		if ((pos % sep_num == 0) && n > 1) {
			builder.append(sep_char);
			}
		}
	return builder.reverse().toString();
	}
	
public static void main (String args[])
	{
	int i = Integer.parseInt("1000", 2);
	System.out.println("[int]isBitSet('1000'/"+i+",3):"+isBitSet(i,3));

	i = Integer.parseInt("0000", 2);
	System.out.println("[int]isBitSet('0000'/"+i+",3):"+isBitSet(i,3));		

	i = Integer.parseInt("1001", 2);
	System.out.println("[int]isBitSet('1001'/"+i+",2):"+isBitSet(i,2));		

	i = Integer.parseInt("1001", 2);
	System.out.println("[int]isBitSet('1001'/"+i+",0):"+isBitSet(i,0));		

	i = Integer.parseInt("1000000000000000000000000000000", 2);
	System.out.println("[int]isBitSet('1000000000000000000000000000000'/"+i+",2):"+isBitSet(i,2));		

	i = Integer.parseInt("1001", 2);
	try {
		System.out.println("[int]isBitSet('1001'/"+i+",31):"+isBitSet(i,31));		
		}
	catch (Exception e) {
		System.out.println(e);
		}
		
	long lng = Long.parseLong("11001100", 2);
	System.out.println("[long]isBitSet('11001100'/"+lng+",3):"+isBitSet(lng,3));

	char c = (char) Integer.parseInt("1001000010010000", 2);
	System.out.println("[char]isBitSet('1001000010010000'/"+c+",1):"+isBitSet(c,1));
	System.out.println("[char]isBitSet('1001000010010000'/"+c+",15):"+isBitSet(c,15));

	byte b = (byte) Integer.parseInt("00000100", 2);

	System.out.println("[byte]isBitSet('00000100'/"+b+",1):"+isBitSet(b,0));
	System.out.println("[byte]isBitSet('00000100'/"+b+",3):"+isBitSet(b,2));

	short s = (short) Integer.parseInt("1000100010001000", 2);
	System.out.println("[short]isBitSet('1000100010001000'/"+s+",1):"+isBitSet(s,0));
	System.out.println("[short]isBitSet('1000100010001000'/"+s+",3):"+isBitSet(s,15));
	
	System.out.println("------- set and clear bits ----------");
	System.out.println("------- char ----------");
	System.out.print("c="+toBinaryString(c));
	c = setBit(c, 0);
	System.out.println("->setBit(c, 0)->"+toBinaryString(c));
	System.out.println("isBitSet(c, 0):"+isBitSet(c, 0));	

	System.out.print("c="+toBinaryString(c));
	c = setBit(c, 1);
	System.out.println("->setBit(c, 1)->"+toBinaryString(c));
	System.out.println("isBitSet(c, 1):"+isBitSet(c, 1));	

	System.out.print("c="+toBinaryString(c));
	c = clearBit(c, 1);
	System.out.println("->clearBit(c, 1)->"+toBinaryString(c));
	System.out.println("isBitSet(c, 1):"+isBitSet(c, 1));	

	System.out.println("------- long ----------");
	System.out.println("lng="+toBinaryString(lng,","));
	lng = setBit(lng, 63);
	System.out.println("->setBit(lng, 63)->\n    "+toBinaryString(lng,","));
	System.out.println("isBitSet(lng, 63):"+isBitSet(lng, 63));	
	lng = clearBit(lng, 63);
	System.out.println("->clearBit(lng, 63)->\n    "+toBinaryString(lng,","));
	System.out.println("isBitSet(lng, 63):"+isBitSet(lng, 63));	

	System.out.println("------- int ----------");
	System.out.print("i="+toBinaryString(i,"|"));
	i = setBit(i, 20);
	System.out.println("->setBit(i, 20)-> "+toBinaryString(i,"|"));
	System.out.println("isBitSet(i, 20):"+isBitSet(i, 20));	
	System.out.print("i="+toBinaryString(i,"|"));
	i = clearBit(i, 20);
	System.out.println("->clearBit(i,20)->"+toBinaryString(i,"|"));
	System.out.println("isBitSet(i, 20):"+isBitSet(i, 20));	

	System.out.println("------- int ----------");
	System.out.print("i="+toBinaryString(i,"|"));
	i = setBit(i, 31);
	System.out.println("->setBit(i, 31)-> "+toBinaryString(i,"|"));
	System.out.println("isBitSet(i, 31):"+isBitSet(i, 31));	
	System.out.print("i="+toBinaryString(i,"|"));
	i = clearBit(i, 31);
	System.out.println("->clearBit(i,31)->"+toBinaryString(i,"|"));
	System.out.println("isBitSet(i, 31):"+isBitSet(i, 31));	

	System.out.println("------- byte ----------");	
	System.out.println("b="+toBinaryString(b));
	b = setBit(b, 7);
	System.out.println("->setBit(b, 7)->" + toBinaryString(b));
	System.out.println("isBitSet(b, 7):"+isBitSet(b, 7));	
	System.out.print("b="+toBinaryString(b));
	b = clearBit(b, 7);
	System.out.println("->clearBit(b, 7)->"+toBinaryString(b));
	System.out.println("isBitSet(b, 7):"+isBitSet(b, 7));	

	System.out.println("------- short ----------");	
	System.out.print("s="+toBinaryString(s,"|"));
	s = setBit(s, 10);
	System.out.println("->setBit(s, 10)->" + toBinaryString(s,"|"));
	System.out.println("isBitSet(s, 10):"+isBitSet(s, 10));	
	System.out.print("s="+toBinaryString(s,"|"));
	s = clearBit(s, 10);
	System.out.println("->clearBit(s, 10)->"+toBinaryString(s,"|"));
	System.out.println("isBitSet(s, 10):"+isBitSet(s, 10));	

	System.out.println("------- argument checks ----------");	
	try { setBit(b, 8); } catch (RuntimeException e) { }
	try { setBit(c, 16); } catch (RuntimeException e) { }
	try { setBit(s, 16); } catch (RuntimeException e) { }
	try { setBit(i, 32); } catch (RuntimeException e) { }
	try { setBit(lng, 64); } catch (RuntimeException e) { }

	try { setBit(b, 7); } catch (RuntimeException e) { }
	try { setBit(c, 15); } catch (RuntimeException e) { }
	try { setBit(s, 15); } catch (RuntimeException e) { }
	try { setBit(i, 31); } catch (RuntimeException e) { }
	try { setBit(lng, 63); } catch (RuntimeException e) { }

	try { setBit(b, -1); } catch (RuntimeException e) { }
	try { setBit(c, -1); } catch (RuntimeException e) { }
	try { setBit(s, -1); } catch (RuntimeException e) { }
	try { setBit(i, -1); } catch (RuntimeException e) { }
	try { setBit(lng, -1); } catch (RuntimeException e) { }

	System.out.println("done...");
	}
}


