// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;
import java.io.*;

import EDU.oswego.cs.dl.util.concurrent.*;
import fc.util.StringUtil;

/* *************************************************************
A array based {@link
EDU.oswego.cs.dl.util.concurrent.BoundedBuffer BoundedBuffer}
implementation that provides a queue of chars that can
be read/written by consumers/producers. Various similar
classes in <a
href="http://gee.cs.oswego.edu/dl/classes/">util.concurrent </a> classes provide queues that only
support Object elements, this queue provides the ability
to use primitive char types. Some other features
include:
<ul>
<li>
Method signatures with <tt>char[]</tt> arrays as parameter - these
can be used for bulk put/get operations. (both blocking and non-
blocking versions are provided).
<li>
Input/Output Streams</tt> interfaces to the buffer are provided
(via the {@link #getReader()} and {@link getWriter()}
methods and these can be used to treat the buffer as both a 
source and sink respectively.
</ul>
@author 	hursh jain (derived from Doug Lea:util.concurrent)
**/
public final class BoundedCharQueue
{
//pretty much a direct approach from CPJ2 pg 223-224

/** The default capacity of the Queue: <tt>1024</tt> chars **/
public 	static 	final int 	 DEFAULT_CAPACITY 	= 1024;
		static 	final boolean dbg 				= false;
/* prints the array after each insertion/extraction if true */
		static 	final boolean dbg2			    = false;
		
final	Semaphore putPermits;
final 	Semaphore takePermits;
final 	CharArray array;  

/**
 * Create a BoundedCharQueue with the given capacity. 
 * @exception IllegalArgumentException if capacity less or equal to zero
**/
public BoundedCharQueue(int capacity) throws IllegalArgumentException 
	{
    if (capacity <= 0) 
		throw new IllegalArgumentException();
	putPermits = new Semaphore(capacity);
	takePermits = new Semaphore(0);
    array = new CharArray(capacity);
  	}

/**
Create a buffer with the current default capacity. The Default
capacity is specified via the {@link #DEFAULT_CAPACITY}.
*/
public BoundedCharQueue() { 
	this(DEFAULT_CAPACITY); 
  	}

/** 
 * Return the number of elements in the buffer.
 * This is only a snapshot value, that may change
 * immediately after returning.
**/
public synchronized int available() { 
  	return (int) takePermits.permits(); 
	}

/** 
Returns the maximum number of chars that can be held
in this queue
**/
public int capacity() { 
	return array.capacity();
	}


//-- puts ----------------------------------------
		
/** 
Places the specified char in the queue, possibly waiting
indefinitely until it can be accepted. 

@param	val		the char to be inserted

@throws 		java.lang.InterruptedException - if the current 
thread has been interrupted at a point at which interruption is
detected, in which case the value is guaranteed not to be
inserted. Otherwise, on normal return, the value is guaranteed
to have been inserted.
**/		
public void put(char val) throws InterruptedException 
	{
	putPermits.acquire();
	array.insert(val);
    takePermits.release();
   	}

/** 
Writes len chars from the specified char array starting
at offset off to this queue. If space for len chars  
are not available, this method will block indefinitely 
until such space <i>is</i> available.

@param	buf		the char array containing values to be inserted
				(offset and len have the usual meanings)
@throws 		java.lang.InterruptedException - if the current 
                thread has been interrupted at a point at which
                interruption is detected, in which case values
                are guaranteed not to be inserted. Otherwise, on
                normal return, values are guaranteed to have
                been inserted.
**/
public void put(char[] buf, int offset, int len) throws InterruptedException 
	{
	bufcheck(buf, offset, len, true);
		
	if (len == 0) { //nothing to do
		return;
		}		

	if (dbg) System.out.println("put(): Producer waiting for " + len + " permits");
	putPermits.acquire(len);
	if (dbg) 
		System.out.println("put(): Producer got for " + len + " permits");
	array.insert(buf, offset, len);
   	takePermits.release(len);
   	}		

/** 
Place item in queue only if it can be accepted within msecs
milliseconds. The time bound is interpreted in a coarse-grained,
best-effort fashion. 

@param	val		the char to be inserted
@param	msecs	the number of milliseconds to wait. If less
                than or equal to zero, the method does not
                perform any timed waits, but might still
                require access to a synchronization lock,
                which can impose unbounded delay if there is
                a lot of contention for the channel.
@return	true 	if accepted, else false 
@throws			java.lang.InterruptedException - if the current 
                thread has been interrupted at a point at which
                interruption is detected, in which case the
                element is guaranteed not to be inserted (i.e.,
                is equivalent to a false return).
**/
public boolean offer(char x, long msecs) throws InterruptedException 
	{
	if (! putPermits.attempt(msecs)) 
		return false;
	array.insert(x);
	takePermits.release();
	return true;
	}

/**
Places the specified chars in the channel only if all the specified
chars can be accepted within msecs milliseconds. The time bound is
interpreted in a coarse-grained, best-effort fashion. 

@param	buf		the char array containing values to be inserted
				(offset and len have the usual meanings)
@param	msecs	the number of milliseconds to wait. If less
                than or equal to zero, the method does not
                perform any timed waits, but might still
                require access to a synchronization lock,
                which can impose unbounded delay if there is
                a lot of contention for the channel.
@throws 		java.lang.InterruptedException - if the current 
                thread has been interrupted at a point at which
                interruption is detected, in which case values
                are guaranteed not to be inserted. Otherwise, on
                normal return, values are guaranteed to have
                been inserted.
**/
public boolean offer(char[] buf, int offset, int len, long msecs) 
throws InterruptedException 
	{
	bufcheck(buf, offset, len, true);

	if (dbg) System.out.println("offer(): Consumer waiting for " + len + " permits");
	if (! putPermits.attempt(len, msecs))
		return false;
	if (dbg) System.out.println("offer(): Consumer got " + len + " permits");
	array.insert(buf, offset, len);
   	takePermits.release(len);
   	return true;		
	}		

//-- takes -----------------------------------------

/* peek is tricky so we don't support it
* public int peek(long msecs) { }
*/

/** 
Return and remove a char from the queue, possibly waiting
indefinitely until a char can be read.

@throws 	java.lang.InterruptedException - if the current thread
            has been interrupted at a point at which interruption
            is detected, in which case state of the channel is
            unchanged.
**/
public char take() throws InterruptedException 
	{ 
    takePermits.acquire();
	char val = (char) array.extract();
   	putPermits.release();
   	return val;
	}

/*
we have to make at least one of put or take methods non-blocking,
meaning the ability to proceed to take/put data even if the entire
array cannot be filled/written. (see ByteCharQueue code for
more details)
*/

/** 
Reads as many chars as possible without blocking into the
specified array. 

@param	buf	 the char array to read the values into.
			 (offset and len have the usual meanings)
@return 	 the number of chars read
@throws 	 java.lang.InterruptedException - if the current thread
             has been interrupted at a point at which interruption
             is detected, in which case state of the channel is
             unchanged.
**/
public int take(char[] buf, int offset, int len) 
throws InterruptedException 
	{
	bufcheck(buf, offset, len, false);
	//capacity check: we can be passed a much bigger array
	//to read into of course (for example BufferedReader might
	//use 8192 bytes and try to read all of them at one go).
	//we only return a max of array.capacity() of course. 

	if (len > array.capacity()) {
		len = array.capacity();
		}	

	String str = null;
	
	if (dbg) {
		str = "take(): buf.length="+buf.length+";offset="+offset+";len="+len;
		}
		
	if (len == 0) { //nothing to do
		return 0;
		}		
	
	//qsize = 60, prod chunk = 35, consumer = 40 (this method)
	//1. prod produces 35, waits
	//2. consumer tries 40, 39, 38, 37, 35, gets 35, back to 1
	//this is very time consuming. Perhaps a initial short cut
	//where we see how much is available initially and make
	//that our start amount. This is tricky since if the available
	//is 0, then we have to skip this step, otherwise we'll try
	//to semaphore.acquire(0) but we need to acquire at least
	//one semaphore
	
	int start = (int) takePermits.permits();
	if (start > 0) {
		len = (len < start) ? len : start;
		}
	else {
		len = 1;
		}
				
	for (;;) {
		if (dbg) System.out.println("take(): Consumer waiting for " + len + " permits");
		if (takePermits.attempt(len, 10)) {
			if (dbg) System.out.println("take(): Consumer got " + len + " permits");
			break;
			}
		if (dbg)
			System.out.println("take(): downgrading read len to " + len);
		len--;
		if (len == 0)
			return 0;
		}

	if (dbg) {
		str += "->buf.length="+buf.length+";offset="+offset+";len="+len;		
		System.out.println(str);
		}
		
	array.extract(buf, offset, len);
	putPermits.release(len);
	return len;
	}		


/** 
Return and remove a char from the channel. The time bound is 
interpreted in a coarse grained, best-effort fashion.

@return		the value (as <u>unsigned integer</u>) or -1 if the channel 
			is empty and no value could be read in the specified 
			time. 
@throws 	java.lang.InterruptedException - if the current thread
            has been interrupted at a point at which interruption
            is detected, in which case state of the channel is
            unchanged (i.e., equivalent to a -1 return).
**/
public int poll(long msecs) throws InterruptedException 
	{ 
	if (! takePermits.attempt(msecs))
		return -1;
	int val = array.extract() & 0xFFFF;
  	putPermits.release();    
	return val;
	}

/** 
Tries to read the specified number of chars into the array.
Does <u>not</u> read a partial number of chars. If the specified
number of chars cannot be read, returns <tt>false</tt>.

@param  buf	the char array to read the values into.
			(offset and len have the usual meanings)
@return 	<tt>true</tt> on success, <tt>false</tt> otherwise
@throws 	java.lang.InterruptedException - if the current thread
            has been interrupted at a point at which interruption
            is detected, in which case state of the channel is
            unchanged (i.e., equivalent to a <tt>false</tt> return).
**/
public boolean poll(char[] buf, int offset, int len, long msecs) 
throws InterruptedException 
	{
	bufcheck(buf, offset, len, true);

	if (len == 0) { //nothing to do
		return true;
		}
	if (! takePermits.attempt(len, msecs))
		return false;
	array.extract(buf, offset, len);
	putPermits.release(len);
	return true;
	}

//-- misc --------------------------------------------

void bufcheck(char[] buf, int offset, int len, boolean capacityCheck) 
	{
	if (buf == null) throw new NullPointerException ();
	if (offset < 0 || len < 0 || offset + len > buf.length) 
      	throw new IndexOutOfBoundsException ();
	if (capacityCheck){
		if (len > array.capacity()) 
			throw new IllegalArgumentException("Queue size=" + array.capacity() + "; cannot write more bytes than qsize. Attempted to write: " + len + " bytes");
		}		
	}


//-- I/O streams ------------------------------------

/**
Returns a Reader that can read from this queue.
**/
public Reader getReader() {
	return new ReaderView();
	}

/** 
Returns a Wrtier that writes into this queue. 
<p>
Note: the various <tt>write</tt> methods provided by this class
will block until the specified chars(s) are can be and are fully
written.
**/
public Writer getWriter() {
	return new WriterView();
	}


class WriterView extends Writer
{		
boolean closed;

public void close() { 
	closed = true;
	}
	
public void flush() { }

public void write(int b) throws IOException 
	{
	checkClosed();
	try {
		put((char)b);
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted: " + e.toString());
		}
	}

public void write(char[] buf, int off, int len) throws IOException
	{		
	checkClosed();
	try {
		put(buf, off, len);
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted or timed-out: " + e.toString());
		}
	}

void checkClosed() throws IOException {
	if (closed) {
		throw new IOException("Stream has been closed");
		}
	}

} //~WriterView


class ReaderView extends Reader
{
boolean closed;

public void close() {
	closed = true;
	}
	
public int read() throws IOException
	{
	try {
		return take();	
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted: " + e.toString());
		}
	}

public int read(char[] buf, int off, int len) throws IOException
	{
	checkClosed();
	try {
		return take(buf, off, len);
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted: " + e.toString());
		}
	}

void checkClosed() throws IOException {
	if (closed) {
		throw new IOException("Stream has been closed");
		}
	}

} //~ReaderView


public class CharArray
{
final 	char[] 	carray;
		int		arraylen;
		int		putPtr = 0;
		int		takePtr = 0;

CharArray(int size) 
	{
	carray = new char[size];
	arraylen = size;
	}

final int capacity() {
	return arraylen;
	}

final void insert(char val) 
	{
	carray[putPtr] = val;
	putPtr = (putPtr + 1) % arraylen;
	}

final void insert(char[] buf, int offset, int len)
	{
	int oldPutPtr = 0;
	if (dbg)
		oldPutPtr = putPtr;

	int contigspace = arraylen - putPtr;  //slots to end of array

	if (len <= contigspace) {
		System.arraycopy(buf, offset, carray, putPtr, len);
		}
	else 
		{
		System.arraycopy(buf, offset, carray, putPtr, contigspace);
		System.arraycopy(buf, 
			offset + contigspace, carray, 0, len - contigspace);
		}	
	putPtr = (putPtr + len) % arraylen;

	if (dbg) 
		{
 		//NOTE: emptySlots can be conc. increased by consumer thread(s) so subsequent
		//printouts of this line may not decrease emptyslots exactly by len
		String str = "--insert(): putPtr=["+oldPutPtr+"->"+putPtr+"];takePtr=["+takePtr+"]";
		if (dbg2) {
			str += "array="+StringUtil.arrayToString(carray);
			}
		System.out.println(str);
		}
	}

// we return unsigned int to support a return value of -1 by
// methods that which to signal a eof type return value
final int extract() 
	{ 
    int val = carray[takePtr] & 0xFFFF;
	//don't need to do this but useful for debugging
	if (dbg) 
		carray[takePtr] = '0';
	takePtr = (takePtr + 1) % arraylen;
    return val;
	}

final void extract(char[] buf, int offset, int len) 
	{
	int oldTakePtr = 0;
	if (dbg)
		oldTakePtr = takePtr;
		
	int buflen = buf.length;
	/* # of slots to end of array */
	int contigspace = arraylen - takePtr;
	
	if (len <= contigspace) {
		System.arraycopy(carray, takePtr, buf, offset, len);
		}
	else 
		{
		System.arraycopy(carray, takePtr, buf, offset, contigspace);
		System.arraycopy(carray, 0, buf, 
			offset + contigspace, len - contigspace);
		}
	takePtr = (takePtr + len) % arraylen;
	if (dbg) {
		//NOTE: usedSlots can be conc. increased by producer thread(s) so subsequent
		//printouts of this line may not decrease usedslots exactly by len
		String str = "extract(): putPtr=["+putPtr+"];takePtr=["+oldTakePtr+"->"+takePtr+"]";
		if (dbg2) {
			str += ";extracted="+StringUtil.arrayToString(buf);
			}
		System.out.println(str);
		}
	}

} //~CharArray

//-- Unit Testing ----------------------------------------

/** Unit Test. Usage: java fc.util.BoundedCharQueue -help **/
public static void main(String[] args) throws InterruptedException
	{
	Args myargs = new Args(args);
	myargs.setUsage("java fc.util.BoundedCharQueue " +
		"\n -help  display these options" +
		"\n -producerSleep  time  sleep millisecs (speed) for producer thread (default: no sleep)" +
		"\n -consumerSleep  time  sleep millisecs (speed) for consumer thread (default: no sleep)" +
		"\n -producerBlock  time  producer blocks only for millisecs (default: blocks forever)" +
		"\n -consumerBlock  time  consumer blocks only for millisecs (default: blocks forever)" +
		"\n -qsize          size  optional queue size (default: " + DEFAULT_CAPACITY + ")" +
		"\n -producerNum    num   the number of producers (default: 1)" +
		"\n -consumerNum    num   the number of consumers (default: 1)" + 
		"\n -consumerChunk  size  use a char array of the specified size to consume" +
		"\n -producerChunk  size  use a char array of the specified size for producing" +
		"\n -useStream            use streams to read/write to the queue" +
		"\n -verbose              to print more detailed information"
		);

	if (myargs.flagExists("help"))
		myargs.showError();
	
	boolean useStream = myargs.flagExists("useStream");
	boolean verbose = myargs.flagExists("verbose");
		
	int producerSleepTime = Integer.parseInt(myargs.get("producerSleep", "-1"));
	int consumerSleepTime = Integer.parseInt(myargs.get("consumerSleep", "-1"));
	
	boolean producerBlockForever = true;
	boolean consumerBlockForever = true;
	
	int producerBlockTime = -1;
	int consumerBlockTime = -1;
	
	if (myargs.flagExists("producerBlock")) {
		producerBlockForever = false;
		producerBlockTime = Integer.parseInt(myargs.getRequired("producerBlock"));	
		}

	if (myargs.flagExists("consumerBlock")) {
		consumerBlockForever = false;
		consumerBlockTime = Integer.parseInt(myargs.getRequired("consumerBlock"));	
		}

	BoundedCharQueue q = null;
	
	if (myargs.flagExists("qsize"))	
		q = new BoundedCharQueue(Integer.parseInt(myargs.getRequired("qsize")));
	else
		q = new BoundedCharQueue();

	int producerChunk = Integer.parseInt(myargs.get("producerChunk", "-1"));
	int consumerChunk = Integer.parseInt(myargs.get("consumerChunk", "-1"));

	int prodNum = Integer.parseInt(myargs.get("producerNum", "1"));
	int conNum = Integer.parseInt(myargs.get("consumerNum", "1"));
	
	//waits for all threads to be constructed before they are
	//started
	Latch ready = new Latch();
	
	if (! useStream)
		{
		System.out.println("Creating " + prodNum + " producer(s)..");	
		for (int n = 0; n < prodNum; n++) 
		  new ProducerThread(
		  	verbose, ready, n+1, q, 
		  	producerSleepTime, producerBlockTime, 
			producerBlockForever, producerChunk);
		
		System.out.println("Creating " + conNum + " consumer(s)..");
		for (int n = 0; n < conNum; n++) 
		  new ConsumerThread(
			verbose, ready, n+1, q, 
			consumerSleepTime, consumerBlockTime, 
			consumerBlockForever, consumerChunk);
		}	
	else
		{
		System.out.println("Creating " + prodNum + " producer(s)..");	
		for (int n = 0; n < prodNum; n++) 
		  new ProducerThreadStream(verbose, ready, n+1, q.getWriter(), 
		  	producerSleepTime, producerChunk);
		
		System.out.println("Creating " + conNum + " consumer(s)..");
		for (int n = 0; n < conNum; n++) 
		  new ConsumerThreadStream(verbose, ready, n+1, q.getReader(), 
			consumerSleepTime, consumerChunk);
		}

	ready.release(); //start them all
	}


private static class ProducerThread extends Thread 
	{
	boolean verbose;
	static volatile int totalWritten = 0;
	String id;
	int waitTime;
	int sleepTime;
	BoundedCharQueue q;
	boolean block;
	char[] buf;
	Latch ready;
		
	public ProducerThread(boolean verbose, Latch ready, int id, BoundedCharQueue q, int sleepTime, 
		int waitTime, boolean block, int chunksize) 
		{
		super("Producer#"+id);
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.q = q;
		this.sleepTime = sleepTime;
		this.waitTime = waitTime;		
		this.block = block;
		if (chunksize >= 0)
			this.buf = new char[chunksize];
		System.out.println("Producer thread# " + id + " start, sleep=" + 
		 sleepTime + "ms " +
		 (! block ? "will wait only for: " + waitTime + "ms" : " "));
		start();
		}
			
	public void run() 
		{
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
		
		while (true) 
			{
			char c = '?';
			
			if (buf == null)
			 	c = testx.nextchar();
			else
				testx.fillbuf(buf);
				
			boolean suxez = true;
			try {
				if (block) 
					{
					if (buf == null) 
						{
						q.put(c);
						totalWritten++;
						if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced char: [" +  c + "]");
						}
					else 
						{
						q.put(buf, 0, buf.length);
						totalWritten += buf.length;
						if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced array: " + StringUtil.arrayToString(buf));
						}
					}
				else //non blocking
					{
					if (buf == null) 
						{
						suxez = q.offer(c, waitTime);
						if (! suxez)
							if (verbose) System.out.println("Producer#" + id + " timeout, could not put: " + c);
						else
							totalWritten++;
						}
					else 
						{
						suxez = q.offer(buf, 0, buf.length, waitTime);
						if (! suxez)
							if (verbose) System.out.println("Producer#" + id + " timeout, could not put: " + StringUtil.arrayToString(buf));					
						else
							totalWritten += buf.length;
						}
					} //non-blocking
				}
			catch (Exception e) {
				e.printStackTrace();
				}
				
			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}
			testx.showStatus(totalWritten);
			}		
		} //~run
		
	} //~class ProducerThread


private static class ConsumerThread extends Thread 
	{
	static volatile int totalRead = 0;
	boolean verbose; 
	String id;
	int waitTime;
	int sleepTime;
	BoundedCharQueue q;
	boolean block;
	char[] buf;
	Latch ready;

	public ConsumerThread(boolean verbose, Latch ready, int id, BoundedCharQueue q, int sleepTime, 
		int waitTime, boolean block, int chunksize) 
		{
		super("Consumer#"+id);
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.q = q;
		this.sleepTime = sleepTime;
		this.waitTime = waitTime;		
		this.block = block;
		if (chunksize >= 0)
			this.buf = new char[chunksize];
		System.out.println("Consumer thread#" + id + " start, sleep=" + 
		 sleepTime + " ms " +  
		 (! block ? "will wait only for: " + waitTime + " ms" : " "));
		start();
		}
	
	public void run() 
		{
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
		
		while (true) 
			{
			try {
				if (block) 
					{
					if (buf == null) 
						{
						char c = q.take();
						totalRead++;
						if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read char: [" + c + "]");
						}
					else 
						{
						int read = q.take(buf, 0, buf.length);	
						totalRead += read;
						if (verbose) System.out.println("Consumer#" + id + " [total=" + totalRead + "]: read buffer: " + StringUtil.arrayToString(buf));
						}
					}
				else { //non-block
					if (buf == null) 
						{
						int b = q.poll(waitTime);
						if (b < 0)
							if (verbose) System.out.println("Consumer#" + id + " timed out...");
						else {
							totalRead++;
							if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read char: " + b);
							}
						}
					else 
						{
						boolean suxez = q.poll(buf, 0, buf.length, waitTime);	
						if (suxez)
							if (verbose) System.out.println("Consumer#" + id + " timed out...");
						else {
							totalRead += buf.length;
							if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read char array: " + StringUtil.arrayToString(buf));
							}
						}
					} //~non-block
				}
			catch (Exception e) {
				e.printStackTrace();
				}

			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}

			}		
		} //~run

	} //~class ConsumerThread


//-- stream versions -------------------------------------------

private static class ProducerThreadStream extends Thread 
	{
	boolean verbose;
	static volatile int totalWritten = 0;
	String id;
	int sleepTime;
	Writer out;
	char[] buf;
	Latch ready;
		
	public ProducerThreadStream(boolean verbose, Latch ready, 
		int id, Writer out, int sleepTime, int chunksize) 
		{
		super("Producer#"+id);		
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.out = out;
		this.sleepTime = sleepTime;
		if (chunksize >= 0)
			this.buf = new char[chunksize];
		System.out.println("Producer thread# " + id + " start, sleep=" + 
		 sleepTime + "ms ");
		start();
		}
			
	public void run() 
		{
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
	
		while (true) 
			{
			char c = 0;
			
			if (buf == null)
			 	c = testx.nextchar();
			else
				testx.fillbuf(buf);
				
			boolean suxez = true;
			try {
                if (buf == null) 
                	{
                	out.write(c);
               	 	totalWritten++;
                	if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced char: [" +  c + "]");
                	}
                else 
                	{
                	out.write(buf, 0, buf.length);
                	totalWritten += buf.length;
                	if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced array: " + StringUtil.arrayToString(buf));
                	}
				}
			catch (Exception e) {
				e.printStackTrace();
				}
				
			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}
			testx.showStatus(totalWritten);	
			} //~while		
		} //~run

	} //~class ProducerThreadStream

private static class ConsumerThreadStream extends Thread 
	{
	boolean verbose;
	static volatile int totalRead = 0;
	String id;
	int sleepTime;
	Reader in;
	char[] buf;
	Latch ready;

	public ConsumerThreadStream(boolean verbose, Latch ready, 
		int id, Reader in, int sleepTime, int chunksize) 
		{
		super("Consumer#"+id);
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.in = in;
		this.sleepTime = sleepTime;
		if (chunksize >= 0)
			this.buf = new char[chunksize];
		System.out.println("Consumer thread#" + id + 
						" start, sleep=" + sleepTime + " ms " );		
		start();
		}
	
	public void run() 
		{
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
		
		while (true) 
			{
			try {
				if (buf == null) 
					{
					int c = in.read();
					if (c >= 0) {
						totalRead++;
						if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read char: [" + c + "]");
						}
					else
						if (verbose) System.out.println("Consumer#" + id + " could not read char");
					}
				else 
					{
					int read = in.read(buf, 0, buf.length);	
					totalRead += read;
					if (verbose) System.out.println("Consumer#" + id + " [total=" + totalRead + "]: read buffer: " + StringUtil.arrayToString(buf));
					}
				}
			catch (Exception e) {
				e.printStackTrace();
				}

			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}
			}		

		} //~run

	} //~class ConsumerThreadStream


//-- support class for test threads -------

private static class testx
	{
	static char fillstart = '0';
	
	static void fillbuf(char[] buf) 
		{
		if (fillstart > 127) 
			fillstart = '0';
		for (int n = 0; n < buf.length; n++)
			buf[n] = fillstart++;
		}
	
	static char nextchar() {
		if (fillstart > 127) fillstart = '0';
		return fillstart++;
		}

	static long prod_oldtime;
	static final int ShowProducedAmount = 100000;
	static int triggerLevel = ShowProducedAmount;
	static boolean showMsg = false;
	
	static void showStatus(int totalWritten) 
		{	
		//Depending on the producer chunk size, we may never
		//write 100,000 chars exactly. so we cannot use 
		//totalWritten % 100,000 for example
	
		if (totalWritten > triggerLevel) {
			showMsg = true;	
			triggerLevel = triggerLevel + ShowProducedAmount ;
			}
		
		if (showMsg) {
			showMsg = false;
			double temp = prod_oldtime;
			prod_oldtime = System.currentTimeMillis();
			System.out.println("=== 100,000 more items produced (total=" + totalWritten + ") in: " + (prod_oldtime - temp)/1000 + " seconds =====");
			}
		}

	}		

} //~class BoundedCharQueue


