// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;
import java.io.*;

import EDU.oswego.cs.dl.util.concurrent.*;
import fc.util.StringUtil;

/* *************************************************************
A array based {@link
EDU.oswego.cs.dl.util.concurrent.BoundedBuffer BoundedBuffer}
implementation that provides a queue of bytes that can
be read/written by consumers/producers. Various similar
classes in <a
href="http://gee.cs.oswego.edu/dl/classes/">util.concurrent </a> classes provide queues that only
support Object elements, this queue provides the ability
to use primitive byte types. Some other features
include:
<ul>
<li>
Method signatures with <tt>byte[]</tt> arrays as parameter - these
can be used for bulk put/get operations. (both blocking and non-
blocking versions are provided).
<li>
Input/Output Streams</tt> interfaces to the buffer are provided
(via the {@link #getInputStream()} and {@link getOutputStream()}
methods and these can be used to treat the buffer as both a source
and sink respectively.
</ul>
<p>
Note: This class does not implement the {@link
EDU.oswego.cs.dl.util.concurrent.Channel Channel}
from util.concurrent because it has method signatures
that are incompatible - methods take primitive types (and
return primitive values) whereas 
{@link EDU.oswego.cs.dl.util.concurrent.Channel Channel} 
only deals with Objects.

@author 	hursh jain (derived from Doug Lea:util.concurrent)
**/
public final class BoundedByteQueue
{
/*
Note, in the original code, notify()'s were sent after
every change via the incXXX() methods - but really notifications
should need only be sent when transitions are made out of
empty/full states. I emailed dlea about this and here's his
reply (mine quoted):

---------------------------------------------
> I can see why you seperated out the locks for put/take
> threads to increase read/write concurrency. But in the
> incEmptySlots() and incUsedSlots() methods, you
> *always* notify waiting threads and the incEmpty/Used
> methods are *always* internally called whenever there 
> is any change to the array via the put/take/offer/poll
> methods..

It turns out that on most current JVMs, the overhead of testing
whether to issue an unnecessary notify is not worth it -- an empty
notify itself should be about equally fast, and makes the code a
little simpler.

(BTW, as of JDK1.5 (JSR166), this code will be massively simpler,
because JVMs will intriniscally support multiple monitor conditions
per object.)
-----------------------------------------

Oh well, it's still aesthetically pleasing to notify only when
necessary and dlea specifically discusses this point in CPJ
contrary to his email. But it becomes very tricky finding
out what the necessary points are, for notification, when
we can put/get byte[] arrays at a time. Initially, for
example, in this code, consumers(takers) where only notifying
producers(putters) when emptyslots went from 0 ->n [n=some
number] (say 1). However, producers who were trying to read a 
byte[10] array would wait until 10 elements would be available 
and would ignore the 0->1 notification (they would get it,
see that avail. slots were less then 10, go back to sleep). 
However, consumers would not notify again from 1->n and
the producer thread(s) would sleep forever. That is, our
state transition that affects waiting threads is not just
those that step away from empty/full [cpj2, pg200] but (in 
the presence of byte[]), those that step away from producers 
waiting for byte[] sized empty spaces. We could keep
track of the different numbers that producer threads was
waiting on and notify anytime that number of slots was reached
(and vice versa) but that becomes a little too complicated.

So we end up notifying after every change, just as in the
original dlea code. We can complicate things further by
using a notify threshold level and notify after _every_ change
as long as the current state is above the threshold level.
All changes above this level result in notifications. This
is error prone too, say:

Queue size = 10, threshold level = 10
producer produces in byte[3] chunks and writes 3 chunks
producer waits until 3 spaces are free. Does not notify
consume since threshold level of used spaces (10) not reached.

consumer reads upto byte 9
does not notify producers since threshold not reached so
producer does not know that more empty spaces are available.

Both block. We can try to solve this by not allowing threshold
level to be equal to queue size but we would still be error-prone
for any threshold level where the producer blocks (waiting
to read a chunk) and the consumer blocks because it cannot
empty enough slots so that a) the theshold is reached b) more
than the producers's needed chunk size (byte[]) are empty.

Say: qsize = 10
notify threshold = 9
producer writes in [4] chunks
writes 8 bytes.
consumer reads 8 bytes
threshold not reached, producer waits forever.
consumer waits forever	

So threshold functionality although implemented has not
been removed.

Notes:
1) we use 2 pointers (take, put) because we use a circular (ring)
buffer and we need to maintain separate positions for
read and write.
2) The read and write wrap around but never cross each
other because when they touch, the pipe is either full
or empty, and if full, no more will be written and if
empty, no more will be read.

The final design experience from this class is to approach
MT design (for any non-trivial class) this way:
1) Isolate underlying class methods/operations in a MT-unsafe
class (CPJ[2ndEd], Pg 223) - treat all operations on this
class as being serialized.
2) Add MT based views to such an underlying class.

This makes it easier to separate/think about MT issues.
This approach has been taken for the BoundedCharQueue class.
*************************************************************/

/** The default capacity of the Queue: <tt>1024</tt> bytes **/
public 	final static int 	 DEFAULT_CAPACITY 	= 1024;
		final static boolean dbg 				= false;
/* prints the array after each insertion/extraction if true */
		final static boolean dbg2			    = false;
		
/* 
 we don't really need 2 locks (we can use the buffer
 itself as one of the locks) but this makes it clearer 
*/
final	Object  takeMonitor = new Object();
final 	Object 	putMonitor 	= new Object();

final 	byte[] 	array;     			// the elements
		int 	arraylen;
 		int 	takePtr 	= 0;	// circular indices
 		int 	putPtr 		= 0;     
 		int 	usedSlots 	= 0;   	// length
 		int 	emptySlots;      	// capacity - length

/**
 * Create a BoundedByteQueue with the given capacity. 
 * @exception IllegalArgumentException if capacity less or equal to zero
**/
public BoundedByteQueue(int capacity) throws IllegalArgumentException 
	{
    if (capacity <= 0) 
		throw new IllegalArgumentException();
    array = new byte[capacity];
	arraylen = array.length;
    emptySlots = capacity;
	if (dbg) {
		Arrays.fill(array, (byte)-1);
		}
	if (dbg) {
		System.out.println("takeMonitor="+takeMonitor);
		System.out.println("putMonitor="+putMonitor);
		}
  	}

/**
Create a buffer with the current default capacity. The Default
capacity is specified via the {@link #DEFAULT_CAPACITY}.
*/
public BoundedByteQueue() { 
	this(DEFAULT_CAPACITY); 
  	}


/** 
 * Return the number of elements in the buffer.
 * This is only a snapshot value, that may change
 * immediately after returning.
**/
public synchronized int available() { 
  	return usedSlots; 
	}

/** 
Returns the maximum number of bytes that can be held
in this queue
**/
public int capacity() { 
	return arraylen; 
	}

final void incUsedSlots(int amount) 
	{
	synchronized (takeMonitor)
		{
		takeMonitor.notify();
		usedSlots = usedSlots + amount;
		}	
	}

final void incEmptySlots(int amount)
	{
    synchronized(putMonitor) 
		{
		putMonitor.notify();
		emptySlots = emptySlots + amount;
    	}	
	}

final void insert(byte x) 
	{ 
    array[putPtr] = x;
    putPtr = (putPtr + 1) % arraylen;
    emptySlots--;
	}

//should only be called when it's guaranteed that we have
//len amount of empty space in the buffer
final void insert(byte[] buf, int offset, int len)
	{
	int oldPutPtr = 0;
	
	if (dbg)
		oldPutPtr = putPtr;

	//# of slots to end of array
	int contigspace = arraylen - putPtr;

	if (len <= contigspace) {
		System.arraycopy(buf, offset, array, putPtr, len);
		}
	else {
		System.arraycopy(buf, offset, array, putPtr, contigspace);
		System.arraycopy(buf, offset + contigspace, 
						 array, 0, len - contigspace);
		}
		
	putPtr = (putPtr + len) % arraylen;
	emptySlots = emptySlots - len;

	if (dbg) 
		{
 		//NOTE: emptySlots can be conc. increased by consumer thread(s) so subsequent
		//printouts of this line may not decrease emptyslots exactly by len
		String str = "--insert(): contigspace="+contigspace+";slots[empty="+emptySlots+"] putPtr=["+oldPutPtr+"->"+putPtr+"];takePtr=["+takePtr+"]";
		if (dbg2) {
			str += "array="+Arrays.toString(array);
			}
		System.out.println(str);
		}
	}

// we return unsigned int to support a return value of -1 by
// methods that which to signal a eof type return value
final int extract() 
	{ 
    int val = array[takePtr] & 0xFF;
 	//don't need to do this (g.c n/a) but useful for debugging
	array[takePtr] = -1;  
	takePtr = (takePtr + 1) % arraylen;
    usedSlots--;
    return val;
	}

//called with a lock on putPtr when we know we have at least len 
//usedSlots available in the buffer
final void extract(byte[] buf, int offset, int len) 
	{
	int oldTakePtr = 0;
	
	if (dbg)
		oldTakePtr = takePtr;

	int buflen = buf.length;
	/* # of slots to end of array */
	int contigspace = arraylen - takePtr;
	
	if (len <= contigspace) {
		System.arraycopy(array, takePtr, buf, offset, len);
		}
	else {
		System.arraycopy(array, takePtr, buf, offset, contigspace);
		System.arraycopy(array, 0, buf, offset + contigspace, 
						 len - contigspace);
		}
	takePtr = (takePtr + len) % arraylen;
	usedSlots = usedSlots - len;
	if (dbg) {
		//NOTE: usedSlots can be conc. increased by producer thread(s) so subsequent
		//printouts of this line may not decrease usedslots exactly by len
		String str = "extract(): contigspace="+contigspace+";slots[used="+usedSlots+"] putPtr=["+putPtr+"];takePtr=["+oldTakePtr+"->"+takePtr+"]";
		if (dbg2) {
			str += ";extracted="+Arrays.toString(buf);
			}
		System.out.println(str);
		}
	}

/**    
Return, but do not remove the value at head of queue, or 
-1 if it is empty.
**/
public byte peek() 
	{
	synchronized(this) 
		{
		if (usedSlots > 0)
        	return array[takePtr];
		else
        	return -1;
    	}
  	}

//-- puts ----------------------------------------
		
/** 
Places the specified byte in the queue, possibly waiting
indefinitely until it can be accepted. 

@param	val		the byte to be inserted

@throws 		java.lang.InterruptedException - if the current 
thread has been interrupted at a point at which interruption is
detected, in which case the value is guaranteed not to be
inserted. Otherwise, on normal return, the value is guaranteed
to have been inserted.
**/		
public void put(byte val) throws InterruptedException 
	{
    if (Thread.interrupted()) throw new InterruptedException();

    synchronized(putMonitor) 
		{
      	while (emptySlots <= 0) {
			try { 
				putMonitor.wait(); 
				}
        	catch (InterruptedException ex) {
          		putMonitor.notify();
          		throw ex;
        		}
      		}
		insert(val);
    	}
	incUsedSlots(1);
	}


/** 
Writes len bytes from the specified byte array starting
at offset off to this queue. If space for len bytes  
are not available, this method will block indefinitely 
until such space <i>is</i> available.

@param	buf		the byte array containing values to be inserted
				(offset and len have the usual meanings)
@throws 		java.lang.InterruptedException - if the current 
                thread has been interrupted at a point at which
                interruption is detected, in which case values
                are guaranteed not to be inserted. Otherwise, on
                normal return, values are guaranteed to have
                been inserted.
**/
public void put(byte[] buf, int offset, int len) throws InterruptedException 
	{
	if (buf == null) throw new NullPointerException ();
	if (offset < 0 || len < 0 || offset + len > buf.length) 
      	throw new IndexOutOfBoundsException ();
	if (len > arraylen) {
		throw new IllegalArgumentException("Queue size=" + arraylen + "; cannot write more bytes than qsize. Attempted to write: " + len + " bytes");
		}		
	if (len == 0) { //nothing to do
		return;
		}		
	if (Thread.interrupted()) throw new InterruptedException();
 	
	synchronized(putMonitor) 
		{
      	while (emptySlots < len) {
			try { 
				if (dbg) System.out.println("put(): " + Thread.currentThread() + " waiting...[emptyslots="+emptySlots+"; len="+len+"]");
				putMonitor.wait(); 
				if (dbg) System.out.println("put(): " + Thread.currentThread() + " woken...[emptyslots="+emptySlots+"; len="+len+"]");
				}
        	catch (InterruptedException ex) {
          		putMonitor.notify();
          		throw ex;
        		}
      		}
		insert(buf, offset, len);
    	}
	incUsedSlots(len);
	}		

/** 
Place item in queue only if it can be accepted within msecs
milliseconds. The time bound is interpreted in a coarse-grained,
best-effort fashion. 

@param	val		the byte to be inserted
@param	msecs	the number of milliseconds to wait. If less
                than or equal to zero, the method does not
                perform any timed waits, but might still
                require access to a synchronization lock,
                which can impose unbounded delay if there is
                a lot of contention for the queue.
@return	true 	if accepted, else false 

@throws			java.lang.InterruptedException - if the current 
                thread has been interrupted at a point at which
                interruption is detected, in which case the
                element is guaranteed not to be inserted (i.e.,
                is equivalent to a false return).
**/
public boolean offer(byte x, long msecs) throws InterruptedException 
	{
    if (Thread.interrupted()) throw new InterruptedException();

    synchronized(putMonitor) 
		{
		long start = (msecs <= 0)? 0 : System.currentTimeMillis();
		long waitTime = msecs;
		while (emptySlots <= 0) 
			{
        	if (waitTime <= 0) {
				return false;
				}
			try { 
				putMonitor.wait(waitTime); 
				}
        	catch (InterruptedException ex) {
          		putMonitor.notify();
          		throw ex;
        		}
			waitTime = msecs - (System.currentTimeMillis() - start);
      		}
		insert(x);
    	}

    incUsedSlots(1);
    return true;
	}

/**
Places the specified bytes in queue only if all the specified
bytes can be accepted within msecs milliseconds. The time bound is
interpreted in a coarse-grained, best-effort fashion. 

@param	buf		the byte array containing values to be inserted
				(offset and len have the usual meanings)
@param	msecs	the number of milliseconds to wait. If less
                than or equal to zero, the method does not
                perform any timed waits, but might still
                require access to a synchronization lock,
                which can impose unbounded delay if there is
                a lot of contention for the queue.

@throws 		java.lang.InterruptedException - if the current 
                thread has been interrupted at a point at which
                interruption is detected, in which case values
                are guaranteed not to be inserted. Otherwise, on
                normal return, values are guaranteed to have
                been inserted.
**/
public boolean offer(byte[] buf, int offset, int len, long msecs) 
throws InterruptedException 
	{
	if (buf == null) throw new NullPointerException ();
	if (offset < 0 || len < 0 || offset + len > buf.length) 
      	throw new IndexOutOfBoundsException ();
	if (len > arraylen) {
		throw new IllegalArgumentException("Queue size=" + arraylen + "; cannot write more bytes than qsize. Attempted to write: " + len + " bytes");
		}				
	if (len == 0) { //nothing to do
		return true;
		}
	if (Thread.interrupted()) throw new InterruptedException();
		
	synchronized(putMonitor) 
		{
		long start = (msecs <= 0)? 0 : System.currentTimeMillis();
		long waitTime = msecs;
		while (emptySlots < len) 
			{
			if (waitTime <= 0) {
				return false;
				}
			try { 
				putMonitor.wait(waitTime); 
				}
        	catch (InterruptedException ex) {
          		putMonitor.notify();
          		throw ex;
        		}
			waitTime = msecs - (System.currentTimeMillis() - start);
      		}
		insert(buf, offset, len);
    	}
	incUsedSlots(len);	
	return true;		
	}		


//-- takes -----------------------------------------

/** 
Return and remove a byte from queue, possibly waiting
indefinitely until a byte can be read.

@throws 	java.lang.InterruptedException - if the current thread
            has been interrupted at a point at which interruption
            is detected, in which case state of the queue is
            unchanged.
**/
public byte take() throws InterruptedException 
	{ 
    if (Thread.interrupted()) throw new InterruptedException();
    byte val = 0; 
    
	synchronized(takeMonitor) 
		{ 
      	while (usedSlots <= 0) {
        	try { 
				takeMonitor.wait(); 
				}
        	catch (InterruptedException ex) {
          		takeMonitor.notify();
          		throw ex; 
				}
			}
		val = (byte) extract();
    	}
	incEmptySlots(1); 	
	return val;
	}


/*
we have to make at least one of put or take methods non-blocking,
meaning the ability to proceed to take/put data even if the entire
array cannot be filled/written.
 
Otherwise if neither can proceed until the entire array can
be filled, we can get a livelock:

Queue:

  0   1   2   3  4   5  6
--------------------------
| x | x | x | x |  |  |  |
-------------------------
              ^
              |
______________|           

1. put(4)
2. read(5) will be blocked (5 elements not present)
3. put (4) will be blocked (since read is blocked and q will 
never clear)

*/

/** 
Reads as many bytes as possible without blocking into the
specified array.

@param	buf	 the byte array to read the values into.
			 (offset and len have the usual meanings)
@return 	 the number of bytes read
@throws 	 java.lang.InterruptedException - if the current thread
             has been interrupted at a point at which interruption
             is detected, in which case state of the queue is
             unchanged.
**/
public int take(byte[] buf, int offset, int len) 
throws InterruptedException 
	{
	if (buf == null) throw new NullPointerException ();
	if (offset < 0 || len < 0 || offset + len > buf.length) 
      	throw new IndexOutOfBoundsException ();
	if (len == 0) { //nothing to do
		return 0;
		}		
	if (Thread.interrupted()) throw new InterruptedException();
			
    synchronized(takeMonitor) 
		{
      	while (usedSlots <= 0) {
			try { 
				if (dbg) System.out.println("take(): " + Thread.currentThread() + " waiting...[usedslots="+usedSlots+"]");
				takeMonitor.wait();
				if (dbg) System.out.println("take(): " + Thread.currentThread() + " woken...[usedslots="+usedSlots+"]");
				}
        	catch (InterruptedException ex) {
          		takeMonitor.notify();
          		throw ex;
        		}
      		}
		//livelock prevention
		len = (len < usedSlots) ? len : usedSlots;
		
		extract(buf, offset, len);
    	}
	incEmptySlots(len);	
	return len;
	}		


/** 
Return and remove a byte from the queue. The time bound is 
interpreted in a coarse grained, best-effort fashion.

@return		the value (as unsigned integer) or -1 if the queue 
			is empty and no value could be read in the specified 
			time. 
@throws 	java.lang.InterruptedException - if the current thread
            has been interrupted at a point at which interruption
            is detected, in which case state of the queue is
            unchanged (i.e., equivalent to a -1 return).
**/
public int poll(long msecs) throws InterruptedException 
	{ 
    if (Thread.interrupted()) throw new InterruptedException();
    
	int val = 0;
	synchronized(takeMonitor) 
		{ 
		long start = (msecs <= 0) ? 0 : System.currentTimeMillis();
      	long waitTime = msecs;
      	while (usedSlots <= 0) 
			{
        	if (waitTime <= 0) {
				//return -1 since we could not read any value
				//at all
				return -1;
				}
			try { 
				takeMonitor.wait(waitTime); 
				}
        	catch (InterruptedException ex) {
          		notify();
          		throw ex; 
        		}
        	waitTime = msecs - (System.currentTimeMillis() - start);
			}
		val = extract();
    	}
	incEmptySlots(1);    
	return val;
	}


/** 
Tries to read the specified number of bytes into the array.
Does <u>not</u> read a partial number of bytes. If the specified
number of bytes cannot be read, returns <tt>false</tt>.

@param	buf	 the byte array to read the values into.
			 (offset and len have the usual meanings)
@return 	 <tt>true</tt> on success, <tt>false</tt> otherwise
@throws 	 java.lang.InterruptedException - if the current thread
             has been interrupted at a point at which interruption
             is detected, in which case state of the queue is
             unchanged (i.e., equivalent to a <tt>false</tt> return).
**/
public boolean poll(byte[] buf, int offset, int len, long msecs) 
throws InterruptedException 
	{
	if (buf == null) throw new NullPointerException ();
	if (offset < 0 || len < 0 || offset + len > buf.length) 
      	throw new IndexOutOfBoundsException ();
	if (len == 0) { //nothing to do
		return true;
		}
    if (Thread.interrupted()) throw new InterruptedException();
    
	int val = 0;
	synchronized(takeMonitor) 
		{ 
		long start = (msecs <= 0)? 0 : System.currentTimeMillis();
      	long waitTime = msecs;
      	while (usedSlots < len) 
			{
        	if (waitTime <= 0) {
				//return false  since we could not read any value
				//at all
				return false;
				}
			try { 
				takeMonitor.wait(waitTime); 
				}
        	catch (InterruptedException ex) {
          		notify();
          		throw ex; 
        		}
        	waitTime = msecs - (System.currentTimeMillis() - start);
			}
		extract(buf, offset, len);
    	}
	incEmptySlots(len);    
	return true;
	}


//-- I/O streams ----------------------------

/**
Returns an input stream that can read from this queue.
<p>
Note: The input stream's available will always return the number
of bytes available <u>at the time</u> this method was invoked.
This should not be relied upon as <b>not</b> accurate since
availability <u>cannot be guaranteed in the face of multiple
read/write threads</u>
**/
public InputStream getInputStream() {
	return new InputStreamView();
	}

/** 
Returns an OutputStream that writes into this queue. 
<p>
Note: the various <tt>write</tt> methods provided by this class
will block until the specified byte(s) are can be and are fully
written.
**/
public OutputStream getOutputStream() {
	return new OutputStreamView();
	}


class OutputStreamView extends OutputStream
{		
boolean closed;

public void close() { 
	closed = true;
	}
	
public void flush() { }

public void write(int b) throws IOException 
	{
	checkClosed();
	try {
		put((byte)b);
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted: " + e.toString());
		}
	}

public void write(byte[] buf) throws IOException
	{
	write(buf, 0, buf.length);
	}

public void write(byte[] buf, int off, int len) throws IOException
	{		
	checkClosed();
	try {
		put(buf, off, len);
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted or timed-out: " + e.toString());
		}
	}

void checkClosed() throws IOException {
	if (closed) {
		throw new IOException("Stream has been closed");
		}
	}

} //~OutputStream


class InputStreamView extends InputStream 
{
boolean closed;

public int available() {
	return BoundedByteQueue.this.available();
	}

public void close() {
	closed = true;
	}
	
public int read() throws IOException
	{
	try {
		return take();	
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted: " + e.toString());
		}
	}

public int read(byte[] buf) throws IOException 
	{
	return read(buf, 0, buf.length);
	}

public int read(byte[] buf, int off, int len) 
throws IOException
	{
	checkClosed();
	try {
		return take(buf, off, len);
		}
	catch (InterruptedException e) {
		throw new IOException("Interrupted: " + e.toString());
		}
	}

void checkClosed() throws IOException {
	if (closed) {
		throw new IOException("Stream has been closed");
		}
	}

} //~InputStream


//-- Unit Testing ----------------------------------------

/** Unit Test. Usage: java fc.util.BoundedByteQueue -help **/
public static void main(String[] args) throws InterruptedException
	{
	Args myargs = new Args(args);
	myargs.setUsage("java fc.util.BoundedByteQueue " +
		"\n -help  display these options" +
		"\n -producerSleep  time  sleep millisecs (speed) for producer thread (default: no sleep)" +
		"\n -consumerSleep  time  sleep millisecs (speed) for consumer thread (default: no sleep)" +
		"\n -producerBlock  time  producer blocks only for millisecs (default: blocks forever)" +
		"\n -consumerBlock  time  consumer blocks only for millisecs (default: blocks forever)" +
		"\n -qsize          size  optional queue size (default: " + DEFAULT_CAPACITY + ")" +
		"\n -producerNum    num   the number of producers (default: 1)" +
		"\n -consumerNum    num   the number of consumers (default: 1)" + 
		"\n -consumerChunk  size  use a byte array of the specified size to consume" +
		"\n -producerChunk  size  use a byte array of the specified size for producing" +
		"\n -useStream            use streams to read/write to the queue" +
		"\n -verbose              to print more detailed information"
		);

	if (myargs.flagExists("help"))
		myargs.showError();
	
	boolean useStream = myargs.flagExists("useStream");
	boolean verbose = myargs.flagExists("verbose");
		
	int producerSleepTime = Integer.parseInt(myargs.get("producerSleep", "-1"));
	int consumerSleepTime = Integer.parseInt(myargs.get("consumerSleep", "-1"));
	
	boolean producerBlockForever = true;
	boolean consumerBlockForever = true;
	
	int producerBlockTime = -1;
	int consumerBlockTime = -1;
	
	if (myargs.flagExists("producerBlock")) {
		producerBlockForever = false;
		producerBlockTime = Integer.parseInt(myargs.getRequired("producerBlock"));	
		}

	if (myargs.flagExists("consumerBlock")) {
		consumerBlockForever = false;
		consumerBlockTime = Integer.parseInt(myargs.getRequired("consumerBlock"));	
		}

	BoundedByteQueue q = null;
	
	if (myargs.flagExists("qsize"))	
		q = new BoundedByteQueue(Integer.parseInt(myargs.getRequired("qsize")));
	else
		q = new BoundedByteQueue();

	int producerChunk = Integer.parseInt(myargs.get("producerChunk", "-1"));
	int consumerChunk = Integer.parseInt(myargs.get("consumerChunk", "-1"));

	int prodNum = Integer.parseInt(myargs.get("producerNum", "1"));
	int conNum = Integer.parseInt(myargs.get("consumerNum", "1"));
	
	//waits for all threads to be constructed before they are
	//started
	Latch ready = new Latch();
	
	if (! useStream)
		{
		System.out.println("Creating " + prodNum + " producer(s)..");	
		for (int n = 0; n < prodNum; n++) 
		  new ProducerThread(
		  	verbose, ready, n+1, q, 
		  	producerSleepTime, producerBlockTime, 
			producerBlockForever, producerChunk);
		
		System.out.println("Creating " + conNum + " consumer(s)..");
		for (int n = 0; n < conNum; n++) 
		  new ConsumerThread(
			verbose, ready, n+1, q, 
			consumerSleepTime, consumerBlockTime, 
			consumerBlockForever, consumerChunk);
		}	
	else
		{
		System.out.println("Creating " + prodNum + " producer(s)..");	
		for (int n = 0; n < prodNum; n++) 
		  new ProducerThreadStream(verbose, ready, n+1, q.getOutputStream(), 
		  	producerSleepTime, producerChunk);
		
		System.out.println("Creating " + conNum + " consumer(s)..");
		for (int n = 0; n < conNum; n++) 
		  new ConsumerThreadStream(verbose, ready, n+1, q.getInputStream(), 
			consumerSleepTime, consumerChunk);
		}

	ready.release(); //start them all
	}


private static class ProducerThread extends Thread 
	{
	boolean verbose;
	static volatile int totalWritten = 0;
	String id;
	int waitTime;
	int sleepTime;
	BoundedByteQueue q;
	boolean block;
	byte[] buf;
	Latch ready;
		
	public ProducerThread(boolean verbose, Latch ready, int id, BoundedByteQueue q, int sleepTime, 
		int waitTime, boolean block, int bytesize) 
		{
		super("Producer#"+id);
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.q = q;
		this.sleepTime = sleepTime;
		this.waitTime = waitTime;		
		this.block = block;
		if (bytesize >= 0)
			this.buf = new byte[bytesize];
		start();
		}
			
	public void run() 
		{
		System.out.println("Producer thread# " + id + " start, sleep=" + 
		 sleepTime + " ms " +
		 (! block ? "will wait only for: " + waitTime + "ms" : " "));
		
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
		
		while (true) 
			{
			byte b = 0;
			
			if (buf == null)
			 	b = testx.nextbyte();
			else
				testx.fillbuf(buf);
				
			boolean suxez = true;
			try {
				if (block) 
					{
					if (buf == null) 
						{
						q.put(b);
						totalWritten++;
						if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced byte: " +  b);
						}
					else 
						{
						q.put(buf, 0, buf.length);
						totalWritten += buf.length;
						if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced bytearray: " + Arrays.toString(buf));
						}
					}
				else //non blocking
					{
					if (buf == null) 
						{
						suxez = q.offer(b, waitTime);
						if (! suxez)
							if (verbose) System.out.println("Producer#" + id + " timeout, could not put: " + b);
						else
							totalWritten++;
						}
					else 
						{
						suxez = q.offer(buf, 0, buf.length, waitTime);
						if (! suxez)
							if (verbose) System.out.println("Producer#" + id + " timeout, could not put: " + Arrays.toString(buf));					
						else
							totalWritten += buf.length;
						}
					} //non-blocking
				}
			catch (Exception e) {
				e.printStackTrace();
				}
				
			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}
			testx.showStatus(totalWritten);
			}		
		} //~run
		
	} //~class ProducerThread


private static class ConsumerThread extends Thread 
	{
	static volatile int totalRead = 0;
	boolean verbose; 
	String id;
	int waitTime;
	int sleepTime;
	BoundedByteQueue q;
	boolean block;
	byte[] buf;
	Latch ready;

	public ConsumerThread(boolean verbose, Latch ready, int id, BoundedByteQueue q, int sleepTime, 
		int waitTime, boolean block, int bytesize) 
		{
		super("Consumer#"+id);
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.q = q;
		this.sleepTime = sleepTime;
		this.waitTime = waitTime;		
		this.block = block;
		if (bytesize >= 0)
			this.buf = new byte[bytesize];
		start();
		}
	
	public void run() 
		{
		System.out.println("Consumer thread#" + id + " start, sleep=" + 
		 sleepTime + " ms " +  
		 (! block ? "will wait only for: " + waitTime + " ms" : " "));
	
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
		
		while (true) 
			{
			try {
				if (block) 
					{
					if (buf == null) 
						{
						byte b = q.take();
						totalRead++;
						if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read byte: " + b);
						}
					else 
						{
						int read = q.take(buf, 0, buf.length);	
						totalRead += read;
						if (verbose) System.out.println("Consumer#" + id + " [total=" + totalRead + "]: read buffer: " + Arrays.toString(buf));
						}
					}
				else { //non-block
					if (buf == null) 
						{
						int b = q.poll(waitTime);
						if (b < 0)
							if (verbose) System.out.println("Consumer#" + id + " timed out...");
						else {
							totalRead++;
							if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read byte: " + b);
							}
						}
					else 
						{
						boolean suxez = q.poll(buf, 0, buf.length, waitTime);	
						if (suxez)
							if (verbose) System.out.println("Consumer#" + id + " timed out...");
						else {
							totalRead += buf.length;
							if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read byte array: " + Arrays.toString(buf));
							}
						}
					} //~non-block
				}
			catch (Exception e) {
				e.printStackTrace();
				}

			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}

			}		
		} //~run

	} //~class ConsumerThread


//-- stream versions -------------------------------------------

private static class ProducerThreadStream extends Thread 
	{
	boolean verbose;
	static volatile int totalWritten = 0;
	String id;
	int sleepTime;
	OutputStream out;
	byte[] buf;
	Latch ready;
		
	public ProducerThreadStream(boolean verbose, Latch ready, int id, OutputStream out, 
		int sleepTime, int bytesize) 
		{
		super("Producer#"+id);		
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.out = out;
		this.sleepTime = sleepTime;
		if (bytesize >= 0)
			this.buf = new byte[bytesize];
		System.out.println("Producer thread# " + id + " start, sleep=" + 
		 sleepTime + "ms ");
		start();
		}
			
	public void run() 
		{
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
	
		while (true) 
			{
			byte b = 0;
			
			if (buf == null)
			 	b = testx.nextbyte();
			else
				testx.fillbuf(buf);
				
			boolean suxez = true;
			try {
                if (buf == null) 
                	{
                	out.write(b);
               	 	totalWritten++;
                	if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced byte: " +  b);
                	}
                else 
                	{
                	out.write(buf, 0, buf.length);
                	totalWritten += buf.length;
                	if (verbose) System.out.println("Producer#" + id + " [total=" + totalWritten + "] produced bytearray: " + Arrays.toString(buf));
                	}
				}
			catch (Exception e) {
				e.printStackTrace();
				}
				
			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}
			testx.showStatus(totalWritten);	
			} //~while		
		} //~run

	} //~class ProducerThreadStream

private static class ConsumerThreadStream extends Thread 
	{
	boolean verbose;
	static volatile int totalRead = 0;
	String id;
	int sleepTime;
	InputStream in;
	byte[] buf;
	Latch ready;

	public ConsumerThreadStream(boolean verbose, Latch ready, int id, InputStream in, 
		int sleepTime, int bytesize) 
		{
		super("Consumer#"+id);
		this.verbose = verbose;
		this.ready = ready;
		this.id = String.valueOf(id);
		this.in = in;
		this.sleepTime = sleepTime;
		if (bytesize >= 0)
			this.buf = new byte[bytesize];
		System.out.println("Consumer thread#" + id + 
						" start, sleep=" + sleepTime + " ms " );		
		start();
		}
	
	public void run() 
		{
		try {
			ready.acquire();
			}
		catch (InterruptedException e) {
			e.printStackTrace();
			}
		
		while (true) 
			{
			try {
				if (buf == null) 
					{
					int b = in.read();
					if (b >= 0) {
						totalRead++;
						if (verbose) System.out.println("Consumer#" + id + " [total="+ totalRead + "]: read byte: " + b);
						}
					else
						if (verbose) System.out.println("Consumer#" + id + " could not read byte");
					}
				else 
					{
					int read = in.read(buf, 0, buf.length);	
					totalRead += read;
					if (verbose) System.out.println("Consumer#" + id + " [total=" + totalRead + "]: read buffer: " + Arrays.toString(buf));
					}
				}
			catch (Exception e) {
				e.printStackTrace();
				}

			if (sleepTime >= 0)
				{	
				try {
					sleep(sleepTime);
					}
				catch (Exception e) {
					e.printStackTrace();
					}
				}
			}		

		} //~run

	} //~class ConsumerThreadStream


//-- support class for test threads -------

private static class testx
	{
	static byte fillstart = 1;
	
	static void fillbuf(byte[] buf) 
		{
		if (fillstart > 99) 
			fillstart = 1;
		for (int n = 0; n < buf.length; n++)
			buf[n] = fillstart++;
		}
	
	static byte nextbyte() {
		if (fillstart > 99) fillstart = 1;
		return fillstart++;
		}

	static long prod_oldtime;
	static final int ShowProducedAmount = 100000;
	static int triggerLevel = ShowProducedAmount;
	static boolean showMsg = false;
	
	static void showStatus(int totalWritten) 
		{	
		//Depending on the producer chunk size, we may never
		//write 100,000 bytes exactly. so we cannot use 
		//totalWritten % 100,000 for example
	
		if (totalWritten > triggerLevel) {
			showMsg = true;	
			triggerLevel = triggerLevel + ShowProducedAmount ;
			}
		
		if (showMsg) {
			showMsg = false;
			double temp = prod_oldtime;
			prod_oldtime = System.currentTimeMillis();
			System.out.println("=== 100,000 more items produced (total=" + totalWritten + ") in: " + (prod_oldtime - temp)/1000 + " seconds =====");
			}
		}

	}		

} //~class BoundedByteQueue


