// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc.dbo;

import java.sql.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

import org.postgresql.util.*;

/**
Represents the information for a table in a database.

@author hursh jain
*/
class Table
{
static Generate				generate;
static DatabaseMetaData 	md;
static SystemLog			log;
static DBspecific 			dbspecific;	

private static boolean is_postgres; 
private static boolean is_mysql;

Connection	con;
String 		tablename;
String		tabletype;
String  	remarks;

//all the columns in the current table being processed
//key = colname and value = coldata
Map 		columnMap	= new HashMap();  

//yeah, we could use a treemap which would keep things
//sorted but fark it, this is theoretically faster, memory
//permitting
List		columnList 		 	= new ArrayList();

//columns defined as Primary keys, could also be FK's
List		pkList			 	= new ArrayList();

//columns defined as Foreign keys, could also be PK's
List		fkList			 	= new ArrayList();

static final ColumnComparator column_comparator = new ColumnComparator();

/**
constructs a new table. tablename, type and remarks are
the only useful properties that work across jdbc drivers.

@param	catalog	 		the catalog paramter to getColumns()
@param	schemapattern 	the schemaPattern to getColumns()
@param	tablename		obtained via getTables -> "TABLE_NAME"
@param	tabletype		obtained via getTables -> "TABLE_TYPE"
@param	remarks			obtained via getTables -> "REMARKS"
*/
Table(	Connection con /*added solely for passthru to dbmysql.java*/,
		String catalogname, String schemapattern, 
		String tablename, String tabletype, String remarks) 
throws SQLException
	{
	this.con = con;
	this.tablename = tablename;	
	this.tabletype = tabletype;
	this.remarks = remarks;
	readTable(catalogname, schemapattern);
	}
	
/**
Must be called once - before instantiating any object of this class
*/
static void init(Generate generate, DatabaseMetaData md, SystemLog log, PropertyMgr props, DBspecific dbspecific)
	{
	Table.generate = generate;
	Table.md = md;
	Table.dbspecific = dbspecific;
	Table.log = log;
	ColumnData.init(log, props);
	}

//Read table, column etc data. We do this as a separate step
//because we need parts of same data in multiple places while 
//writing out the class. For example, we need column length
//when writing validating functions and column type when 
//writing field definitions.
private void readTable(String catalogname, String schemaname) 
throws SQLException
	{
	String columnpattern = "%"; //all columns
	ResultSet rs = md.getColumns(catalogname, schemaname, tablename, columnpattern);
	QueryUtil.ensureScrollable(rs);
	rs.beforeFirst();
	while (rs.next()) 
		{
		String colname 		= rs.getString	("COLUMN_NAME");
		int datatype 		= rs.getInt		("DATA_TYPE");
		String typename 	= rs.getString	("TYPE_NAME");
		int	colsize 		= rs.getInt		("COLUMN_SIZE");	
		int nullable 		= rs.getInt		("NULLABLE");
		boolean isautoinc 	= dbspecific.isAutoIncrement(con, tablename, colname, rs);
		String remarks 		= rs.getString	("REMARKS");
		int	colnum			= rs.getInt		("ORDINAL_POSITION");
		//a default value for this column such as now(), true, etc.
		//this will be null if there is no default value;
		String default_val	= rs.getString  ("COLUMN_DEF");

		if (MiscUtil.isJavaKeyword(colname)) {
			throw new SQLException("The colname [" + colname + "] for table [" + tablename + "] is a Java Reserved Keyword. Either change the column name in the DB or exclude this table in the config file");	
			}
		
		//pg driver can return enum as varchar, which is wrong. because when sending it back to the db
		//as string/varchar, the db throws an exception. it should be OTHER!
		//of course, the fix is to convert this into an Object and not a String but that's too much work right now
		if (typename.toLowerCase().indexOf("enum") >= 0 
			&& datatype != java.sql.Types.OTHER
			&& generate.isPostgres()) 
			{
			log.warn("postgres 'enum' type found for column (", colname, ") - the driver thinks this is a String");
			log.warn("and will then cause a Postgres error when saving it back to the DB. To get around this:");
			log.warn("specify 'stringtype=unspecified' in the JDBC url!");
			}
		
		ColumnData cd = new ColumnData(this,
			colname, colnum,  datatype, typename, colsize, nullable, isautoinc, remarks, default_val
			);
		
		//log.bug("created new column data", cd);
		columnMap.put(cd.getName(), cd);
		columnList.add(cd);
		}
		
	Collections.sort(columnList, column_comparator);

	rs = md.getPrimaryKeys(catalogname, null, tablename);
	while (rs.next())
		{
		String colname = rs.getString("COLUMN_NAME");
		ColumnData cd = (ColumnData) columnMap.get(colname);
		if (cd == null) {
			throw new IllegalStateException("Cannot locate PK column. This should not happen since DatabaseMetaData.getPrimaryKeys() should be a subset of DatabaseMetaData.getColumns()");
			}
		cd.setPK(true);
		pkList.add(cd);
		}

	Collections.sort(pkList, column_comparator);

	rs = md.getImportedKeys(catalogname, null, tablename);
	while (rs.next())
		{
		//the referenced primary table
		String pk_table = rs.getString("PKTABLE_NAME");
		//the referenced primary table column
		String pk_colname = rs.getString("PKCOLUMN_NAME");
		//the fk column (may be a different name than what
		//it's called in the pk table
		String fk_name = rs.getString("FKCOLUMN_NAME");
		//System.out.println("FK: " + pk_table + ";" + pk_colname + ";" + fk_name);
		ColumnData cd = (ColumnData) columnMap.get(fk_name);
		if (cd == null) {
			throw new IllegalStateException("Cannot locate FK column. This should not happen since DatabaseMetaData.getImportedKeys() should be a subset of DatabaseMetaData.getColumns()");
			}
		cd.setFK(fk_name, pk_table, pk_colname);
		fkList.add(cd);	
		}
		
	//System.out.println(">>>>>>>>>>fklist> "+fkList);
	Collections.sort(fkList, column_comparator);

	log.bug("Columns=",columnList);
	log.bug("PK List=",pkList);
	log.bug("FK List=",fkList);
	}

String getName() {
	return tablename;
	}

/**
returns the TABLE_TYPE value for this column
*/
String getType() {
	return tabletype;
	}

String getRemarks() {
	return remarks;
	}

/**
Returns a list of all {@link ColumnData} objects, sorted by column order
*/
List getColumnList() {
	return columnList;
	}

/**
returns the number of columns in this table
*/
int getColumnCount() {
	return columnList.size();
	}

/**
returns the ColumnData with the specified colname or null
if such a column does not exist.
*/
ColumnData getColumn(String colname) {
	return (ColumnData) columnMap.get(colname);
	}
	
/**
returns the foreign key list sorted by column order. Returns
<tt>null</tt> if there are no fk's defined for this table.
*/
List getFKList() {
	return fkList;
	}

/**
returns the primary key list sorted by column order. Returns
<tt>null</tt> if there are no pk's defined for this table.
*/
List getPKList() {
	return pkList;
	}
	
public String toString() {
	return tablename;	
	}
	
private static class ColumnComparator implements Comparator
	{
	//we use default reference equality in columndata
	//since columns ordinals positions are always 
	//different, this compare is consistent with equals
	public int compare(Object a, Object b)
		{
		//what happens when there is only 1 item in a list
		//does the comparator get called when sorting that
		//list ? not clear, but was seeing some weird behavior
		//so this check will solve this case (and cannot hurt
		//in any way)
		if (a == b) {
			return 0;
			}
			
		int a1 = ((ColumnData) a).getNumber();
		int b1 = ((ColumnData) b).getNumber();
		
		if ( a1 < b1 ) return -1;
		else if (a1 == b1) {
			log.error("Column numbers ('ORDINAL_POSITION') of 2 columns are identical: ", a, b);
			log.error("Maybe your JDBC driver is buggy. Fix this problem, else the generated code for this table will be BUGGY");
			System.exit(1);
			return 0;
			}
		else return 1;
		}
	
	public boolean equals(Object obj) {
		return super.equals(obj);
		}
	}

/**
Returns a comma delimited string consisting of all column
names. Examples: 
<blockquote>
<br>	"col_a, col_b, col_c" [table with 3 columns]
<br>	"col_a"				  [table with 1 column]
</blockquote>
Returns an empty string if the specified list has no
elements.

@param	columns			a List containing ColumnData objects
*/
static String getListAsString(List columns) 
	{
	int size = columns.size();	
	if (size == 0)
		return "";
	//rough heuristic
	StringBuffer buf = new StringBuffer(size * 10);	
	ColumnData cd = (ColumnData) columns.get(0);
	buf.append(cd.getName());

	if (size > 1) {
		for (int n = 1; n < size; n++) {
			buf.append(", ");
			cd = (ColumnData) columns.get(n);
			buf.append(cd.getName());
			}
		}
	return buf.toString();
	}

String getFullyQualifiedColumnString() 
	{
	int size = columnList.size();	
	if (size == 0)
		return "";

	//rough heuristic
	StringBuffer buf = new StringBuffer(size * 10);
	ColumnData cd = (ColumnData) columnList.get(0);
	buf.append(tablename);
	buf.append(".");
	buf.append(cd.getName());
	buf.append(" as ");
	buf.append(tablename);
	buf.append("_");
	buf.append(cd.getName());

	if (size > 1) {
		for (int n = 1; n < size; n++) {
			buf.append(", ");
			cd = (ColumnData) columnList.get(n);
			buf.append(tablename);
			buf.append(".");
			buf.append(cd.getName());
			buf.append(" as ");
			buf.append(tablename);
			buf.append("_");
			buf.append(cd.getName());
			}
		}
	buf.append(" ");
	return buf.toString();
	}

//select table.column from table
//select t.column from table AS t  [in this case have to use t.column and prefix == t]
//returns runtime code, not a static string (as inthe case of the prefix less version), since
//the user can use any arbitrary prefix at runtime
String getPrefixQualifiedColumnString() 
	{
	int size = columnList.size();	
	if (size == 0)
		return "	return \"\";\n";

	//rough heuristic
	String code = "";
	
	code += "	final StringBuffer buf = new StringBuffer(" + size + " * 10);\n\n";
	ColumnData cd = (ColumnData) columnList.get(0);
	
	code +=	"	buf.append(prefix);\n";
	code +=	"	buf.append(\".\");\n";
	code +=	"	buf.append(\"" + cd.getName() + "\");\n";
	code +=	"	buf.append(" + "\" as \"" + ");\n";
	code +=	"	buf.append(prefix);\n";
	code +=	"	buf.append(" + "\"_\"" + ");\n";
	code +=	"	buf.append(\"" + cd.getName() + "\");\n";

	if (size > 1) {
		for (int n = 1; n < size; n++) {
			code += "\n";
			code += "	buf.append(" + "\", \"" + ");\n";
			cd = (ColumnData) columnList.get(n);
			code += "	buf.append(prefix);\n";
			code += "	buf.append(" + "\".\"" + ");\n";
			code += "	buf.append(\"" + cd.getName() + "\");\n";
			code +=	"	buf.append(" + "\" as \"" + ");\n";
			code += "	buf.append(prefix);\n";
			code +=	"	buf.append(" + "\"_\"" + ");\n";
			code +=	"	buf.append(\"" + cd.getName() + "\");\n";
			}
		}
	code += "	buf.append(\" \");\n\n";
	code +=	"	return buf.toString();\n";
	
	return code;
	}


/**
Returns a string consisting of '?' placeholders for all
the columns in the specified list. Examples: 
<blockquote>
<br>	"col_a = ?, col_b = ?, col_c = ?"	[table with 3 columns]
<br>	"col_a = ?"			[table with 1 column]
</blockquote>
Returns an empty string if the specified list has no
elements

@param	columns		a List containing {@link ColumnData} objects
*/
static String getPreparedStmtPlaceholders(List columns) {
	final int size = columns.size();	
	if (size == 0)
		return "";
	final StringBuffer buf = new StringBuffer(size);	
	ColumnData cd = (ColumnData) columns.get(0);
	buf.append(cd.getName());
	buf.append("=?");
	if (size > 1) {
		for (int n = 1; n < size; n++) {
			cd = (ColumnData) columns.get(n);
			buf.append(" and ");
			buf.append(cd.getName());
			buf.append("=?");
			}
		}
	return buf.toString();
	}
	
} //~Table

