// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import fc.app.*;
import java.io.*;
import java.util.*;

/** 
Logs messages to some user specified destination(s). A destination
is any arbitrary OutputStream which is associated with a category. 
A category is simply any arbitrary name of your liking.

Certain default categories are already provided by this class. Their
names are:
<ul>
<li><tt>stdout</tt> (name also stored in {@link #STDOUT})
<li><tt>stderr</tt> (name also stored in {@link #STDERR})
<li><tt>debug</tt>  (name also stored in {@link #DEBUG})
</ul>
It is the responsibility of subclasses to return a concrete OutputStream 
for each default category. Subclasses and clients of this class can
also add and use any additional category they desire.
<p>
Each category (destination) has a logging level. Methods in this class accept
messages for a specified category. Each message has a message level and the
message is printed only if the message level is equal to or lower than the
current maximum level of that catagory. 
<p>
Convenience methods for the <b>default</b> categories also provided. These
are of the form: <tt>foo(), foo<b>2</b>(), foo<b>3</b>(), etc.</tt> The
name of the method itself ("foo") denotes a specific category and is followed by a
number that denotes the log level for that method. So instead of saying:
<blockquote>
<tt>someMethod("category-<b>foo</b>", "the message", Logger.Level.<b>two</b>)</tt>
</blockquote>
one can say:
<blockquote>
<tt><b>foo2</b>("the message")</tt>
</blockquote>
Of course, for categories other than the default ones, one will have
to use the longer form.
<p>
In addition to log levels, this class also provides an overall option to 
turn off all logging messages (except debugging messages). Debugging
messages can also be turned off, but this has to be done separately
from log messages.
<p>
Messages can be transformed arbitrarily but passing them through log
filters. Examples include adding log stamps such as time, date, thread and
class information.
<p>
A logger class, then, consists of a set of filters (none, one or many) with a
set of output destinations (one or many). This class is declared abstract, it
manages core functionality (filter and destinations) but leaves the actual
configuration of various destinations to subclasses.
<p>
Messages written to an output destinations depend upon the flushing policy of that 
output destination. The {@link fc.app.App} class automatically flushes all
pending output when the application is closed. The <tt>bug/tt> methods are unique
in that they always flush the underlying stream after every invocation.
<p>
Thread Safety: This class <tt>is</tt> Threadsafe and all it's methods can
be used concurrently.

@author 	hursh jain
@version 	1.3 1/6/2002
@see 		fc.app.App
**/

public abstract class Logger implements IAppElement
{
protected static  Checker check = null;

private String		name;

/** 
Whether the log start and stop messages will be printed to each log. 
**/
protected boolean	printHeaders;  //latched true|false in constructor

/** maps a destination name to a LogDest object **/
protected Map		logDestinations;	

volatile protected boolean finished = false;

/**
<tt>true</tt> if currently debugging, <tt>false</tt> otherwise. Not necessary 
for simple debugging statements of the form:
<blockquote>
	mylog.bug("abc");
</blockquote>
because all <tt>bug</tt> methods simply return if this variable is
false. But for complex statements, the following is suggested:
<blockquote>
if (mylog.dbugon) {
	bug("abc" + some-variable + ";" + some-other-variable)
	}
</blockquote>
This avoids needless string creation/concatenation is debugging
is turned off. Do not <i>set</i> this variable directly but use the 
{@link startDebug(boolean)} method.
**/
volatile public boolean	dbgon; 

/**
<tt>true</tt> if currently logging, <tt>false</tt> otherwise. Affects all 
logging except the debug log. Do not <i>set</i> this variable directly but 
use the {@link startLog(boolean)} method.
**/
volatile public boolean	logon;

/**
The category name for the default stdout log. 
**/
public final static String STDOUT = "stdout";

/**
The category name for the default stderr log. 
**/
public final static String STDERR = "stderr";

/**
The category name for the default debug log.
**/
public final static String DEBUG = "debug";

//don't use IOUtil here, since this is called non wired from the constructor
protected static final byte[] linesepbytes = System.getProperty("line.separator").getBytes();

/**
Constructs a Logger. This constructor calls the abstract methods {@link
getStdOutStream()}, {@link getStdErrStream()} and {@link getDebugStream()}
(which are implemented in a subclass). This violates a general design
principle of never calling non-private methods from a constructor, but in 
some cases (like this one) it makes sense to do so.
<p>
All of the default categories are set to a level of Logger.Level.one
and debugging is initially turned off.
@param	name		 the name of this logger	object (any arbitrary string)
@param  printHeaders specify true to print log headers (stop/start date) to each log.
**/	
public Logger(String name, boolean printHeaders)
	{
	//cannot use Checker in the constructor (not yet wired) so we
	//create another temporary Checker (methods like addCategory called
	//from the constructor use a Checker).
	check = App.sysChecker();
	this.name = name;
	this.printHeaders = printHeaders;	
	logon = true;
  	dbgon = false;
	logDestinations = Collections.synchronizedMap(new HashMap());
	addCategory(STDOUT, Level.one, getStdOutStream());
	addCategory(STDERR, Level.one, getStdErrStream());
	addCategory(DEBUG, Level.one, getDebugStream());
	}

/**
If subclasses overrride this method, they should also make sure to first
call this method via <tt>super</tt>
**/
public void init(IApplication app) {
	check = app.checker();
	}
	
/**
This template method should return a OutputStream that will be associated 
with the default <tt>"stdout"</tt> category of this class.
<p>
<b>Important</b> : the implementation of these method in a subclass
should not rely on any instance variable in that subclass. This
method will be called from this superclass at the time of superclass
construction, and at that time, no subclass variable will be available
**/
abstract protected OutputStream getStdOutStream();

/**
This template method should return a OutputStream that will be associated 
with the default <tt>"stderr"</tt> category of this class.
<p>
<b>Important</b> : the implementation of these method in a subclass
should not rely on any instance variable in that subclass. This
method will be called from this superclass at the time of superclass
construction, and at that time, no subclass variable will be available
**/
abstract protected OutputStream getStdErrStream();


/**
This template method should return a OutputStream that will be associated 
with the default <tt>"debug"</tt> category of this class.
<p>
<b>Important</b> : the implementation of these method in a subclass
should not rely on any instance variable in that subclass. This
method will be called from this superclass at the time of superclass
construction, and at that time, no subclass variable will be available
**/
abstract protected OutputStream getDebugStream();

protected void checkFinished(String str) 
	{
	if (finished) {
		throw new IllegalStateException("Logger: attempt to log() the following message after Logger has been closed: " + str);
		}
	}

/**
Adds the specified logging destination and associates it with the specified
category. If the category already exists, then the destination is not added
and <tt>false</tt> is returned, otherwise true is returned.
@param	category	the name to associate with the log destination
@param	level		initial output level for the category
@param	out			the log desination
**/
public boolean addCategory(String category, Logger.Level level, OutputStream out)
	{
	check.notnull(category);	
	LogDest dest = null;
	synchronized (logDestinations) {
		if (logDestinations.containsKey(category)) {
			return false;
			}
		dest = new LogDest(category, level,  out);	
		logDestinations.put(category, dest);
		}
	if (this.printHeaders) {
		String str = category + ": Log opened on: " + new Date();
		dest.logascii(null, null, Level.one, str);
		}
	return true;
	}	

/**
Removes the log destination associated with the specified category. Returns
true if this was successful, false if the destination with the specified
name was not found (and hence not removed).
<p><b>Note:</b>No scheduling guarantees are made. The logging thread may
end up logging older messages to the newly removed log, depending on the 
speed of it's execution. The logging thread is a separate thread that runs
in the background and does the actual logging of messages.
@param	category	the name of the category to remove
*/
public boolean removeCategory(String category) 
	{ 
	check.notnull(category);
	if (isDefaultCategory(category)) return false; //cannot remove default
	Object obj = logDestinations.get(name);
	if (obj == null) { 
		return false; 
		}
	((LogDest) obj).flush();
	logDestinations.remove(obj);	
	return true;				
	}

/**
Returns <tt>true</tt> if the category name is a default category
name, i.e., <tt>stderr</tt>, <tt>stdout</tt>, <tt>debug</tt> etc.
**/
protected boolean isDefaultCategory(String name)
	{
	//assumes passed in name will not be null
	if ( name.equals("stdout")	|| name.equals("stderr") ||
		 name.equals("debug") ) 
		{
		return true;
		} 
	return false;
	}	

/**
Toggles the printing of category names before each line logged under that
category. By default, this is true.
@param	val		true to print names, false otherwise
**/
public void printCategoryNames(String category, boolean val)
	{
	Object obj = logDestinations.get(category); 
	if (obj == null) { 
		return; 
		}
	((LogDest) obj).printCategoryNames(val);
	}

/**
Toggles the printing of category names for all categories.

@param	val		true to print names, false otherwise
**/
public void printCategoryNamesForAll(boolean val)
	{
	String[] keys = null;
	synchronized (logDestinations) {
		keys = (String[]) logDestinations.keySet().toArray(new String[0]);
		}
	for (int i = 0; i < keys.length; i++) {
		LogDest d = (LogDest) logDestinations.get(keys[i]);
		d.printCategoryNames(val);
		}
	}

/**
Adds the specified Filter to the specified category's filter chain. The
specified name for the filter has to be unique string and can be used 
to manage/remove the filter later on. Always adds the specified filter,
if a filter with that name already exists, it is replaced. Returns false
if the filter could not added (maybe the specified category or filter does
not exist), true otherwise.
@param	category 	category to add the filter to	
@param	filtername	an arbitrary string for the name of the filter
@param	filter		an object implementing the ILogFilter interface
*/
public boolean addFilter(String category, String filtername, ILogFilter filter) 
	{ 
	LogDest dest = (LogDest) logDestinations.get(category);
	if (dest == null) { 
		return false; 
		}
	return dest.addFilter(filtername, filter);
	}

/**
Adds the specified filter to all categories
@param	category 	category to add the filter to	
@param	filtername	an arbitrary string for the name of the filter
@param	filter		an object implementing the ILogFilter interface
**/
public void addFilterForAll(String filtername, ILogFilter filter) 
	{ 
	synchronized (logDestinations) 
		{ 
		String[] keys = (String[]) logDestinations.keySet().toArray(new String[0]);
		for (int i = 0; i < keys.length; i++) {
			LogDest d = (LogDest) logDestinations.get(keys[i]);
			d.addFilter(filtername, filter);
			}
		}
	}

/**
Removes the filter for the specified category. Returns true if this was
successful, false if the filter could not be removed (maybe the category
or filter was not found).
@param	category category to remove the filter from
@param	filtername	the name of the filter to remove
*/
public boolean removeFilter(String category, String filtername)  
	{ 	
	LogDest dest = (LogDest) logDestinations.get(category);
	if (dest == null) { 
		return false; 
		}
	return dest.removeFilter(filtername);
	}

/**
Removes the specified filter for the all categories. 
@param	filtername	the name of the filter to remove
*/
public void removeFilterForAll(String filtername)  
	{ 	
	synchronized (logDestinations) 
		{ 
		String[] keys = (String[]) logDestinations.keySet().toArray(new String[0]);
		for (int i = 0; i < keys.length; i++) {
			LogDest d = (LogDest) logDestinations.get(keys[i]);
			d.removeFilter(filtername);
			}
		}
	}


/**
Flushes any pending logs and stops logging. After this method is called,
any attempt to use this instance of the logger (by calling it's log method)
will throw a runtime <tt>IllegalStateException</tt>. Note: the output logs
destinations are flushed (but not closed). If an output desination was, say
<tt>System.out</tt> and we closed it, then no other classes would be able
to use System.out at all.
*/
synchronized public void close()  
	{
	this.finished = true;
	printCategoryNamesForAll(true);
	String str = "Log closed on: " + new Date();
	synchronized (logDestinations) {  //lock prevents conc. mod whilst in toArray()
		String[] keys = (String[]) logDestinations.keySet().toArray(new String[0]);
		for (int i = 0; i < keys.length; i++) {
			LogDest d = (LogDest) logDestinations.get(keys[i]);
			if (printHeaders) {
				d.logascii(null, Thread.currentThread(), Level.one, str);
				}
			d.close();
			}
		} //~synch block
	}
		
/**
Logs the given string to the specified category destination.
Since this method is <b>not</b> synchronized, adding or deleting output
destinations is permissible, while this method is being executed. However,
changes any such concurrent changes will not be seen by this method until 
the next time it is executed. 
<p>
The message to be logged is run through the current set of filters 
before it is logged.
@param	category		the log category
@param	level			the message level
@param	caller			reference to the calling object, usually <tt>this</tt>. Needed
						for certain filters, pass <tt>null</tt> if calling this method
						from a static method.
@param  callingThread	reference to the calling thread. Pass <tt>null</tt> if not using
						a thread information filter.
@param	str				the string/message to be logged.
*/
public void log(String category, Logger.Level level, Object caller, 
				Thread callingThread, String str)
	{
	if (finished) {
		throw new IllegalStateException("Logger: attempt to log() the following message after Logger has been closed: " + str);
		}
	if (category == null || level == null) { 
		return; 
		}
	LogDest dest = (LogDest) logDestinations.get(category);
	if (dest == null) { 
		return; 
		}
	dest.logascii(caller, callingThread, level, str);
	}  

/**
Convenience method that logs to the <tt>stdout</tt> category at a log level
of <tt>Level.one.</tt> Delegates to the {@link #log(String, Logger.Level, Object, Thread, String)} 
method.
@param caller 	the calling object
@param thread	the calling thread
@param str		the message to be logged
**/
public void log(Object caller, Thread thread, String str)
	{
	log(STDOUT, Level.one, caller, thread, str);
	}

/**
Convenience method that logs to the <tt>stdout</tt> category at a log level
of <tt>Level.one.</tt> Delegates to the {@link #log(String, Logger.Level, Object, Thread, String)} 
method. Since the calling thread are not specified, that information
will not be logged (even if a filter is set up to do so).
@param caller 	the calling object
@param str		the message to be logged
**/
public void log(Object caller, String str)
	{
	log(STDOUT, Level.one, caller, null, str);
	}

/**
Convenience method that logs to the <tt>stdout</tt> category, at a log
level of <tt>Level.one.</tt>. Delegates to the {@link #log(String, Logger.Level, Object, Thread, String)} 
method. Since calling object and thread are not specified, that information
will not be logged (even if a filter is set up to do so).
@param str		the message to be logged
**/
public void log(String str)
	{
	log(STDOUT, Level.one, null, null, str);
	}
	
/**
Convenience method that logs to the specified category, at a log
level of <tt>Level.one.</tt>. 
@param category	the destination category
@param str		the message to be logged
**/
public void log(String category, String str)
	{
	log(category, Level.one, null, null, str);
	}
		
/**
Convenience method that logs to the specified at a log level of 
<tt>Level.one.</tt>. 
@param category	the destination category
@param caller 	the calling object
@param thread	the calling thread
@param str		the message to be logged
**/
public void log(String category, Object caller, Thread thread, String str)
	{
	log(category, Level.one, caller, thread, str);
	}

/**
Flushes all categories
**/
public void flushAll()
	{
	String[] keys = null;
	synchronized (logDestinations) {
		keys = (String[]) logDestinations.keySet().toArray(new String[0]);
		}
	for (int i = 0; i < keys.length; i++) {
		LogDest d = (LogDest) logDestinations.get(keys[i]);
		d.flush();
		}
	}

/**
Sets all current categories to the specified logging level.
@param level	 the new level
**/
public void setLevelForAll(Logger.Level level)
	{
	String[] keys = null;
	synchronized (logDestinations) {
		keys = (String[]) logDestinations.keySet().toArray(new String[0]);
		}
	for (int i = 0; i < keys.length; i++) {
		LogDest d = (LogDest) logDestinations.get(keys[i]);
		d.setLevel(level);
		}
	}

/**
Sets the logging level for the specified category. A message in that category
will only be logged if the log level for that particular message is less than
or equal to the logging level for the category. The level of the default
categories can also be changed by using this method.
<p>
A log level of {@link Logger.Level.zero} means all messages will be logged. 
@param category		the category name
@param level		the new level
**/
public void setLevel(String category, Logger.Level level)
	{
	Object obj = logDestinations.get(category); 
	if (obj == null) { 
		return; 
		}
	((LogDest) obj).setLevel(level);
	}


/**
Turns logging on or off. If logging is turned off, no messages are printed
regardless of message level. This method affects all logging categories
<b>except</b> debugging, for which {@link willDebug(booelan)} should be used.
**/
public void startLog(boolean val)  {
	this.logon = val;
	}

/**
Turns debugging on or off. If debugging is turned off, no debug messages 
are printed regardless of message level.
**/
public void startDebug(boolean val)  {
	this.dbgon = val;
	}


/**
Returns true if debugging is currently turned on, false otherwise.
**/
public boolean isDebugging() {
	return dbgon;
	}

/**
Returns true if debugging is currently turned on, false otherwise.
**/
public boolean bugon() {
	return dbgon;
	}


//:::::::::: Debug Methods ::::::::::::::::

final static String me = "():enter";
final static String mx = "():exit";

/** 
Records method enter to the {@link #DEBUG} category at debug level 3. 
**/
public void bugme(String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.three, null, Thread.currentThread(), str + me);
	}

/** 
Records method enter to the {@link #DEBUG} category at debug level 3. 
**/
public void bugme(Object caller, String str)  {
	if ( ! dbgon ) return;
	log(DEBUG, Level.three, caller, Thread.currentThread(), str + me);
	}
	
/** 
Records method exit to the {@link #DEBUG} category at debug level 3. 
**/
public void bugmx(String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.three, null, Thread.currentThread(), str + mx);
	}

/** 
Records method exit to the {@link #DEBUG} category at debug level 3. 
**/
public void bugmx(Object caller, String str)  {
	if ( ! dbgon ) return;
	log(DEBUG, Level.three, caller, Thread.currentThread(), str + mx);
	}
	
/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 1 (the "1" does not have to be specified in the method name).
**/
public void bug(Object caller, Thread thread, String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.one, caller, thread, str);
	}

/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 1 (the "1" does not have to be specified in the method name).
**/
public void bug(Object caller, String str)  {
	if ( ! dbgon ) return;
	log(DEBUG, Level.one, caller, null, str);
	}
	
/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 1 (the "1" does not have to be specified in the method name).
**/
public void bug(String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.one, null, null, str);
	}

/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 2
*/
public void bug2(Object caller, Thread thread, String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.two, caller, thread, str);
	}
	
/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 2
*/
public void bug2(Object caller, String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.two, caller, null, str);
	}

/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 2
*/
public void bug2(String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.two, null, null, str);
	}

/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 3
*/
public void bug3(Object caller, Thread thread, String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.three, caller, thread, str);
	}
	
/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 3
*/
public void bug3(Object caller, String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.three, caller, null, str);
	}

/**
Convenience method to write a string to the {@link #DEBUG} category
at debug level 3
*/
public void bug3(String str) {
	if ( ! dbgon ) return;
	log(DEBUG, Level.three, null, null, str);
	}

//:::::::: Standard out Methods :::::::::::::

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 1 (the "1" does not have to be specified)
*/
public void stdout(Object caller, Thread thread, String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.one, caller, thread, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 1 (the "1" does not have to be specified)
*/
public void stdout(Object caller, String str)  {
	if ( ! logon ) return;
	log(STDOUT, Level.one, caller, null, str);
	}
	
/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 1 (the "1" does not have to be specified)
*/
public void stdout(String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.one, null, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 2
*/
public void stdout2(Object caller, Thread thread, String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.two, caller, thread, str);
	}
	
/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 2
*/
public void stdout2(Object caller, String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.two, caller, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 2
*/
public void stdout2(String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.two, null, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 3
*/
public void stdout3(Object caller, Thread thread, String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.three, caller, thread, str);
	}
	
/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 3
*/
public void stdout3(Object caller, String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.three, caller, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 3
*/
public void stdout3(String str) {
	if ( ! logon ) return;
	log(STDOUT, Level.three, null, null, str);
	}

//:::::::: Standard error methods ::::::::::::

/**
Convenience method to write a string to the {@link #STDERR} category
at log level 1 (the "1" does not have to be specified)
*/
public void stderr(Object caller, Thread thread, String str) {
	if ( ! logon ) return;
	log(STDERR, Level.one, caller, thread, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 1 (the "1" does not have to be specified)
*/
public void stderr(Object caller, String str)  {
	if ( ! logon ) return;
	log(STDERR, Level.one, caller, null, str);
	}
	
/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 1 (the "1" does not have to be specified)
*/
public void stderr(String str) {
	if ( ! logon ) return;
	log(STDERR, Level.one, null, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 2
*/
public void stderr2(Object caller, Thread thread, String str) {
	if ( ! logon ) return;
	log(STDERR, Level.two, caller, thread, str);
	}
	
/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 2
*/
public void stderr2(Object caller, String str) {
	if ( ! logon ) return;
	log(STDERR, Level.two, caller, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 2
*/
public void stderr2(String str) {
	if ( ! logon ) return;
	log(STDERR, Level.two, null, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 3
*/
public void stderr3(Object caller, Thread thread, String str) {
	if ( ! logon ) return;
	log(STDERR, Level.three, caller, thread, str);
	}
	
/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 3
*/
public void stderr3(Object caller, String str) {
	if ( ! logon ) return;
	log(STDERR, Level.three, caller, null, str);
	}

/**
Convenience method to write a string to the {@link #STDOUT} category
at log level 3
*/
public void stderr3(String str) {
	if ( ! logon ) return;
	log(STDERR, Level.three, null, null, str);
	}

public String toString() 
	{
	String result = getClass().getName();
	result += " [Status:" + ((finished) ? "closed" : "running") + "]";
	List copy = null;
	synchronized (logDestinations) {
		copy = new ArrayList(logDestinations.entrySet());
		}
	result += " Categories: " + copy.toString();
	return result;		
	}

/* 
Manges a log (i.e a destination). Log destinations are
not heirarchical, this is for simplicity and to avoid unnecessary 
duplication of the work java.util.log.* implemenation. 

A max level of 0 means always log, otherwise log only if specified
level is lesser/equal than current cutoff level
*/
static class LogDest
{
String name;
byte[] namebytes;
OutputStream out; 
PrintStream pout; 
volatile Logger.Level level;
volatile boolean printCatNames;
protected List	filters;	
protected Map	filterNames;	

/*
Useful for toArray(<Type>) function of various Collections
needed to be called often to take a snapshot before multicasting
(can't use an iterator because we would have to lock the entire
collections for the duration of the iterator use).
*/
protected final static ILogFilter[] iLogFilterTypeArray = {};

//don't use IOUtil here, since this is called non wired from the constructor
LogDest(String name, Logger.Level level, OutputStream outstream) 
	{
	this.name = name;
	this.namebytes = (name + ": ").getBytes();
	this.level = level;
	this.out = outstream;
	this.pout = (outstream instanceof PrintStream) ? 
					(PrintStream)outstream : new PrintStream(this.out);
	filters = Collections.synchronizedList(new LinkedList());
	filterNames = Collections.synchronizedMap(new HashMap());
	} 

public boolean addFilter(String name, ILogFilter filter) 
	{ 
	check.notnull(name);
	check.notnull(filter);
	//adding the same filter to both collections needs to be atomic
	synchronized (filterNames) {
		filters.add(filter);
		filterNames.put(name, filter);
		}
	return true; //always true
	}

public boolean removeFilter(String name)  
	{ 	
	Object filter = filterNames.get(name);
	if (filter == null) { 
		return false; 
		}
	filters.remove(filter);	
	return true;				
	}

void setLevel(Logger.Level level) {
	this.level = level;
	}

void printCategoryNames(boolean val) {
	this.printCatNames = val;
	}

//fast ascii write
void logascii(Object caller, Thread callingThread, 
			  Logger.Level msglevel, String str ) 
	{
	if (msglevel.compareTo(this.level) > 0) { 
		return;
		}
	if (str == null) { 
		return; 
		}
	//--filter pass begin
	if ( filters.size() > 0)  {
		ILogFilter[] allfilters = null;
		synchronized (filters) {  
			//static access is really slow as of hotspot jdk1.3.x
			allfilters = (ILogFilter[]) filters.toArray(iLogFilterTypeArray);
			//presumably, above line improved in 1.4
			//allfilters = (ILogFilter[]) filters.toArray(new ILogFilter[] {});
			}
		StringBuffer buf = new StringBuffer(str); 
		for (int i = 0; i < allfilters.length; i++) {
			allfilters[i].transform(caller, callingThread, buf);
			}
		str = buf.toString();
		buf = null;
		}
	//--filter pass end
	int strlen = str.length();
	byte[] byteascii = new byte[strlen];
	str.getBytes(0, strlen, byteascii, 0);
	try { 
		if (printCatNames) { 
			out.write(namebytes); 
			}
		out.write(byteascii); 
		out.write(linesepbytes);
		}
	catch (Exception e) { 
		e.printStackTrace(); 
		}
	}
	
//converts the specified string into default platform encoding,
void log(String str, Logger.Level msglevel) 
	{
	if (msglevel.compareTo(this.level) > 0) { 
		return;
		}
	//filter(); //xxx: later
	pout.print(name);
	pout.println(str);
	}
	

//underlying streams are *not* closed (example System.out), but flushed
void close() 
	{
	flush();
	}
	
void flush() 
	{
	try { 
		pout.flush();
		out.flush();
		}
	catch (Exception e) { /*ignore*/ }
	}

public String toString()
	{
	String result =  "(" + level + ";destination=" + out;
	List copy = null;
	synchronized (filters) {
		copy = new ArrayList(filters);
		}
	// Sort the filter list
    Collections.sort(copy, new Comparator() {
    	public int compare(Object obj1, Object obj2) {
    		return obj1.toString().compareTo(obj2.toString());
       		}
     	});
	result += "; filters: " + copy.toString() + ")";
	return result;
	}
	
}		//~inner class LogDest


/** 
Sets various pre defined log levels. A level of say, zero, can be accessed 
by saying <tt>Logger.Level.zero</tt>.
**/
public static class Level 
	{ 
	public static final Level one = new Level("level one(sparse), logs levels <=1 ", 1);
	public static final Level two = new Level("level two (medium), logs levels <=2 ", 2);
	public static final Level three = new Level("level three(full), logs levels <=3", 3);

	protected String desc;
	protected int level;	
	protected Level(String desc, int level) {
		this.desc = desc;
		this.level = level;
		}
	protected int intVal() { 
		return level; 
		}
	
	//remember, the static enums are instnaces too, so this method is possible.
	public int compareTo(Object obj) {
		if ( (obj == null) || (!(obj instanceof Logger.Level)) ) { 
			return -1; 
			}
		else return intVal() - ((Level)obj).intVal();
		}
	
	public String toString() 
		{
		return "level=" + intVal();
		}
		
	/**
	Returns a correponding Logger.Level object for the specified integer, if
	the integer is within the Level range (otherwise returns null).
	**/	
	public static Logger.Level forInt(int level) {
		switch (level) {
			 case 1 : return Logger.Level.one;
			 case 2 : return Logger.Level.two;
			 case 3 : return Logger.Level.three;
			 }
		return null;
		}
		
	} //~Level
	
}   //~class Logger


