// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import fc.app.*;
import java.util.*;
import java.util.logging.*;

/** 
Misc. logging related functions for java.util.logging. 
Debugging information is always sent to <tt>System.out</tt> 
because the status of the various loggers is unknown/and or 
affected by methods of this class.
<p>
**/
public final class FCLogUtil 
{
private static boolean dbg = false;

/** 
Use this method to enable/disable any debugging messages from this 
class. By default, debugging is enabled.
@param val	boolean variable, set to true/false 
**/
public static void setDebug(boolean val)
	{
	dbg = val;
	}

/** 
Removes all handlers for all default loggers set up by the JVM
(via the logging.properties file). This is <b>essential</b> and
must be called in the beginning of the application. Otherwise
any logging sent to the application's loggers will also be sent
to handler's for the root ("") logger and hence will result in
logging everything twice (once for the application's logger
and once for the root logger (since all child loggers send their
output to the parent logger and the JDK1.4 configures the root 
logger to have a ConsoleHandler). We don't use the root ("")
loggers at all and this method can be called more than once without
any ill affect.
<p>
Of course, we can specify a different logging property file
to the JVM or modify the default logging file supplied by 
the jvm but using this method can be easier than doing so.

See {@link java.util.logging.LogManager}
*/
public static void clearRoot()
	{
	Logger root = Logger.getLogger("");
	Handler[] handlers = root.getHandlers();
	for (int n = 0; n < handlers.length; n++) {
		root.removeHandler(handlers[n]);
		}
	}
	
/**
Sets all specified loggers in java.util.logging.LogManager to the 
specified level. All handlers for each logger is also set to the same 
specified level.

@param	name		only loggers starting with this string will be
					affected. Specify an empty string to change the
					level for all loggers
@param	level		the new level
**/
public static void setLevelForAll(String name, Level level)
	{
	Argcheck.notnull(level);
	Argcheck.notnull(name);
	LogManager logmgr = LogManager.getLogManager();
	List loggers = getLoggersStartingWith(name);
	Iterator it = loggers.iterator();
	while (it.hasNext()) {
		Logger logger = (Logger) it.next();
		if (dbg) System.out.println("FCLogUtil.setLevelForAll(Level." + level + "): setting new level for Logger = [" + logger.getName() + "] ; Handlers = " + Arrays.asList(logger.getHandlers()) );
		logger.setLevel(level);
		Handler[] handlers = logger.getHandlers();
		for (int n = 0; n < handlers.length; n++) { 
			handlers[n].setLevel(level);	
			}
		}
	}
				
/** 
Returns information for all loggers (starting with the specified name)
installed in the log manager. If the name is an empty string,
returns information for all loggers in the log manager.
**/
public static String getLoggerInfo(String name) 
	{
	if (name == null) name = "";
	String result = "";
	LogManager logmgr = LogManager.getLogManager();
	List loggernames = Collections.list(logmgr.getLoggerNames());
	Iterator it = loggernames.iterator();
	while (it.hasNext()) {
		String itemname = (String)it.next();
		if (! itemname.startsWith(name) )
			continue;
		Logger logger = (Logger) logmgr.getLogger(itemname);
		result += "Logger= [\"" + logger.getName() +
				  "\"] ; Handlers = ";
		Handler[] handlers = logger.getHandlers();
		for (int n = 0; n < handlers.length; n++) {
			Handler h = handlers[n];
			result += "[" + h + "," +
						"level=" + h.getLevel() + "," + 
						"filter=" +	h.getFilter() + "," + 
						"formatter=" + h.getFormatter() + "]";
			}
		result += IOUtil.LINE_SEP;
		}
	return result;
	}

private static List getLoggersStartingWith(String name) {
	List list = new ArrayList();
	LogManager logmgr = LogManager.getLogManager();
	List loggernames = Collections.list(logmgr.getLoggerNames() );
	Iterator it = loggernames.iterator();
	while (it.hasNext()) {
		String itemname = (String)it.next();
		if (! itemname.startsWith(name) )
			continue;	
		Logger logger = (Logger) logmgr.getLogger(itemname);
		list.add(logger);
		}
	return list;
	}

}           //~class FCLogUtil