// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io.fileselectors;

import java.io.*;
import java.util.*;

/** 
Allows for chaining FileFilters, used in java.io.File directory
listing methods. Chained filters allow all filters in the chain
to take turns accepting/rejecting a particular file name. The file
name is only accepted if all filters in the chain accept that
file. Example usage:
<blockquote>
<pre>
java.io.File startdir = new java.io.File("\some\path\here");
String filepattern = "gif,jpg";
ChainedFileFilter filter = new SuffixFilter(filepattern, new NormalFileFilter());	
File[] files = startdir.listFiles(filter);
</pre>
</blockquote>
The above will only list files (not directories) ending with 
<tt>gif</tt> or <tt>jpg</tt>. 
**/
abstract public class ChainedFileSelector implements FileFilter
{
ChainedFileSelector next;

/** 
@param	cf	the next selector in the chain. Pass in <tt>null</tt> if there 
			is no following selector in the chain.
*/
public ChainedFileSelector(ChainedFileSelector cf) { 
	next = cf; 
	}
	
public boolean accept(File name) {
	boolean val = thisfilter(name);
	if (val) 
		val = nextfilter(name);
	return val;
	}

protected boolean nextfilter(File name) {
	if (next != null) 
		return next.accept(name);
	return true;
	}

/** 
Should be implemented by subclasses to accept or reject a filename
based on criteria appropriate for that subclass. 
**/
protected abstract boolean thisfilter(File name);
}
