// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import java.io.*;

import fc.util.*;

/** 
Writes supplied bytes in hex/ascii form. Useful for hex dumps and
debugging. Each <tt>write(..)</tt> method call is independent of previous
invokations and prints data seperately from previous lines. This stream
can also optionally print in other bases instead of hex.
<p>
In addition to it's streamy goodness, this class also provided some misc.
static utility functions such as {@link #toHex}

@author hursh jain
**/
public final class HexOutputStream extends FilterOutputStream
{
static final boolean dbg = false;

final PrintStream 	out;
final String		sp1	 = " ";
final String		sp2  = "  ";
final String		sp3  = "   ";
final String 		nl	 = IOUtil.LINE_SEP;

boolean			showHex	= true;
int				baseValue;

//max digits needed to print 0..255 for any given base
int				baseMaxWidth = 2;	//default for hex  

boolean 		autoflush = true;
boolean 		linenumbers = true;
int 			width;
byte[] 			tempBuf;	//to collect write(int) calls
int				tempBufPtr;
int				tempBufLimit;

/** 
Constructs a new HexOutputStream with a default width of
16 hex numbers (and corresponding ascii values) per line.

@param	 out the underlying outputstream to send the data to
**/
public HexOutputStream(OutputStream out) 
	{
	this (out, 16);
	}

/** 
Constructs a new HexOutputStream with the specified column width.

@param	 out 	the underlying outputstream to send the data to
@param	 width	the number of hex numbers to print per line
**/	
public HexOutputStream(OutputStream out, int width) 
	{
	super(null);
	
	Argcheck.notnull(out);
	Argcheck.istrue(width > 0, "the specified width must be greater than 0");

	if (out instanceof PrintStream) 
		this.out = (PrintStream) out;
	else
		this.out = new PrintStream(out);

	this.width = width;	
	tempBuf = new byte[width];
	tempBufLimit = width - 1;
	if (dbg) System.out.println("Constructed HexOutputStream, out="+out);
	}	
	
/** 
Be careful with this call, if the underlying stream
is say, <tt>System.out</tt>, then calling this method will 
close <tt>System.out</tt>
**/
public void close() {
	out.close();
	}
	
public void flush() {
	out.flush();
	}

public void write(int b) 
	{
	tempBuf[tempBufPtr++] = (byte) b;

	if (tempBufPtr == tempBufLimit) {
		write(tempBuf, 0, tempBufLimit);
		tempBufPtr = 0;
		}
	}

public void write(byte[] buf) 
	{
	if (buf == null) {
		out.println("null");
		return;
		}
	write(buf, 0, buf.length);
	}

public void write(byte[] buf, int off, int len) 
	{
	if (buf == null) {
		out.println("null");
		return;
		}
	
	//faster if we use a buffer as opposed to multiple println()
	//calls	
	StringBuffer strbuf = new StringBuffer(buf.length * 3);
		
	int totalLen = len - off ;

	//the absolute index into the buffer of the last character
	int end = off + len; 

	int linesToPrint = totalLen / width;
	int lineNumPrintWidth = Integer.toString(totalLen).length(); 
	int byteCount = 0;
	
	if (dbg) System.out.println("width="+width+"; totallen="+totalLen+"; end="+end+"; linesToPrint="+linesToPrint+"; lineNumPrintWidth="+lineNumPrintWidth);

	for (int n = off; n < totalLen; n += width)
		{
		if (linenumbers && linesToPrint > 0) {
			strbuf.append(StringUtil.fixedWidth(
							String.valueOf(byteCount), 
							lineNumPrintWidth, HAlign.RIGHT, '0'));
			byteCount += width;
			}
		
		strbuf.append("| ");
		// k cannot exceed lesser of [width (like 16) or the
		// number of actual characters left in this buffer]
		int k, c = 0;	
			
		int kmax  = n + width;	
		//k is an absolute index into buf
		for (k = n; k < kmax && k < end; k++) 
			{
			c = buf[k] & 0xFF;	//byte value is now unsigned
			
			if (showHex) 
				{
				/**
				//This works but let's see if we can use a 
				//faster custom impl.
				if (c < 16) {
					strbuf.append('0');  //we want 0A not A
					}
				strbuf.append(Integer.toHexString(c));
				**/
			    //->the custom impl.
				strbuf.append(toHex(c));
				}
			else { 
				strbuf.append(StringUtil.fixedWidth(
								Integer.toString(c, baseValue),			
								baseMaxWidth, HAlign.RIGHT, '0'));
				}	
				
			strbuf.append(sp1);						
			}
								
		int pad = 0;
		if (k == end) { //loop finished before width, pad to align
			pad = n + width - k;
			
			// add 1 to baseMaxWidth to account for space between
			// each digit group
	
			strbuf.append(StringUtil.repeat(' ', 
									(baseMaxWidth+1) * pad));  
			}
		
		strbuf.append("| ");

		//we can repeat the loop, but timewise it seems to be 
		//about the same as creating 2 string buf's and 
		//populating them within 1 loop. 2 loop is slower
		//but 2 bufs is also slow, when appending buf2 to buf1
		//an arraycopy is done internally inside of stringbuffer
	
		for (k = n; k < kmax && k < end; k++) 
			{
			c = buf[k] & 0xFF;	//byte value is now unsigned
			if (c < 32 || c >= 127)
				strbuf.append('.');			
			else
				strbuf.append((char)c);
			}
	
		if (k == end) { //loop finished before width, pad to align
			strbuf.append(StringUtil.repeat(" ", pad));  
			}

		strbuf.append(" |");
		strbuf.append(nl);
		}
		
	out.print(strbuf.toString());	
	if (autoflush) {
		out.flush();	
		}
	}


/** 
Sets this stream to flush it's contents after every
write statement. By default, this is <tt>true</tt>.
**/
public void setAutoFlush(boolean val) {
	autoflush = val;
	}

/** 
<tt>true/false</tt> enables or disables showing line numbers at
the beginning of each line, <i>if</i> there is more than 1 line in
the output. The line number prefix is a running counter of the
number of bytes displayed so far. This counter is reset after at
the beginning of each write method call.
**/
public void showLineNumbers(boolean val) {
	linenumbers = val;
	}

/** 
Shows output in the specified base (instead of the default hex).
To redundantly set to hex, specify <tt>16</tt>. To show decimal,
specify <tt>10</tt>.

@param	base	the base to show each byte in. Must be between
				2 and 36 (inclusive)
@throws IllegalArgumentException 
				if the specified base is not between [2, 36]
**/
public void setBase(int base) 
	{
	if (base < 2 || base > 36)
		throw new IllegalArgumentException("The base must be between 2 and 36 (both inclusive)");
	baseValue = base;		
	baseMaxWidth = calcBaseMaxWidth(base); 
	if (baseValue != 16) {
		showHex = false;
		}
	}


// essentially we need to ask the question:
// is the number greater than range [0 - base] ? if yes:
// then is the number greater than range [0 - base * base] ? 
// and so on
int calcBaseMaxWidth(int base) {
	int n = 0;
	int start = 255;
	do {
		start = start / base;
		n++;
		}
	while ( start > 0);
	return n;
	}


//Fast hex conversion

static final char[] hex = "0123456789abcdef".toCharArray();

static final char[] toHex(final int c) //the lower 8 bits of c are hexed
	{
	final char[] buf = new char[2];
	if (c < 16) {
		buf[0] = '0';  //we want 0A not A
		}
	else {
		buf[0] = hex[ (c >> 4) & 0x0F ];
		}
	buf[1] = hex[ c & 0x0f ];
	return buf;
	}

/**
Utility method to convert a byte buffer to a hexadecimal string.
useful for md5 and other hashes.
*/
public static final String toHex(final byte[] buf)
	{
	Argcheck.notnull(buf, "buf was null");
	final StringBuilder tmp = new StringBuilder();
	for (int n = 0; n < buf.length; n++)
		{
		int c = buf[n] & 0xFF;	//byte value is now unsigned
		tmp.append(toHex(c));		
		}
	return tmp.toString();
	}

/** 
Unit Test: 
<pre>
Usage: java HexOutputStream <options>
options:
	-file filename 
	or
	-data string
</pre>
**/
public static void main(String[] args) throws Exception
	{
	int cols = -1;
	HexOutputStream hexout = null;
	Args myargs = new Args(args);
	myargs.setUsage("Usage: java HexOutputStream <options>\noptions:\n\t-file filename or -data string\r\n\t-cols num (optional number of display columns)\n\t-base num (optional base 2..36 to show bytes in)");

	if (myargs.getFlagCount() == 0)
		myargs.showError();
		
	byte[] buf = null;

	if (myargs.flagExists("file")) {
		String filename = myargs.get("file");
		buf = IOUtil.fileToByteArray(filename);
		}
	else {	
		String temp = myargs.getRequired("data");
		if (temp != null)
			buf = temp.getBytes();
		}

	if (myargs.flagExists("cols"))
		cols = Integer.parseInt(myargs.get("cols"));

	if (cols >= 0)
		hexout= new HexOutputStream(System.out, cols);
	else
		hexout = new HexOutputStream(System.out);
	
	if (myargs.flagExists("base"))
		hexout.setBase(Byte.parseByte(myargs.get("base")));

	long start = -1;
	if (dbg) start = System.currentTimeMillis();
	hexout.write(buf);
	if (dbg) System.out.println("Time: " + (System.currentTimeMillis() - start) + " ms");
	hexout.close();
	}

}          //~class HexOutputStream
